"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualServiceProvider = exports.VirtualService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const utils_1 = require("./private/utils");
/**
 * VirtualService represents a service inside an AppMesh.
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 * @stability stable
 */
class VirtualService extends cdk.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        const providerConfig = props.virtualServiceProvider.bind(this);
        this.mesh = providerConfig.mesh;
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            meshOwner: utils_1.renderMeshOwner(this.env.account, this.mesh.env.account),
            virtualServiceName: this.physicalName,
            spec: {
                provider: providerConfig.virtualNodeProvider || providerConfig.virtualRouterProvider
                    ? {
                        virtualNode: providerConfig.virtualNodeProvider,
                        virtualRouter: providerConfig.virtualRouterProvider,
                    }
                    : undefined,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${this.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN.
     *
     * @stability stable
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.virtualServiceArn = virtualServiceArn;
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(virtualServiceArn).resourceName);
                this.virtualServiceName = cdk.Fn.select(2, this.parsedArn);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
            }
        }(scope, id);
    }
    /**
     * Import an existing VirtualService given its attributes.
     *
     * @stability stable
     */
    static fromVirtualServiceAttributes(scope, id, attrs) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.virtualServiceName = attrs.virtualServiceName;
                this.mesh = attrs.mesh;
                this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualService`,
                    resourceName: this.virtualServiceName,
                });
            }
        }(scope, id);
    }
}
exports.VirtualService = VirtualService;
_a = JSII_RTTI_SYMBOL_1;
VirtualService[_a] = { fqn: "@aws-cdk/aws-appmesh.VirtualService", version: "1.131.0" };
/**
 * Represents the properties needed to define the provider for a VirtualService.
 *
 * @stability stable
 */
class VirtualServiceProvider {
    /**
     * Returns a VirtualNode based Provider for a VirtualService.
     *
     * @stability stable
     */
    static virtualNode(virtualNode) {
        return new VirtualServiceProviderImpl(virtualNode, undefined);
    }
    /**
     * Returns a VirtualRouter based Provider for a VirtualService.
     *
     * @stability stable
     */
    static virtualRouter(virtualRouter) {
        return new VirtualServiceProviderImpl(undefined, virtualRouter);
    }
    /**
     * Returns an Empty Provider for a VirtualService.
     *
     * This provides no routing capabilities
     * and should only be used as a placeholder
     *
     * @stability stable
     */
    static none(mesh) {
        return new VirtualServiceProviderImpl(undefined, undefined, mesh);
    }
}
exports.VirtualServiceProvider = VirtualServiceProvider;
_b = JSII_RTTI_SYMBOL_1;
VirtualServiceProvider[_b] = { fqn: "@aws-cdk/aws-appmesh.VirtualServiceProvider", version: "1.131.0" };
class VirtualServiceProviderImpl extends VirtualServiceProvider {
    constructor(virtualNode, virtualRouter, mesh) {
        var _c, _d, _e, _f;
        super();
        this.virtualNode = virtualNode;
        this.virtualRouter = virtualRouter;
        const providedMesh = (_f = (_d = (_c = this.virtualNode) === null || _c === void 0 ? void 0 : _c.mesh) !== null && _d !== void 0 ? _d : (_e = this.virtualRouter) === null || _e === void 0 ? void 0 : _e.mesh) !== null && _f !== void 0 ? _f : mesh;
        this.mesh = providedMesh;
    }
    bind(_construct) {
        return {
            mesh: this.mesh,
            virtualNodeProvider: this.virtualNode
                ? {
                    virtualNodeName: this.virtualNode.virtualNodeName,
                }
                : undefined,
            virtualRouterProvider: this.virtualRouter
                ? {
                    virtualRouterName: this.virtualRouter.virtualRouterName,
                }
                : undefined,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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