import * as acmpca from '@aws-cdk/aws-acmpca';
import { CfnVirtualNode } from './appmesh.generated';
import { Construct } from '@aws-cdk/core';
/**
 * Represents the properties needed to define TLS Validation context
 */
interface TlsValidationCommon {
    /**
     * Represents the subject alternative names (SANs) secured by the certificate.
     *
     * SANs must be in the FQDN or URI format.
     *
     * @default - If you don't specify SANs on the terminating mesh endpoint,
     * the Envoy proxy for that node doesn't verify the SAN on a peer client certificate.
     * If you don't specify SANs on the originating mesh endpoint,
     * the SAN on the certificate provided by the terminating endpoint must match the mesh endpoint service discovery configuration.
     * @stability stable
     */
    readonly subjectAlternativeNames?: SubjectAlternativeNames;
}
/**
 * Represents the properties needed to define TLS Validation context.
 *
 * @stability stable
 */
export interface TlsValidation extends TlsValidationCommon {
    /**
     * Reference to where to retrieve the trust chain.
     *
     * @stability stable
     */
    readonly trust: TlsValidationTrust;
}
/**
 * Represents the properties needed to define TLS Validation context that is supported for mutual TLS authentication.
 *
 * @stability stable
 */
export interface MutualTlsValidation extends TlsValidationCommon {
    /**
     * Reference to where to retrieve the trust chain.
     *
     * @stability stable
     */
    readonly trust: MutualTlsValidationTrust;
}
/**
 * All Properties for TLS Validation Trusts for both Client Policy and Listener.
 *
 * @stability stable
 */
export interface TlsValidationTrustConfig {
    /**
     * VirtualNode CFN configuration for client policy's TLS Validation Trust.
     *
     * @stability stable
     */
    readonly tlsValidationTrust: CfnVirtualNode.TlsValidationContextTrustProperty;
}
/**
 * Defines the TLS Validation Context Trust.
 *
 * @stability stable
 */
export declare abstract class TlsValidationTrust {
    /**
     * Tells envoy where to fetch the validation context from.
     *
     * @stability stable
     */
    static file(certificateChain: string): MutualTlsValidationTrust;
    /**
     * TLS Validation Context Trust for ACM Private Certificate Authority (CA).
     *
     * @stability stable
     */
    static acm(certificateAuthorities: acmpca.ICertificateAuthority[]): TlsValidationTrust;
    /**
     * TLS Validation Context Trust for Envoy' service discovery service.
     *
     * @stability stable
     */
    static sds(secretName: string): MutualTlsValidationTrust;
    /**
     * Returns Trust context based on trust type.
     *
     * @stability stable
     */
    abstract bind(scope: Construct): TlsValidationTrustConfig;
}
/**
 * Represents a TLS Validation Context Trust that is supported for mutual TLS authentication.
 *
 * @stability stable
 */
export declare abstract class MutualTlsValidationTrust extends TlsValidationTrust {
    /**
     * @stability stable
     */
    protected readonly differentiator = false;
}
/**
 * All Properties for Subject Alternative Names Matcher for both Client Policy and Listener.
 *
 * @stability stable
 */
export interface SubjectAlternativeNamesMatcherConfig {
    /**
     * VirtualNode CFN configuration for subject alternative names secured by the certificate.
     *
     * @stability stable
     */
    readonly subjectAlternativeNamesMatch: CfnVirtualNode.SubjectAlternativeNameMatchersProperty;
}
/**
 * Used to generate Subject Alternative Names Matchers.
 *
 * @stability stable
 */
export declare abstract class SubjectAlternativeNames {
    /**
     * The values of the SAN must match the specified values exactly.
     *
     * @param names The exact values to test against.
     * @stability stable
     */
    static matchingExactly(...names: string[]): SubjectAlternativeNames;
    /**
     * Returns Subject Alternative Names Matcher based on method type.
     *
     * @stability stable
     */
    abstract bind(scope: Construct): SubjectAlternativeNamesMatcherConfig;
}
export {};
