# JWST General Target Visibility Tool (jwst_gtvt)

JWST requires shielding from the Sun for operation, which limits the available position angles observable at a given time. This script calculates the allowed position angle for a given Right Ascension and Declination for each instrument on the telescope.To report any issues, please use the JWST help portal https://jwsthelp.stsci.edu or feel free to open a github issue https://github.com/spacetelescope/jwst_gtvt/issues.

# Dependencies

This tool requires a few packages all of which are included in the Anaconda Python distribution.

* numpy

* maplotlib

* astropy

* astroquery (For moving target support)

* pysiaf

# Installation

You can install the tool using `pip` with 
`pip install jwst-gtvt`

Alternatively, you can clone the respository from GitHub and install the tool from inside the resulting directory with
`python setup.py install`

# Usage

There are two scripts available.  `jwst_gtvt` for fixed targets, and `jwst_mtvt` for moving targets.  To see the help info use

    $ jwst_gtvt -h
        usage: jwst_gtvt [-h] [--v3pa V3PA] [--save_plot SAVE_PLOT]
                 [--save_table SAVE_TABLE] [--instrument INSTRUMENT]
                 [--name NAME] [--start_date START_DATE] [--end_date END_DATE]
                 [--no_verbose]
                 ra dec

        positional arguments:
        ra                    Right Ascension of target in either sexagesimal
                                (hh:mm:ss.s) or degrees.
        dec                   Declination of target in either sexagesimal
                                (dd:mm:ss.s) or degrees.

        optional arguments:
        -h, --help            show this help message and exit
        --v3pa V3PA           Specify a desired V3 (telescope frame) Position Angle.
        --save_plot SAVE_PLOT
                                Path of file to save plot output.
        --save_table SAVE_TABLE
                                Path of file to save table output.
        --instrument INSTRUMENT
                                If specified plot shows only windows for this
                                instrument. Options: nircam, nirspec, niriss, miri,
                                fgs, v3 (case insensitive).
        --name NAME           Target Name to appear on plots. Names with space
                                should use double quotes e.g. "NGC 6240".
        --start_date START_DATE
                                Start date for visibility search in yyyy-mm-dd format.
                                Earliest available is 2021-12-26.
        --end_date END_DATE   End date for visibility search in yyyy-mm-dd format.
                                Latest available is 2024-10-02.
        --no_verbose          Suppress table output to screen

# Example

By default you need only specify R.A. and Dec. in either sexigesimal or degrees.
The observability windows will be printed to the terminal and a plot showing the windows for each instrument will pop up.

`$ jwst_gtvt 16:52:58.9 02:24:03`

`$ jwst_gtvt 253.2458 2.4008`

![Example Plot](docs/jwst_target_visibility.png "Example default plot output.")

For moving targets, use `jwst_mtvt`

`$ jwst_mtvt Ceres`

Periodic comets and most asteroids benefit from using the `--smallbody` flag

`$ jwst_mtvt 2`  # Venus (no windows)

`$ jwst_mtvt 2 --smallbody`  # 2 Pallas

`$ jwst_mtvt C/2016 M1`  # works with or without --smallbody

![Example Plot](docs/jwst_moving_target_visibility.png "A moving target example.")

Setting the `--name` flag will add a target name to the plot title

`$ jwst_gtvt 16:52:58.9 02:24:03 --name "NGC 6240"`

You can specify the instrument via the `--instrument` flag.

`$ jwst_gtvt 16:52:58.9 02:24:03 --name "NGC 6240" --instrument nircam`

and the resulting plot will only contain the windows for the specified instrument.
The allowed values for `--instrument` are 'nircam', 'nirspec', 'niriss', 'miri', 'fgs', and 'v3' (case insensitive).

![Example Plot](docs/nircam_target_visibility.png "Example plot output when specifying instrument.")

You can save the text ouput to a file instead of having it output to terminal with `--save_table`.  
Likewise, you can save the plot with `--save_plot`.
The plot can be saved in any format supported by matplotlib (.png, .jpeg, .pdf, .eps) by specifying the desired extension in the filename.

`$ jwst_gtvt 16:52:58.9 02:24:03 --save_table visibility.txt --save_plot visibility.png`

If you only want to plot a specific range of dates, rather than the entire available ephemeris you specify a `--start_date` or `--end_date` in ISO format (yyyy-mm-dd).
For example

`$ jwst_gtvt 16:52:58.9 02:24:03 --name "NGC 6240" --start_date 2022-01-01 --end_date 2023-01-01`

![Example Plot](docs/jwst_target_visibility_2022.png "Example plot output when specifying start and end dates.")

Specifying the `--v3pa` will display the observing windows which contain the desired V3 position angle in the text output.

Below is an example of the full text output

    $ jwst_gtvt 16:52:58.9 02:24:03


    Using Equatorial Coordinates

           Target
                    ecliptic
    RA      Dec     latitude
    253.245   2.401  24.771

    Checked interval [2021-12-26, 2024-10-02]
    |           Window [days]                 |    Normal V3 PA [deg]    |
    Start           End         Duration         Start         End         RA            Dec     
    2022-02-25      2022-04-22        56.00     279.26298     249.96549     253.24542       2.40083 
    2022-07-12      2022-09-10        59.92     125.04211      94.63733     253.24542       2.40083 
    2023-02-25      2023-04-23        57.00     279.39286     249.27157     253.24542       2.40083 
    2023-07-13      2023-09-10        59.00     124.55368      94.71103     253.24542       2.40083 
    2024-02-25      2024-04-22        57.00     279.51367     249.42124     253.24542       2.40083 
    2024-07-12      2024-09-09        59.10     124.77976      94.75328     253.24542       2.40083 


                    V3PA          NIRCam           NIRSpec         NIRISS           MIRI          FGS
    Date      min    max      min    max       min    max     min    max      min    max      min    max

    2022-02-26   275.26 283.18   275.15 283.07    53.76  61.67   275.83 283.75   280.10 288.01   274.01 281.93
    2022-02-27   274.27 283.33   274.16 283.22    52.76  61.83   274.84 283.90   279.11 288.17   273.02 282.08
    2022-02-28   273.28 283.49   273.17 283.38    51.77  61.98   273.85 284.06   278.11 288.32   272.03 282.24
    2022-03-01   272.87 283.07   272.75 282.96    51.36  61.56   273.44 283.64   277.70 287.90   271.62 281.82
    2022-03-02   272.45 282.65   272.34 282.54    50.94  61.14   273.02 283.22   277.29 287.49   271.20 281.40
    2022-03-03   272.04 282.24   271.92 282.12    50.53  60.73   272.61 282.81   276.87 287.07   270.79 280.99
    2022-03-04   271.62 281.82   271.51 281.71    50.11  60.31   272.19 282.39   276.45 286.66   270.37 280.57
    2022-03-05   271.20 281.41   271.09 281.29    49.69  59.90   271.77 281.98   276.03 286.24   269.95 280.16
    2022-03-06   270.78 280.99   270.67 280.88    49.27  59.49   271.35 281.56   275.61 285.83   269.53 279.74
    2022-03-07   270.36 280.58   270.24 280.47    48.85  59.07   270.93 281.15   275.19 285.42   269.11 279.33
    2022-03-08   269.93 280.17   269.82 280.06    48.42  58.66   270.50 280.74   274.77 285.00   268.68 278.92
    2022-03-09   269.51 279.76   269.39 279.64    48.00  58.25   270.08 280.33   274.34 284.59   268.25 278.51
    2022-03-10   269.08 279.35   268.96 279.23    47.57  57.84   269.65 279.92   273.91 284.18   267.83 278.09
    2022-03-11   268.64 278.93   268.53 278.82    47.14  57.43   269.21 279.50   273.48 283.77   267.39 277.68
    2022-03-12   268.21 278.52   268.10 278.41    46.70  57.01   268.78 279.09   273.04 283.36   266.96 277.27
    2022-03-13   267.77 278.11   267.66 278.00    46.26  56.60   268.34 278.68   272.60 282.94   266.52 276.86
    2022-03-14   267.33 277.69   267.21 277.58    45.82  56.19   267.90 278.26   272.16 282.53   266.08 276.44
    2022-03-15   266.88 277.28   266.77 277.17    45.37  55.77   267.45 277.85   271.71 282.11   265.63 276.03
    2022-03-16   266.43 276.86   266.32 276.75    44.92  55.36   267.00 277.43   271.26 281.70   265.18 275.61
    2022-03-17   265.97 276.45   265.86 276.33    44.47  54.94   266.54 277.01   270.81 281.28   264.72 275.19
    2022-03-18   265.51 276.03   265.40 275.91    44.01  54.52   266.08 276.60   270.35 280.86   264.26 274.77
    2022-03-19   265.05 275.60   264.94 275.49    43.54  54.10   265.62 276.17   269.88 280.44   263.80 274.35
    2022-03-20   264.58 275.18   264.47 275.07    43.07  53.67   265.15 275.75   269.41 280.01   263.33 273.93
    2022-03-21   264.11 274.75   263.99 274.64    42.60  53.25   264.68 275.32   268.94 279.59   262.85 273.50
    2022-03-22   263.62 274.32   263.51 274.21    42.12  52.82   264.19 274.89   268.46 279.16   262.37 273.07
    2022-03-23   263.14 273.89   263.02 273.78    41.63  52.39   263.71 274.46   267.97 278.73   261.89 272.64
    2022-03-24   262.64 273.46   262.53 273.34    41.14  51.95   263.21 274.03   267.48 278.29   261.39 272.21
    2022-03-25   262.14 273.02   262.03 272.91    40.64  51.51   262.71 273.59   266.98 277.85   260.89 271.77
    2022-03-26   261.64 272.58   261.52 272.46    40.13  51.07   262.21 273.15   266.47 277.41   260.39 271.33
    2022-03-27   261.12 272.13   261.01 272.02    39.61  50.62   261.69 272.70   265.96 276.96   259.87 270.88
    2022-03-28   260.60 271.68   260.49 271.57    39.09  50.17   261.17 272.25   265.43 276.51   259.35 270.43
    2022-03-29   260.07 271.22   259.96 271.11    38.56  49.71   260.64 271.79   264.90 276.06   258.82 269.97
    2022-03-30   259.53 270.76   259.42 270.65    38.02  49.25   260.10 271.33   264.36 275.60   258.28 269.51
    2022-03-31   258.98 270.30   258.87 270.18    37.47  48.79   259.55 270.86   263.81 275.13   257.73 269.04
    2022-04-01   258.42 269.82   258.31 269.71    36.91  48.32   258.99 270.39   263.26 274.66   257.17 268.57
    2022-04-02   257.85 269.35   257.74 269.23    36.35  47.84   258.42 269.92   262.69 274.18   256.60 268.09
    2022-04-03   257.28 268.86   257.16 268.75    35.77  47.35   257.85 269.43   262.11 273.70   256.03 267.61
    2022-04-04   256.69 268.37   256.57 268.26    35.18  46.86   257.26 268.94   261.52 273.20   255.44 267.12
    2022-04-05   256.09 267.87   255.97 267.76    34.58  46.36   256.66 268.44   260.92 272.71   254.84 266.62
    2022-04-06   255.48 267.37   255.36 267.25    33.97  45.86   256.05 267.94   260.31 272.20   254.22 266.12
    2022-04-07   254.85 266.85   254.74 266.74    33.34  45.34   255.42 267.42   259.69 271.69   253.60 265.60
    2022-04-08   254.21 266.33   254.10 266.22    32.71  44.82   254.78 266.90   259.05 271.16   252.96 265.08
    2022-04-09   253.56 265.80   253.45 265.69    32.06  44.29   254.13 266.37   258.40 270.63   252.31 264.55
    2022-04-10   252.90 265.26   252.79 265.15    31.39  43.75   253.47 265.83   257.73 270.09   251.65 264.01
    2022-04-11   252.22 264.71   252.11 264.60    30.71  43.20   252.79 265.28   257.05 269.54   250.97 263.46
    2022-04-12   251.52 264.15   251.41 264.04    30.02  42.64   252.09 264.72   256.36 268.98   250.27 262.90
    2022-04-13   250.81 263.58   250.70 263.46    29.31  42.07   251.38 264.15   255.65 268.41   249.56 262.33
    2022-04-14   250.09 262.99   249.97 262.88    28.58  41.49   250.66 263.56   254.92 267.83   248.83 261.74
    2022-04-15   249.34 262.40   249.23 262.29    27.83  40.89   249.91 262.97   254.17 267.23   248.09 261.15
    2022-04-16   248.58 261.79   248.46 261.68    27.07  40.28   249.15 262.36   253.41 266.62   247.33 260.54
    2022-04-17   247.79 261.17   247.68 261.06    26.29  39.66   248.36 261.74   252.63 266.00   246.54 259.92
    2022-04-18   246.99 260.53   246.88 260.42    25.48  39.03   247.56 261.10   251.83 265.37   245.74 259.28
    2022-04-19   246.17 259.88   246.06 259.77    24.66  38.37   246.74 260.45   251.00 264.72   244.92 258.63
    2022-04-20   245.32 259.22   245.21 259.10    23.82  37.71   245.89 259.79   250.16 264.05   244.07 257.97
    2022-04-21   244.46 258.53   244.34 258.42    22.95  37.03   245.03 259.10   249.29 263.37   243.21 257.28
    2022-04-22   243.57 257.83   243.45 257.72    22.06  36.33   244.14 258.40   248.40 262.67   242.31 256.58

    2022-07-13   117.81 132.25   117.69 132.13   256.30 270.74   118.38 132.82   122.64 137.08   116.56 131.00
    2022-07-14   117.11 131.37   117.00 131.25   255.61 269.86   117.68 131.94   121.95 136.20   115.86 130.12
    2022-07-15   116.44 130.51   116.33 130.40   254.93 269.00   117.01 131.08   121.27 135.34   115.19 129.26
    2022-07-16   115.78 129.67   115.67 129.56   254.27 268.17   116.35 130.24   120.62 134.51   114.53 128.42
    2022-07-17   115.14 128.86   115.03 128.75   253.63 267.35   115.71 129.43   119.97 133.69   113.89 127.61
    2022-07-18   114.51 128.07   114.40 127.95   253.00 266.56   115.08 128.64   119.34 132.90   113.26 126.81
    2022-07-19   113.90 127.29   113.78 127.18   252.39 265.78   114.47 127.86   118.73 132.13   112.64 126.04
    2022-07-20   113.29 126.53   113.18 126.42   251.79 265.03   113.86 127.10   118.13 131.37   112.04 125.28
    2022-07-21   112.70 125.80   112.59 125.68   251.20 264.29   113.27 126.37   117.54 130.63   111.45 124.55
    2022-07-22   112.13 125.07   112.01 124.96   250.62 263.57   112.70 125.64   116.96 129.91   110.88 123.82
    2022-07-23   111.56 124.37   111.45 124.26   250.05 262.86   112.13 124.94   116.40 129.20   110.31 123.12
    2022-07-24   111.01 123.68   110.89 123.57   249.50 262.17   111.58 124.25   115.84 128.51   109.75 122.43
    2022-07-25   110.46 123.00   110.35 122.89   248.95 261.50   111.03 123.57   115.29 127.84   109.21 121.75
    2022-07-26   109.92 122.34   109.81 122.23   248.42 260.84   110.49 122.91   114.76 127.18   108.67 121.09
    2022-07-27   109.40 121.70   109.28 121.58   247.89 260.19   109.97 122.27   114.23 126.53   108.15 120.45
    2022-07-28   108.88 121.06   108.77 120.95   247.37 259.55   109.45 121.63   113.71 125.90   107.63 119.81
    2022-07-29   108.37 120.44   108.26 120.33   246.86 258.93   108.94 121.01   113.20 125.27   107.12 119.19
    2022-07-30   107.87 119.83   107.75 119.72   246.36 258.32   108.44 120.40   112.70 124.66   106.62 118.58
    2022-07-31   107.37 119.23   107.26 119.12   245.86 257.72   107.94 119.80   112.21 124.07   106.12 117.98
    2022-08-01   106.88 118.64   106.77 118.53   245.38 257.14   107.45 119.21   111.72 123.48   105.63 117.39
    2022-08-02   106.40 118.07   106.29 117.95   244.89 256.56   106.97 118.64   111.24 122.90   105.15 116.82
    2022-08-03   105.93 117.50   105.82 117.39   244.42 255.99   106.50 118.07   110.76 122.33   104.68 116.25
    2022-08-04   105.46 116.94   105.35 116.83   243.95 255.43   106.03 117.51   110.29 121.78   104.21 115.69
    2022-08-05   105.00 116.39   104.88 116.28   243.49 254.89   105.56 116.96   109.83 121.23   103.74 115.14
    2022-08-06   104.54 115.85   104.42 115.74   243.03 254.35   105.11 116.42   109.37 120.69   103.29 114.60
    2022-08-07   104.08 115.32   103.97 115.21   242.58 253.81   104.65 115.89   108.92 120.16   102.83 114.07
    2022-08-08   103.64 114.80   103.52 114.69   242.13 253.29   104.20 115.37   108.47 119.63   102.38 113.55
    2022-08-09   103.19 114.28   103.08 114.17   241.68 252.77   103.76 114.85   108.03 119.12   101.94 113.03
    2022-08-10   102.75 113.77   102.64 113.66   241.24 252.27   103.32 114.34   107.58 118.61   101.50 112.52
    2022-08-11   102.31 113.27   102.20 113.16   240.81 251.76   102.88 113.84   107.15 118.11   101.06 112.02
    2022-08-12   101.88 112.78   101.77 112.66   240.37 251.27   102.45 113.35   106.72 117.61   100.63 111.53
    2022-08-13   101.45 112.29   101.34 112.17   239.94 250.78   102.02 112.86   106.29 117.12   100.20 111.04
    2022-08-14   101.03 111.80   100.91 111.69   239.52 250.30   101.60 112.37   105.86 116.64    99.77 110.55
    2022-08-15   100.60 111.33   100.49 111.21   239.09 249.82   101.17 111.90   105.44 116.16    99.35 110.08
    2022-08-16   100.18 110.86   100.07 110.74   238.67 249.35   100.75 111.43   105.02 115.69    98.93 109.60
    2022-08-17    99.76 110.39    99.65 110.28   238.25 248.88   100.33 110.96   104.60 115.22    98.51 109.14
    2022-08-18    99.35 109.93    99.23 109.81   237.84 248.42    99.92 110.50   104.18 114.76    98.10 108.68
    2022-08-19    98.93 109.47    98.82 109.36   237.42 247.96    99.50 110.04   103.77 114.30    97.68 108.22
    2022-08-20    98.52 109.02    98.41 108.90   237.01 247.51    99.09 109.59   103.35 113.85    97.27 107.77
    2022-08-21    98.11 108.57    97.99 108.46   236.60 247.06    98.68 109.14   102.94 113.40    96.86 107.32
    2022-08-22    97.70 108.12    97.58 108.01   236.19 246.62    98.27 108.69   102.53 112.96    96.45 106.87
    2022-08-23    97.29 107.68    97.18 107.57   235.78 246.17    97.86 108.25   102.12 112.52    96.04 106.43
    2022-08-24    96.88 107.24    96.77 107.13   235.37 245.74    97.45 107.81   101.71 112.08    95.63 105.99
    2022-08-25    96.47 106.81    96.36 106.70   234.97 245.30    97.04 107.38   101.31 111.64    95.22 105.56
    2022-08-26    96.07 106.38    95.95 106.27   234.56 244.87    96.64 106.95   100.90 111.21    94.82 105.13
    2022-08-27    95.66 105.95    95.55 105.84   234.15 244.44    96.23 106.52   100.49 110.78    94.41 104.70
    2022-08-28    95.25 105.52    95.14 105.41   233.75 244.02    95.82 106.09   100.09 110.36    94.00 104.27
    2022-08-29    94.85 105.10    94.73 104.99   233.34 243.59    95.42 105.67    99.68 109.93    93.60 103.85
    2022-08-30    94.44 104.68    94.33 104.57   232.93 243.17    95.01 105.25    99.27 109.51    93.19 103.43
    2022-08-31    94.03 104.26    93.92 104.15   232.53 242.75    94.60 104.83    98.87 109.09    92.78 103.01
    2022-09-01    93.63 103.84    93.51 103.73   232.12 242.33    94.20 104.41    98.46 108.68    92.37 102.59
    2022-09-02    93.22 103.43    93.10 103.31   231.71 241.92    93.79 104.00    98.05 108.26    91.97 102.17
    2022-09-03    92.81 103.01    92.70 102.90   231.30 241.50    93.38 103.58    97.64 107.85    91.56 101.76
    2022-09-04    92.40 102.60    92.28 102.48   230.89 241.09    92.97 103.17    97.23 107.43    91.15 101.35
    2022-09-05    91.98 102.18    91.87 102.07   230.48 240.68    92.55 102.75    96.82 107.02    90.73 100.93
    2022-09-06    91.57 101.77    91.46 101.66   230.06 240.27    92.14 102.34    96.41 106.61    90.32 100.52
    2022-09-07    91.16 101.36    91.04 101.25   229.65 239.85    91.73 101.93    95.99 106.20    89.90 100.11
    2022-09-08    91.03 100.66    90.92 100.54   229.53 239.15    91.60 101.23    95.87 105.49    89.78  99.41
    2022-09-09    91.17  99.69    91.06  99.58   229.66 238.18    91.74 100.26    96.01 104.52    89.92  98.44
    2022-09-10    91.31  98.72    91.20  98.61   229.80 237.21    91.88  99.29    96.14 103.55    90.06  97.47

    2023-02-26   275.52 283.18   275.41 283.07    54.01  61.67   276.09 283.75   280.36 288.01   274.27 281.93
    2023-02-27   274.53 283.33   274.41 283.22    53.02  61.83   275.10 283.90   279.36 288.17   273.28 282.08
    2023-02-28   273.53 283.49   273.42 283.38    52.03  61.98   274.10 284.06   278.37 288.33   272.28 282.24
    2023-03-01   272.99 283.20   272.88 283.08    51.48  61.69   273.56 283.77   277.83 288.03   271.74 281.95
    2023-03-02   272.58 282.78   272.46 282.66    51.07  61.27   273.15 283.35   277.41 287.61   271.32 281.53
    2023-03-03   272.16 282.36   272.05 282.25    50.65  60.85   272.73 282.93   276.99 287.19   270.91 281.11
    2023-03-04   271.74 281.94   271.63 281.83    50.23  60.43   272.31 282.51   276.57 286.78   270.49 280.69
    2023-03-05   271.32 281.53   271.21 281.41    49.81  60.02   271.89 282.10   276.16 286.36   270.07 280.28
    2023-03-06   270.90 281.11   270.79 281.00    49.39  59.60   271.47 281.68   275.73 285.95   269.65 279.86
    2023-03-07   270.48 280.70   270.36 280.59    48.97  59.19   271.05 281.27   275.31 285.53   269.23 279.45
    2023-03-08   270.05 280.28   269.94 280.17    48.54  58.78   270.62 280.85   274.88 285.12   268.80 279.03
    2023-03-09   269.62 279.87   269.51 279.76    48.11  58.36   270.19 280.44   274.46 284.71   268.37 278.62
    2023-03-10   269.19 279.46   269.08 279.35    47.68  57.95   269.76 280.03   274.03 284.29   267.94 278.21
    2023-03-11   268.76 279.04   268.65 278.93    47.25  57.54   269.33 279.61   273.59 283.88   267.51 277.79
    2023-03-12   268.32 278.63   268.21 278.52    46.81  57.12   268.89 279.20   273.16 283.46   267.07 277.38
    2023-03-13   267.88 278.22   267.77 278.10    46.37  56.71   268.45 278.79   272.72 283.05   266.63 276.97
    2023-03-14   267.44 277.80   267.33 277.69    45.93  56.29   268.01 278.37   272.27 282.63   266.19 276.55
    2023-03-15   266.99 277.38   266.88 277.27    45.48  55.88   267.56 277.95   271.83 282.22   265.74 276.13
    2023-03-16   266.54 276.97   266.43 276.85    45.03  55.46   267.11 277.54   271.37 281.80   265.29 275.72
    2023-03-17   266.08 276.55   265.97 276.43    44.58  55.04   266.65 277.12   270.92 281.38   264.83 275.30
    2023-03-18   265.62 276.13   265.51 276.01    44.12  54.62   266.19 276.70   270.46 280.96   264.37 274.88
    2023-03-19   265.16 275.70   265.05 275.59    43.65  54.20   265.73 276.27   269.99 280.54   263.91 274.45
    2023-03-20   264.69 275.28   264.58 275.17    43.18  53.77   265.26 275.85   269.52 280.11   263.44 274.03
    2023-03-21   264.21 274.85   264.10 274.74    42.70  53.34   264.78 275.42   269.05 279.69   262.96 273.60
    2023-03-22   263.73 274.42   263.62 274.31    42.22  52.91   264.30 274.99   268.56 279.26   262.48 273.17
    2023-03-23   263.24 273.99   263.13 273.88    41.74  52.48   263.81 274.56   268.08 278.82   261.99 272.74
    2023-03-24   262.75 273.55   262.64 273.44    41.24  52.04   263.32 274.12   267.58 278.39   261.50 272.30
    2023-03-25   262.25 273.11   262.14 273.00    40.74  51.60   262.82 273.68   267.08 277.95   261.00 271.86
    2023-03-26   261.74 272.67   261.63 272.56    40.23  51.16   262.31 273.24   266.57 277.50   260.49 271.42
    2023-03-27   261.23 272.22   261.11 272.11    39.72  50.71   261.79 272.79   266.06 277.05   259.97 270.97
    2023-03-28   260.70 271.77   260.59 271.66    39.19  50.26   261.27 272.34   265.54 276.60   259.45 270.52
    2023-03-29   260.17 271.31   260.06 271.20    38.66  49.80   260.74 271.88   265.01 276.15   258.92 270.06
    2023-03-30   259.63 270.85   259.52 270.74    38.12  49.34   260.20 271.42   264.47 275.68   258.38 269.60
    2023-03-31   259.08 270.38   258.97 270.27    37.58  48.88   259.65 270.95   263.92 275.22   257.83 269.13
    2023-04-01   258.53 269.91   258.41 269.80    37.02  48.40   259.10 270.48   263.36 274.74   257.27 268.66
    2023-04-02   257.96 269.43   257.85 269.32    36.45  47.92   258.53 270.00   262.79 274.27   256.71 268.18
    2023-04-03   257.38 268.95   257.27 268.83    35.87  47.44   257.95 269.52   262.21 273.78   256.13 267.70
    2023-04-04   256.79 268.46   256.68 268.34    35.28  46.95   257.36 269.03   261.63 273.29   255.54 267.21
    2023-04-05   256.19 267.96   256.08 267.85    34.68  46.45   256.76 268.53   261.03 272.79   254.94 266.71
    2023-04-06   255.58 267.45   255.47 267.34    34.07  45.94   256.15 268.02   260.42 272.29   254.33 266.20
    2023-04-07   254.96 266.94   254.85 266.83    33.45  45.43   255.53 267.51   259.79 271.77   253.71 265.69
    2023-04-08   254.32 266.42   254.21 266.30    32.81  44.91   254.89 266.99   259.16 271.25   253.07 265.17
    2023-04-09   253.67 265.89   253.56 265.77    32.17  44.38   254.24 266.46   258.51 270.72   252.42 264.64
    2023-04-10   253.01 265.35   252.90 265.23    31.50  43.84   253.58 265.92   257.84 270.18   251.76 264.10
    2023-04-11   252.33 264.80   252.22 264.68    30.82  43.29   252.90 265.37   257.17 269.63   251.08 263.55
    2023-04-12   251.64 264.24   251.53 264.12    30.13  42.73   252.21 264.81   256.47 269.07   250.39 262.99
    2023-04-13   250.93 263.67   250.82 263.55    29.42  42.16   251.50 264.24   255.77 268.50   249.68 262.42
    2023-04-14   250.21 263.08   250.09 262.97    28.70  41.58   250.78 263.65   255.04 267.92   248.96 261.83
    2023-04-15   249.46 262.49   249.35 262.38    27.96  40.98   250.03 263.06   254.30 267.33   248.21 261.24
    2023-04-16   248.70 261.89   248.59 261.77    27.20  40.38   249.27 262.46   253.54 266.72   247.45 260.63
    2023-04-17   247.93 261.27   247.81 261.15    26.42  39.76   248.50 261.84   252.76 266.10   246.67 260.02
    2023-04-18   247.13 260.63   247.01 260.52    25.62  39.13   247.70 261.20   251.96 265.47   245.88 259.38
    2023-04-19   246.31 259.99   246.20 259.87    24.80  38.48   246.88 260.56   251.14 264.82   245.06 258.73
    2023-04-20   245.47 259.32   245.36 259.21    23.96  37.81   246.04 259.89   250.30 264.16   244.22 258.07
    2023-04-21   244.61 258.64   244.49 258.53    23.10  37.14   245.18 259.21   249.44 263.48   243.35 257.39
    2023-04-22   243.72 257.95   243.61 257.83    22.21  36.44   244.29 258.52   248.55 262.78   242.47 256.70
    2023-04-23   242.81 257.23   242.70 257.12    21.30  35.73   243.38 257.80   247.64 262.07   241.56 255.98

    2023-07-14   117.32 131.63   117.21 131.52   255.81 270.12   117.89 132.20   122.15 136.47   116.07 130.38
    2023-07-15   116.64 130.77   116.53 130.65   255.13 269.26   117.21 131.34   121.47 135.60   115.39 129.52
    2023-07-16   115.97 129.92   115.86 129.81   254.47 268.42   116.54 130.49   120.81 134.76   114.72 128.67
    2023-07-17   115.32 129.10   115.21 128.99   253.82 267.59   115.89 129.67   120.16 133.94   114.07 127.85
    2023-07-18   114.69 128.30   114.58 128.19   253.18 266.79   115.26 128.87   119.53 133.13   113.44 127.05
    2023-07-19   114.07 127.52   113.96 127.41   252.56 266.01   114.64 128.09   118.90 132.35   112.82 126.27
    2023-07-20   113.46 126.75   113.35 126.64   251.96 265.25   114.03 127.32   118.30 131.59   112.21 125.50
    2023-07-21   112.87 126.01   112.76 125.90   251.36 264.50   113.44 126.58   117.70 130.84   111.62 124.76
    2023-07-22   112.29 125.28   112.17 125.17   250.78 263.77   112.86 125.85   117.12 130.12   111.04 124.03
    2023-07-23   111.72 124.57   111.60 124.46   250.21 263.06   112.29 125.14   116.55 129.40   110.47 123.32
    2023-07-24   111.16 123.87   111.04 123.76   249.65 262.37   111.73 124.44   115.99 128.71   109.91 122.62
    2023-07-25   110.61 123.19   110.49 123.08   249.10 261.69   111.18 123.76   115.44 128.03   109.36 121.94
    2023-07-26   110.07 122.53   109.95 122.41   248.56 261.02   110.64 123.10   114.90 127.36   108.82 121.28
    2023-07-27   109.54 121.87   109.42 121.76   248.03 260.37   110.11 122.44   114.37 126.71   108.29 120.62
    2023-07-28   109.01 121.23   108.90 121.12   247.51 259.73   109.58 121.80   113.85 126.07   107.76 119.98
    2023-07-29   108.50 120.61   108.39 120.50   246.99 259.10   109.07 121.18   113.34 125.44   107.25 119.36
    2023-07-30   108.00 119.99   107.88 119.88   246.49 258.49   108.57 120.56   112.83 124.83   106.74 118.74
    2023-07-31   107.50 119.39   107.39 119.28   245.99 257.88   108.07 119.96   112.33 124.22   106.25 118.14
    2023-08-01   107.01 118.80   106.89 118.69   245.50 257.29   107.58 119.37   111.84 123.63   105.76 117.55
    2023-08-02   106.52 118.22   106.41 118.10   245.02 256.71   107.09 118.79   111.36 123.05   105.27 116.97
    2023-08-03   106.05 117.65   105.93 117.53   244.54 256.14   106.61 118.21   110.88 122.48   104.79 116.39
    2023-08-04   105.57 117.08   105.46 116.97   244.07 255.58   106.14 117.65   110.41 121.92   104.32 115.83
    2023-08-05   105.11 116.53   104.99 116.42   243.60 255.02   105.68 117.10   109.94 121.37   103.86 115.28
    2023-08-06   104.65 115.99   104.53 115.87   243.14 254.48   105.22 116.56   109.48 120.82   103.40 114.74
    2023-08-07   104.19 115.45   104.08 115.34   242.68 253.94   104.76 116.02   109.03 120.29   102.94 114.20
    2023-08-08   103.74 114.93   103.63 114.81   242.23 253.42   104.31 115.50   108.57 119.76   102.49 113.67
    2023-08-09   103.29 114.41   103.18 114.29   241.79 252.90   103.86 114.98   108.13 119.24   102.04 113.16
    2023-08-10   102.85 113.89   102.74 113.78   241.34 252.39   103.42 114.46   107.69 118.73   101.60 112.64
    2023-08-11   102.41 113.39   102.30 113.28   240.91 251.88   102.98 113.96   107.25 118.22   101.16 112.14
    2023-08-12   101.98 112.89   101.87 112.78   240.47 251.38   102.55 113.46   106.81 117.73   100.73 111.64
    2023-08-13   101.55 112.40   101.43 112.29   240.04 250.89   102.12 112.97   106.38 117.23   100.30 111.15
    2023-08-14   101.12 111.91   101.01 111.80   239.61 250.41   101.69 112.48   105.95 116.75    99.87 110.66
    2023-08-15   100.69 111.43   100.58 111.32   239.19 249.93   101.26 112.00   105.53 116.27    99.44 110.18
    2023-08-16   100.27 110.96   100.16 110.85   238.76 249.45   100.84 111.53   105.11 115.79    99.02 109.71
    2023-08-17    99.85 110.49    99.74 110.38   238.34 248.98   100.42 111.06   104.69 115.33    98.60 109.24
    2023-08-18    99.43 110.03    99.32 109.91   237.93 248.52   100.00 110.60   104.27 114.86    98.18 108.78
    2023-08-19    99.02 109.57    98.91 109.46   237.51 248.06    99.59 110.14   103.85 114.40    97.77 108.32
    2023-08-20    98.60 109.11    98.49 109.00   237.10 247.61    99.17 109.68   103.44 113.95    97.35 107.86
    2023-08-21    98.19 108.66    98.08 108.55   236.68 247.15    98.76 109.23   103.03 113.50    96.94 107.41
    2023-08-22    97.78 108.22    97.67 108.10   236.27 246.71    98.35 108.79   102.61 113.05    96.53 106.96
    2023-08-23    97.37 107.77    97.26 107.66   235.86 246.27    97.94 108.34   102.20 112.61    96.12 106.52
    2023-08-24    96.96 107.33    96.85 107.22   235.45 245.83    97.53 107.90   101.80 112.17    95.71 106.08
    2023-08-25    96.55 106.90    96.44 106.78   235.05 245.39    97.12 107.47   101.39 111.73    95.30 105.65
    2023-08-26    96.15 106.46    96.03 106.35   234.64 244.96    96.72 107.03   100.98 111.30    94.89 105.21
    2023-08-27    95.74 106.03    95.63 105.92   234.23 244.53    96.31 106.60   100.57 110.87    94.49 104.78
    2023-08-28    95.33 105.61    95.22 105.49   233.82 244.10    95.90 106.18   100.17 110.44    94.08 104.36
    2023-08-29    94.92 105.18    94.81 105.07   233.42 243.67    95.49 105.75    99.76 110.02    93.67 103.93
    2023-08-30    94.52 104.76    94.40 104.65   233.01 243.25    95.09 105.33    99.35 109.59    93.27 103.51
    2023-08-31    94.11 104.34    94.00 104.23   232.60 242.83    94.68 104.91    98.94 109.17    92.86 103.09
    2023-09-01    93.70 103.92    93.59 103.81   232.19 242.41    94.27 104.49    98.54 108.75    92.45 102.67
    2023-09-02    93.29 103.50    93.18 103.39   231.78 241.99    93.86 104.07    98.13 108.34    92.04 102.25
    2023-09-03    92.88 103.09    92.77 102.97   231.37 241.58    93.45 103.66    97.72 107.92    91.63 101.84
    2023-09-04    92.47 102.67    92.36 102.56   230.96 241.16    93.04 103.24    97.31 107.51    91.22 101.42
    2023-09-05    92.06 102.26    91.95 102.15   230.55 240.75    92.63 102.83    96.89 107.09    90.81 101.01
    2023-09-06    91.65 101.85    91.53 101.73   230.14 240.34    92.22 102.42    96.48 106.68    90.39 100.60
    2023-09-07    91.23 101.44    91.12 101.32   229.72 239.93    91.80 102.01    96.06 106.27    89.98 100.18
    2023-09-08    90.98 100.86    90.86 100.75   229.47 239.35    91.55 101.43    95.81 105.69    89.73  99.61
    2023-09-09    91.12  99.89    91.00  99.78   229.61 238.38    91.69 100.46    95.95 104.72    89.87  98.64
    2023-09-10    91.25  98.92    91.14  98.81   229.75 237.41    91.82  99.49    96.09 103.75    90.00  97.67

    2024-02-26   275.77 283.17   275.66 283.06    54.26  61.67   276.34 283.74   280.60 288.01   274.52 281.92
    2024-02-27   274.77 283.33   274.66 283.21    53.27  61.82   275.34 283.90   279.61 288.16   273.52 282.08
    2024-02-28   273.78 283.48   273.66 283.37    52.27  61.97   274.35 284.05   278.61 288.32   272.53 282.23
    2024-02-29   273.11 283.31   272.99 283.20    51.60  61.81   273.68 283.88   277.94 288.15   271.86 282.06
    2024-03-01   272.69 282.89   272.58 282.78    51.18  61.38   273.26 283.46   277.52 287.73   271.44 281.64
    2024-03-02   272.27 282.47   272.16 282.36    50.76  60.96   272.84 283.04   277.11 287.31   271.02 281.22
    2024-03-03   271.85 282.05   271.74 281.94    50.34  60.55   272.42 282.62   276.69 286.89   270.60 280.80
    2024-03-04   271.43 281.64   271.32 281.52    49.92  60.13   272.00 282.21   276.27 286.47   270.18 280.39
    2024-03-05   271.01 281.22   270.90 281.11    49.50  59.71   271.58 281.79   275.84 286.05   269.76 279.97
    2024-03-06   270.58 280.80   270.47 280.69    49.08  59.30   271.15 281.37   275.42 285.64   269.33 279.55
    2024-03-07   270.16 280.39   270.05 280.28    48.65  58.88   270.73 280.96   274.99 285.22   268.91 279.14
    2024-03-08   269.73 279.97   269.62 279.86    48.22  58.47   270.30 280.54   274.56 284.81   268.48 278.72
    2024-03-09   269.30 279.56   269.18 279.45    47.79  58.05   269.87 280.13   274.13 284.39   268.05 278.31
    2024-03-10   268.86 279.14   268.75 279.03    47.36  57.64   269.43 279.71   273.70 283.98   267.61 277.89
    2024-03-11   268.43 278.73   268.31 278.62    46.92  57.22   269.00 279.30   273.26 283.56   267.17 277.48
    2024-03-12   267.98 278.31   267.87 278.20    46.48  56.81   268.55 278.88   272.82 283.15   266.73 277.06
    2024-03-13   267.54 277.90   267.43 277.78    46.03  56.39   268.11 278.47   272.37 282.73   266.29 276.65
    2024-03-14   267.09 277.48   266.98 277.37    45.58  55.97   267.66 278.05   271.93 282.31   265.84 276.23
    2024-03-15   266.64 277.06   266.53 276.95    45.13  55.55   267.21 277.63   271.47 281.89   265.39 275.81
    2024-03-16   266.18 276.64   266.07 276.53    44.68  55.13   266.75 277.21   271.02 281.47   264.93 275.39
    2024-03-17   265.72 276.22   265.61 276.10    44.21  54.71   266.29 276.79   270.56 281.05   264.47 274.97
    2024-03-18   265.26 275.79   265.14 275.68    43.75  54.29   265.83 276.36   270.09 280.63   264.01 274.54
    2024-03-19   264.79 275.37   264.67 275.25    43.28  53.86   265.36 275.94   269.62 280.20   263.53 274.12
    2024-03-20   264.31 274.94   264.20 274.83    42.80  53.43   264.88 275.51   269.14 279.77   263.06 273.69
    2024-03-21   263.83 274.51   263.71 274.39    42.32  53.00   264.40 275.08   268.66 279.34   262.58 273.26
    2024-03-22   263.34 274.07   263.23 273.96    41.83  52.57   263.91 274.64   268.17 278.91   262.09 272.82
    2024-03-23   262.84 273.64   262.73 273.52    41.34  52.13   263.41 274.21   267.68 278.47   261.59 272.38
    2024-03-24   262.34 273.19   262.23 273.08    40.84  51.69   262.91 273.76   267.18 278.03   261.09 271.94
    2024-03-25   261.84 272.75   261.72 272.64    40.33  51.24   262.40 273.32   266.67 277.58   260.58 271.50
    2024-03-26   261.32 272.30   261.21 272.19    39.81  50.79   261.89 272.87   266.15 277.14   260.07 271.05
    2024-03-27   260.80 271.85   260.68 271.74    39.29  50.34   261.37 272.42   265.63 276.68   259.55 270.60
    2024-03-28   260.27 271.39   260.15 271.28    38.76  49.88   260.84 271.96   265.10 276.23   259.02 270.14
    2024-03-29   259.73 270.93   259.61 270.82    38.22  49.42   260.30 271.50   264.56 275.76   258.48 269.68
    2024-03-30   259.18 270.46   259.07 270.35    37.67  48.95   259.75 271.03   264.01 275.30   257.93 269.21
    2024-03-31   258.62 269.99   258.51 269.88    37.11  48.48   259.19 270.56   263.46 274.82   257.37 268.74
    2024-04-01   258.05 269.51   257.94 269.40    36.55  48.00   258.62 270.08   262.89 274.35   256.80 268.26
    2024-04-02   257.48 269.03   257.37 268.91    35.97  47.52   258.05 269.60   262.31 273.86   256.23 267.78
    2024-04-03   256.89 268.54   256.78 268.42    35.38  47.03   257.46 269.11   261.72 273.37   255.64 267.28
    2024-04-04   256.29 268.04   256.18 267.93    34.78  46.53   256.86 268.61   261.13 272.87   255.04 266.79
    2024-04-05   255.68 267.53   255.57 267.42    34.18  46.02   256.25 268.10   260.52 272.37   254.43 266.28
    2024-04-06   255.06 267.02   254.95 266.91    33.55  45.51   255.63 267.59   259.90 271.85   253.81 265.77
    2024-04-07   254.43 266.50   254.31 266.39    32.92  44.99   255.00 267.07   259.26 271.33   253.18 265.25
    2024-04-08   253.78 265.97   253.67 265.86    32.27  44.46   254.35 266.54   258.61 270.80   252.53 264.72
    2024-04-09   253.12 265.43   253.01 265.32    31.61  43.92   253.69 266.00   257.95 270.26   251.87 264.18
    2024-04-10   252.44 264.88   252.33 264.77    30.94  43.37   253.01 265.45   257.28 269.72   251.19 263.63
    2024-04-11   251.76 264.32   251.64 264.21    30.25  42.82   252.32 264.89   256.59 269.16   250.50 263.07
    2024-04-12   251.05 263.76   250.94 263.64    29.54  42.25   251.62 264.33   255.88 268.59   249.80 262.50
    2024-04-13   250.33 263.18   250.22 263.06    28.82  41.67   250.90 263.75   255.16 268.01   249.08 261.92
    2024-04-14   249.59 262.58   249.48 262.47    28.08  41.08   250.16 263.15   254.42 267.42   248.34 261.33
    2024-04-15   248.83 261.98   248.72 261.87    27.33  40.47   249.40 262.55   253.67 266.82   247.58 260.73
    2024-04-16   248.06 261.36   247.95 261.25    26.55  39.86   248.63 261.93   252.89 266.20   246.81 260.11
    2024-04-17   247.27 260.73   247.15 260.62    25.76  39.23   247.84 261.30   252.10 265.57   246.02 259.48
    2024-04-18   246.45 260.09   246.34 259.98    24.94  38.58   247.02 260.66   251.29 264.92   245.20 258.84
    2024-04-19   245.62 259.43   245.50 259.32    24.11  37.92   246.19 260.00   250.45 264.26   244.37 258.18
    2024-04-20   244.76 258.76   244.65 258.64    23.25  37.25   245.33 259.33   249.59 263.59   243.51 257.50
    2024-04-21   243.88 258.06   243.77 257.95    22.37  36.56   244.45 258.63   248.72 262.90   242.63 256.81
    2024-04-22   242.98 257.35   242.86 257.24    21.47  35.85   243.55 257.92   247.81 262.19   241.73 256.10

    2024-07-13   117.51 131.89   117.40 131.78   256.00 270.38   118.08 132.46   122.35 136.72   116.26 130.64
    2024-07-14   116.83 131.01   116.71 130.90   255.32 269.51   117.40 131.58   121.66 135.85   115.57 129.76
    2024-07-15   116.16 130.16   116.04 130.05   254.65 268.66   116.73 130.73   120.99 135.00   114.90 128.91
    2024-07-16   115.50 129.33   115.39 129.22   253.99 267.83   116.07 129.90   120.34 134.17   114.25 128.08
    2024-07-17   114.86 128.52   114.75 128.41   253.35 267.02   115.43 129.09   119.70 133.36   113.61 127.27
    2024-07-18   114.24 127.74   114.12 127.62   252.73 266.23   114.81 128.31   119.07 132.57   112.99 126.48
    2024-07-19   113.62 126.97   113.51 126.85   252.12 265.46   114.19 127.54   118.46 131.80   112.37 125.71
    2024-07-20   113.03 126.21   112.91 126.10   251.52 264.71   113.60 126.78   117.86 131.05   111.77 124.96
    2024-07-21   112.44 125.48   112.33 125.37   250.93 263.97   113.01 126.05   117.27 130.31   111.19 124.23
    2024-07-22   111.86 124.76   111.75 124.65   250.36 263.25   112.43 125.33   116.70 129.60   110.61 123.51
    2024-07-23   111.30 124.06   111.19 123.95   249.79 262.55   111.87 124.63   116.13 128.89   110.05 122.81
    2024-07-24   110.75 123.37   110.63 123.26   249.24 261.87   111.32 123.94   115.58 128.21   109.49 122.12
    2024-07-25   110.20 122.70   110.09 122.59   248.69 261.19   110.77 123.27   115.04 127.54   108.95 121.45
    2024-07-26   109.67 122.04   109.56 121.93   248.16 260.54   110.24 122.61   114.50 126.88   108.42 120.79
    2024-07-27   109.14 121.40   109.03 121.29   247.63 259.89   109.71 121.97   113.98 126.23   107.89 120.15
    2024-07-28   108.63 120.77   108.51 120.66   247.12 259.26   109.20 121.34   113.46 125.60   107.37 119.52
    2024-07-29   108.12 120.15   108.00 120.04   246.61 258.64   108.69 120.72   112.95 124.98   106.87 118.90
    2024-07-30   107.62 119.54   107.50 119.43   246.11 258.03   108.19 120.11   112.45 124.38   106.37 118.29
    2024-07-31   107.12 118.94   107.01 118.83   245.61 257.44   107.69 119.51   111.96 123.78   105.87 117.69
    2024-08-01   106.64 118.36   106.52 118.25   245.13 256.85   107.21 118.93   111.47 123.19   105.38 117.11
    2024-08-02   106.16 117.78   106.04 117.67   244.65 256.28   106.73 118.35   110.99 122.62   104.90 116.53
    2024-08-03   105.68 117.22   105.57 117.11   244.17 255.71   106.25 117.79   110.52 122.05   104.43 115.97
    2024-08-04   105.21 116.66   105.10 116.55   243.71 255.15   105.78 117.23   110.05 121.50   103.96 115.41
    2024-08-05   104.75 116.12   104.64 116.00   243.24 254.61   105.32 116.69   109.58 120.95   103.50 114.86
    2024-08-06   104.29 115.58   104.18 115.46   242.78 254.07   104.86 116.15   109.13 120.41   103.04 114.33
    2024-08-07   103.84 115.05   103.73 114.93   242.33 253.54   104.41 115.62   108.67 119.88   102.59 113.80
    2024-08-08   103.39 114.52   103.28 114.41   241.88 253.02   103.96 115.09   108.23 119.36   102.14 113.27
    2024-08-09   102.95 114.01   102.83 113.90   241.44 252.50   103.52 114.58   107.78 118.84   101.70 112.76
    2024-08-10   102.51 113.50   102.39 113.39   241.00 251.99   103.08 114.07   107.34 118.34   101.26 112.25
    2024-08-11   102.07 113.00   101.96 112.89   240.56 251.49   102.64 113.57   106.91 117.84   100.82 111.75
    2024-08-12   101.64 112.51   101.53 112.40   240.13 251.00   102.21 113.08   106.47 117.34   100.39 111.26
    2024-08-13   101.21 112.02   101.10 111.91   239.70 250.51   101.78 112.59   106.04 116.85    99.96 110.77
    2024-08-14   100.78 111.54   100.67 111.43   239.27 250.03   101.35 112.11   105.62 116.37    99.53 110.29
    2024-08-15   100.36 111.06   100.25 110.95   238.85 249.55   100.93 111.63   105.19 115.90    99.11 109.81
    2024-08-16    99.94 110.59    99.82 110.48   238.43 249.08   100.51 111.16   104.77 115.42    98.69 109.34
    2024-08-17    99.52 110.12    99.41 110.01   238.01 248.62   100.09 110.69   104.35 114.96    98.27 108.87
    2024-08-18    99.10 109.66    98.99 109.55   237.59 248.16    99.67 110.23   103.94 114.50    97.85 108.41
    2024-08-19    98.69 109.21    98.57 109.09   237.18 247.70    99.26 109.78   103.52 114.04    97.44 107.96
    2024-08-20    98.27 108.75    98.16 108.64   236.77 247.25    98.84 109.32   103.11 113.59    97.02 107.50
    2024-08-21    97.86 108.31    97.75 108.19   236.35 246.80    98.43 108.88   102.70 113.14    96.61 107.06
    2024-08-22    97.45 107.86    97.34 107.75   235.94 246.35    98.02 108.43   102.29 112.70    96.20 106.61
    2024-08-23    97.04 107.42    96.93 107.31   235.53 245.91    97.61 107.99   101.88 112.26    95.79 106.17
    2024-08-24    96.63 106.98    96.52 106.87   235.13 245.48    97.20 107.55   101.47 111.82    95.38 105.73
    2024-08-25    96.23 106.55    96.11 106.44   234.72 245.04    96.79 107.12   101.06 111.38    94.97 105.30
    2024-08-26    95.82 106.12    95.71 106.01   234.31 244.61    96.39 106.69   100.65 110.95    94.57 104.87
    2024-08-27    95.41 105.69    95.30 105.58   233.90 244.18    95.98 106.26   100.24 110.53    94.16 104.44
    2024-08-28    95.00 105.27    94.89 105.15   233.50 243.76    95.57 105.83    99.84 110.10    93.75 104.01
    2024-08-29    94.60 104.84    94.48 104.73   233.09 243.33    95.17 105.41    99.43 109.68    93.35 103.59
    2024-08-30    94.19 104.42    94.08 104.31   232.68 242.91    94.76 104.99    99.02 109.25    92.94 103.17
    2024-08-31    93.78 104.00    93.67 103.89   232.27 242.49    94.35 104.57    98.61 108.84    92.53 102.75
    2024-09-01    93.37 103.58    93.26 103.47   231.86 242.08    93.94 104.15    98.21 108.42    92.12 102.33
    2024-09-02    92.96 103.17    92.85 103.05   231.45 241.66    93.53 103.74    97.80 108.00    91.71 101.92
    2024-09-03    92.55 102.75    92.44 102.64   231.04 241.25    93.12 103.32    97.39 107.59    91.30 101.50
    2024-09-04    92.14 102.34    92.03 102.23   230.63 240.83    92.71 102.91    96.97 107.17    90.89 101.09
    2024-09-05    91.73 101.93    91.61 101.81   230.22 240.42    92.30 102.50    96.56 106.76    90.48 100.68
    2024-09-06    91.31 101.52    91.20 101.40   229.80 240.01    91.88 102.09    96.15 106.35    90.06 100.26
    2024-09-07    90.93 101.07    90.81 100.96   229.42 239.56    91.50 101.64    95.76 105.91    89.68  99.82
    2024-09-08    91.07 100.10    90.95  99.99   229.56 238.59    91.64 100.67    95.90 104.94    89.82  98.85
    2024-09-09    91.21  99.13    91.09  99.02   229.70 237.62    91.78  99.70    96.04 103.96    89.96  97.88

# Troubleshooting

Unless specified, astroquery will cache queries to Horizons. This can cause some issues when trying to query targets
when changes are made upstream at JPL. To find and clear the cache, you can use the astropy.

    >>> import astropy
    >>> path = astropy.config.get_cache_dir()
    >>> print(path)

This will print where astropy is caching all of it's data. If astroquery has been used to search for targets in the past,
`<value_of_path>/astroquery/Horizons/` will exist. Now you have the full path and can remove it manually.

You can also use our `delete_cache` utility function. After the software is installed you can enter:
   
    $ delete_cache

If cached results exist you will be prompted whether you want to delete the cache folder or not.


# Contributing

The `jwst_gtvt` follows the STScI ["forking workflow"](https://github.com/spacetelescope/style-guides/blob/master/guides/git-workflow.md#forking-workflow).


# Citation

If you use the JWST General Target Visibility Tool (GTVT) or JWST Moving Target Visibility Tool (MTVT) for work/research presented in a publication, or as a dependency in a software package, we recommend and encourage the following acknowledgment:

This work made use of the JWST General Target Visibility Tool (GTVT) and/or the JWST Moving Target Visibility Tool (MTVT) originally written by Matthew Hill, expanded for moving targets by Michael Kelley, and maintained by Mees Fix.