# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['ikea_api',
 'ikea_api.endpoints',
 'ikea_api.executors',
 'ikea_api.wrappers',
 'ikea_api.wrappers.parsers']

package_data = \
{'': ['*']}

extras_require = \
{':python_version < "3.10"': ['typing-extensions>=4.0.0,<5.0.0'],
 'all': ['requests', 'pydantic>=1.8.0', 'httpx>=0.22.0'],
 'httpx': ['httpx>=0.22.0'],
 'requests': ['requests'],
 'wrappers': ['pydantic>=1.8.0']}

setup_kwargs = {
    'name': 'ikea-api',
    'version': '2.0.1',
    'description': "Client for several IKEA's APIs",
    'long_description': 'Client for several IKEA APIs.\n\n[![Test](https://github.com/vrslev/ikea-api-client/actions/workflows/test.yml/badge.svg)](https://github.com/vrslev/ikea-api-client/actions/workflows/test.yml)\n[![Python](https://img.shields.io/pypi/pyversions/ikea_api?label=Python)](https://pypi.org/project/ikea_api)\n[![Downloads](https://img.shields.io/pypi/dm/ikea_api?label=Downloads&color=blueviolet)](https://pypi.org/project/ikea_api)\n\n# Features\n\nWith this library you can access following IKEA\'s APIs:\n\n- Cart,\n- Home Delivery and Collect Services (actually, Order Capture),\n- Items info (3 different services),\n- Purchases (history and order info),\n- Search,\n- Stock.\n\nAlso the package:\n\n- Is **backend agnostic**: choose HTTP library you want (async, too!),\n- Is **fully typed and tested**,\n- Has optional wrappers around some APIs based on Pydantic.\n\n# Installation\n\n```bash\npip install ikea_api\n```\n\n- Use [httpx](https://www.python-httpx.org) — awesome async HTTP library —\xa0as backend:\n\n```bash\npip install "ikea_api[httpx]"\n```\n\n- Use [requests](https://docs.python-requests.org)\xa0as backend:\n\n```bash\npip install "ikea_api[requests]"\n```\n\n- Use wrappers:\n\n```bash\npip install "ikea_api[wrappers]"\n```\n\n- Install everything:\n\n```bash\npip install "ikea_api[all]"\n```\n\n# Usage\n\n_ikea_api_ is unusual API client. It decouples I/O from logic for easier testing and maintenance. As a bonus, you can use literally _any_ HTTP library.\nLet\'s have a look at how to work with ikea_api.\n\n```python\nimport ikea_api\n\n# Constants like country, language, base url\nconstants = ikea_api.Constants(country="us", language="en")\n# Search API\nsearch = ikea_api.Search(constants)\n# Search endpoint with prepared data\nendpoint = search.search("Billy")\n```\n\nAs you can see, nothing happened up to this point. Code suggests that we already should get the result of the search but we don\'t. What happened is `search()` returned a data class that contains information about endpoint that can be interpreted by an endpoint runner. There are two built-in: for [requests](https://docs.python-requests.org) (sync) and [httpx](https://www.python-httpx.org) (async), but you can easily write one yourself.\n\nHere\'s how you would use _requests_ one:\n\n```python\nikea_api.run(endpoint)\n```\n\nAnd _httpx_ one:\n\n```python\nawait ikea_api.run_async(endpoint)\n```\n\n`ikea_api.run_async()` is async function, so you have to "await" it or run using `asyncio.run()`.\n\n## Endpoints reference\n\n### 🔑 Authorization\n\nFirst time you open ikea.com, guest token is being generated and stored in cookies. Same thing must be done in here before using any endpoint.\n\nThis token expires in 30 days.\n\n```python\nikea_api.Auth(constants).get_guest_token()\n```\n\nPreviously you could login as user (with login and password), but now there\'s very advanced telemetry that I wouldn\'t be able to solve in hundred years 🤪\n\n### 🛒 Cart\n\nWith this endpoint you can do everything you can using IKEA\'s frontend:\n\n```python\ncart = ikea_api.Cart(constants, token=...)\n```\n\n- Show the cart\n\n```python\ncart.show()\n```\n\n- Clear it\n\n```python\ncart.clear()\n```\n\n- Add, update and delete items\n\n```python\ncart.add_items({"30457903": 1})  # { item_code: quantity }\n\ncart.update_items({"30457903": 5})\n\ncart.remove_items(["30457903"])\n```\n\n- Set and clear coupon\n\n```python\ncart.set_coupon(...)\n\ncart.clear_coupon()\n```\n\n- and even copy another user\'s cart.\n\n```python\ncart.copy_items(source_user_id=...)\n```\n\nYou can edit your user\'s actual cart if you use authorized token (copy-paste from cookies).\n\n> 💡\xa0There\'s wrapper that clears current cart and adds items with error handling: if requested item doesn\'t exist, the function just skips it and tries again.\n>\n> ```python\n> ikea_api.add_items_to_cart(  # Function returns items that can\'t be added. In this case: [\'11111111\']\n>     cart=cart,\n>     items={\n>         "30457903": 1,\n>         "11111111": 2,  # invalid item that will be skipped\n>     },\n> )\n> ```\n\n### 🚛 Order Capture\n\nCheck pickup or delivery availability. If you need to know whether items are available _in stores_, use [Item availability endpoint](#%F0%9F%9F%A2-item-availability) or [ikea-availability-checker](https://github.com/Ephigenia/ikea-availability-checker).\n\n```python\norder = ikea_api.OrderCapture(constants, token=token)\n\ncart_show = run(cart.show())\nitems = ikea_api.convert_cart_to_checkout_items(cart_show)\n\ncheckout_id = run(order.get_checkout(items))\nservice_area_id = run(\n    order.get_service_area(\n        checkout_id,\n        zip_code="02215",\n        state_code="MA",  # pass State Code only if your country has them\n    )\n)\nhome_services = run(order.get_home_delivery_services(checkout_id, service_area_id))\ncollect_services = run(\n    order.get_collect_delivery_services(checkout_id, service_area_id)\n)\n```\n\n> 💡\xa0You can use wrapper to add items to cart (clearing cart before and handling unknown item errors if they appear) and parse response in nice Pydantic models:\n>\n> ```python\n> services = await ikea_api.get_delivery_services(\n>     constants=constants,\n>     token=...,\n>     items={\n>         "30457903": 1,\n>         "11111111": 2,  # invalid item that will be skipped\n>     },\n>     zip_code="101000",\n> )\n> services.delivery_options  # List of parsed delivery services\n> services.cannot_add  # [\'11111111\']\n> ```\n\n### 📦 Purchases\n\n```python\npurchases = ikea_api.Purchases(constants, token=token)\n```\n\n#### History\n\nThis method requires authentication, so if you don\'t have authorized token, it won\'t work.\n\n```python\npurchases.history()\n\n# Get all purchases:\npurchases.history(take=10000)\n\n# Pagination:\npurchases.history(take=10, skip=1)\n```\n\n> 💡 Get parsed response with the wrapper:\n>\n> ```python\n> ikea_api.get_purchase_history(purchases)  # Returns a list of parsed purchases\n> ```\n\n#### Order info\n\n```python\npurchases.order_info(order_number=..., email=...)\n\n# If you have authorized token, you can drop email:\npurchases.order_info(order_number="111111111")\n\n# The method also has other params but they\'re mostly internal:\npurchases.order_info(\n    order_number=...,\n    email=...,\n    queries=...,  # Queries that will be included in request, combine any of: ["StatusBannerOrder", "CostsOrder", "ProductListOrder"]. By default, all of them are included.\n    # Params below are relevant to ProductListOrder\n    skip_products=...,\n    skip_product_prices=...,\n    take_products=...,\n)\n```\n\n> 💡 Get parsed response with the wrapper:\n>\n> ```python\n> ikea_api.get_purchase_info(  # Returns parsed purchase object. Items are not listed.\n>    purchases=purchases,\n>    order_number=...,\n>    email=...,\n> )\n> ```\n\n### 🪑 Item info\n\nGet item specification by item code (product number or whatever). There are 3 endpoints to do this because you can\'t get all the data about all the items using only one endpoint.\n\n```python\niows_items = ikea_api.IowsItems(constants)\niows_items.get_items(["30457903"])\n\ningka_items = ikea_api.IngkaItems(constants)\ningka_items.get_items(["30457903"])\n\npip_item = ikea_api.PipItem(constants)\npip_item.get_item("30457903")\n```\n\n> 💡 You probably won\'t want to use raw APIs when there\'s convenient "smart" wrapper that combines them all and parses basic info:\n>\n> ```python\n> ikea_api.get_items(["30457903"])\n> ```\n\n### 🟢 Item availability\n\nGet availability by item code (product number or whatever).\n\n```python\nstock = ikea_api.Stock(constants)\nstock.get_stock("30457903")\n```\n\n### 🔎 Search\n\nSearch for products in the product catalog by product name. Optionally also specify a maximum amount of returned search results (defaults to 24) and types of required search results.\n\n```python\nsearch = ikea_api.Search(constants)\nsearch.search("Billy")\n\n# Retrieve 10 search results (default is 24)\nsearch.search("Billy", limit=10)\n\n# Configure search results types\nsearch.search(\n    "Billy",\n    types=...,  # Combine any of: ["PRODUCT", "CONTENT", "PLANNER", "REFINED_SEARCHES", "ANSWER"]\n)\n```\n\n### 🛠 Utilities\n\n#### Parse item codes\n\nParse item codes from string or list.\n\n```python\nassert ikea_api.parse_item_codes("111.111.11") == ["11111111"]\nassert ikea_api.parse_item_codes("11111111, 222.222.22") == ["11111111", "22222222"]\nassert ikea_api.parse_item_codes("111") == []\n```\n\n#### Format item code\n\nParse item code and format it.\n\n```python\nassert ikea_api.format_item_code("11111111") == "111.111.11"\nassert ikea_api.format_item_code("111-111-11") == "111.111.11"\nassert ikea_api.format_item_code("111.111.11") == "111.111.11"\n```\n',
    'author': 'Lev Vereshchagin',
    'author_email': 'mail@vrslev.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/vrslev/ikea-api-client',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
