"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
// tslint:disable: object-literal-key-quotes
describe('LambdaInvoke', () => {
    let stack;
    let lambdaFunction;
    beforeEach(() => {
        // GIVEN
        stack = new core_1.Stack();
        lambdaFunction = new lambda.Function(stack, 'Fn', {
            code: lambda.Code.fromInline('foo'),
            handler: 'handler',
            runtime: lambda.Runtime.NODEJS_12_X,
        });
    });
    test('default settings', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                'Payload.$': '$',
            },
        });
    });
    test('optional settings', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payload: sfn.TaskInput.fromObject({
                foo: 'bar',
            }),
            invocationType: lib_1.LambdaInvocationType.REQUEST_RESPONSE,
            clientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            qualifier: '1',
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            End: true,
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                Payload: {
                    foo: 'bar',
                },
                InvocationType: 'RequestResponse',
                ClientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
                Qualifier: '1',
            },
        });
    });
    test('invoke Lambda function and wait for task token', () => {
        // GIVEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: sfn.TaskInput.fromObject({
                token: sfn.Context.taskToken,
            }),
            qualifier: 'my-alias',
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke.waitForTaskToken',
                    ],
                ],
            },
            End: true,
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                Payload: {
                    'token.$': '$$.Task.Token',
                },
                Qualifier: 'my-alias',
            },
        });
    });
    test('pass part of state input as input to Lambda function ', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payload: sfn.TaskInput.fromDataAt('$.foo'),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            End: true,
            Parameters: {
                'FunctionName': {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                'Payload.$': '$.foo',
            },
        });
    });
    test('fails when WAIT_FOR_TASK_TOKEN integration pattern is used without supplying a task token in payload', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            });
        }).toThrow(/Task Token is required in `payload` for callback. Use Context.taskToken to set the token./);
    });
    test('fails when RUN_JOB integration pattern is used', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                integrationPattern: sfn.IntegrationPattern.RUN_JOB,
            });
        }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,WAIT_FOR_TASK_TOKEN. Received: RUN_JOB/);
    });
});
//# sourceMappingURL=data:application/json;base64,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