"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const task_utils_1 = require("../private/task-utils");
/**
 * Invoke a Lambda function as a Task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class LambdaInvoke extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is required in `payload` for callback. Use Context.taskToken to set the token.');
        }
        this.taskMetrics = {
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: {
                LambdaFunctionArn: this.props.lambdaFunction.functionArn,
                ...(this.props.qualifier && { Qualifier: this.props.qualifier }),
            },
        };
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.props.lambdaFunction.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
    }
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('lambda', 'invoke', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                FunctionName: this.props.lambdaFunction.functionArn,
                Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier,
            }),
        };
    }
}
exports.LambdaInvoke = LambdaInvoke;
LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
/**
 * Invocation type of a Lambda
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * Invoke the function synchronously.
     *
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke the function asynchronously.
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * Validate parameter values and verify that the user or role has permission to invoke the function.
     */
    LambdaInvocationType["DRY_RUN"] = "DryRun";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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