"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 */
class BatchSubmitJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS);
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        // tslint:disable-next-line:no-unused-expression
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`attempt duration must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is required since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_b = props.containerOverrides) === null || _b === void 0 ? void 0 : _b.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
        this.taskPolicies = this.configurePolicyStatements();
    }
    renderTask() {
        var _a;
        return {
            Resource: task_utils_1.integrationResourceArn('batch', 'submitJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: (_a = this.props.payload) === null || _a === void 0 ? void 0 : _a.value,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            }),
            TimeoutSeconds: undefined,
        };
    }
    configurePolicyStatements() {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a, _b;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: (_b = containerOverrides.memory) === null || _b === void 0 ? void 0 : _b.toMebibytes(),
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.BatchSubmitJob = BatchSubmitJob;
BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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