import re

from pkg_resources import resource_filename
from typing import Set, List, TYPE_CHECKING

if TYPE_CHECKING:
    from antlr4.atn.ATN import ATN

from antlr4.atn import ATNDeserializer
from antlr4.PredictionContext import PredictionContextCache
from antlr4.Lexer import Lexer
from antlr4.dfa.DFA import DFA
from antlr4.InputStream import InputStream
from antlr4.Token import Token, CommonToken
from queue import Queue

from zemberek.core.turkish import TurkishAlphabet
from zemberek.tokenization.antlr.custom_lexer_ATN_simulator import CustomLexerATNSimulator


class TurkishLexer(Lexer):
    _ATN: 'ATN'
    abbreviations: Set[str] = set()
    fi = resource_filename("zemberek", "resources/abbreviations.txt")
    with open(fi, "r", encoding="utf-8") as f:
        for line in f:
            if len(line.strip()) > 0:
                abbr = re.sub("\\s+", "", line.strip())
                if abbr.endswith("."):
                    abbreviations.add(abbr)
                    abbreviations.add(abbr.lower())
                    abbreviations.add(abbr.translate(TurkishAlphabet.lower_map).lower())
    del f
    _decision_to_DFA: List[DFA]
    _shared_context_cache = PredictionContextCache()

    def __init__(self, inp: InputStream):
        super().__init__(inp)
        self._interp = CustomLexerATNSimulator(self, self._ATN, self._decision_to_DFA, self._shared_context_cache)
        self.queue = Queue()

    def nextToken(self) -> Token:
        if not self.queue.empty():
            return self.queue.get(block=False)
        else:
            next_: Token = super(TurkishLexer, self).nextToken()
            if next_.type != 16:
                return next_
            else:
                next2: Token = super(TurkishLexer, self).nextToken()
                if next2.type == 19 and next2.text == ".":
                    abbrev = next_.text + "."
                    if self.abbreviations and abbrev in self.abbreviations:
                        common_token: CommonToken = CommonToken(type=1)
                        common_token.text = abbrev
                        common_token.start = next_.start
                        common_token.stop = next2.stop
                        common_token.tokenIndex = next_.tokenIndex
                        common_token.column = next_.column
                        common_token.line = next_.line
                        return common_token

                self.queue.put_nowait(next2)
                return next_


TurkishLexer._ATN = ATNDeserializer.ATNDeserializer().deserialize(
    "\u0003悋Ꜫ脳맭䅼㯧瞆奤\u0002\u0017ȩ\b\u0001\u0004\u0002\t\u0002\u0004\u0003\t\u0003\u0004\u0004\t\u0004\u0004\u0005\t"
    "\u0005\u0004\u0006\t\u0006\u0004\u0007\t\u0007\u0004\b\t\b\u0004\t\t\t\u0004\n\t\n\u0004\u000b\t\u000b\u0004\f\t"
    "\f\u0004\r\t\r\u0004\u000e\t\u000e\u0004\u000f\t\u000f\u0004\u0010\t\u0010\u0004\u0011\t\u0011\u0004\u0012\t"
    "\u0012\u0004\u0013\t\u0013\u0004\u0014\t\u0014\u0004\u0015\t\u0015\u0004\u0016\t\u0016\u0004\u0017\t\u0017\u0004"
    "\u0018\t\u0018\u0004\u0019\t\u0019\u0004\u001a\t\u001a\u0004\u001b\t\u001b\u0004\u001c\t\u001c\u0004\u001d\t"
    "\u001d\u0004\u001e\t\u001e\u0004\u001f\t\u001f\u0004 \t "
    "\u0004!\t!\u0004\"\t\"\u0003\u0002\u0003\u0002\u0003\u0003\u0003\u0003\u0003\u0004\u0003\u0004\u0003\u0005\u0003"
    "\u0005\u0003\u0006\u0003\u0006\u0003\u0007\u0003\u0007\u0003\b\u0003\b\u0003\t\u0003\t\u0003\n\u0003\n\u0006\nX"
    "\n\n\r\n\u000e\nY\u0003\u000b\u0006\u000b]\n\u000b\r\u000b\u000e\u000b^\u0003\f\u0003\f\u0003\r\u0003\r\u0003\r"
    "\u0003\r\u0003\r\u0003\r\u0003\r\u0003\r\u0005\rk\n\r\u0003\r\u0005\rn\n\r\u0003\u000e\u0005\u000eq\n\u000e"
    "\u0003\u000e\u0003\u000e\u0003\u000e\u0005\u000ev\n\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003"
    "\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0005\u000e\u0084\n"
    "\u000e\u0003\u000e\u0005\u000e\u0087\n\u000e\u0003\u000e\u0005\u000e\u008a\n\u000e\u0003\u000e\u0003\u000e\u0003"
    "\u000e\u0005\u000e\u008f\n\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003"
    "\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0003\u000e\u0005\u000e\u009d\n\u000e\u0003\u000e\u0005"
    "\u000e \n\u000e\u0005\u000e¢\n\u000e\u0003\u000f\u0003\u000f\u0003\u000f\u0003\u0010\u0005\u0010¨\n\u0010\u0003"
    "\u0010\u0003\u0010\u0003\u0010\u0003\u0010\u0005\u0010®\n\u0010\u0003\u0010\u0005\u0010±\n\u0010\u0003\u0010"
    "\u0005\u0010´\n\u0010\u0003\u0010\u0003\u0010\u0003\u0010\u0005\u0010¹\n\u0010\u0003\u0010\u0005\u0010¼\n\u0010"
    "\u0003\u0010\u0003\u0010\u0005\u0010À\n\u0010\u0003\u0010\u0005\u0010Ã\n\u0010\u0003\u0010\u0003\u0010\u0003"
    "\u0010\u0003\u0010\u0005\u0010É\n\u0010\u0003\u0010\u0003\u0010\u0003\u0010\u0006\u0010Î\n\u0010\r\u0010\u000e"
    "\u0010Ï\u0003\u0010\u0003\u0010\u0005\u0010Ô\n\u0010\u0003\u0010\u0003\u0010\u0003\u0010\u0006\u0010Ù\n\u0010\r"
    "\u0010\u000e\u0010Ú\u0003\u0010\u0003\u0010\u0005\u0010ß\n\u0010\u0003\u0010\u0003\u0010\u0005\u0010ã\n\u0010"
    "\u0003\u0010\u0005\u0010æ\n\u0010\u0005\u0010è\n\u0010\u0003\u0011\u0006\u0011ë\n\u0011\r\u0011\u000e\u0011"
    "ì\u0003\u0012\u0003\u0012\u0005\u0012ñ\n\u0012\u0003\u0012\u0003\u0012\u0003\u0013\u0006\u0013ö\n\u0013\r\u0013"
    "\u000e\u0013÷\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014"
    "\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0005\u0014ĉ\n\u0014\u0003"
    "\u0014\u0003\u0014\u0005\u0014č\n\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014"
    "\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014"
    "\u0005\u0014Ğ\n\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0005"
    "\u0014ħ\n\u0014\u0003\u0014\u0006\u0014Ī\n\u0014\r\u0014\u000e\u0014ī\u0003\u0014\u0003\u0014\u0003\u0014\u0003"
    "\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003"
    "\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0003"
    "\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0005\u0014Ň\n\u0014\u0003\u0014\u0003\u0014\u0003\u0014\u0005\u0014"
    "Ō\n\u0014\u0003\u0014\u0003\u0014\u0005\u0014Ő\n\u0014\u0003\u0014\u0005\u0014œ\n\u0014\u0005\u0014ŕ\n\u0014"
    "\u0003\u0015\u0006\u0015Ř\n\u0015\r\u0015\u000e\u0015ř\u0003\u0015\u0005\u0015ŝ\n\u0015\u0003\u0015\u0006\u0015"
    "Š\n\u0015\r\u0015\u000e\u0015š\u0003\u0015\u0003\u0015\u0006\u0015Ŧ\n\u0015\r\u0015\u000e\u0015ŧ\u0003\u0015"
    "\u0003\u0015\u0006\u0015Ŭ\n\u0015\r\u0015\u000e\u0015ŭ\u0006\u0015Ű\n\u0015\r\u0015\u000e\u0015ű\u0003\u0015"
    "\u0005\u0015ŵ\n\u0015\u0003\u0016\u0003\u0016\u0006\u0016Ź\n\u0016\r\u0016\u000e\u0016ź\u0003\u0016\u0005\u0016"
    "ž\n\u0016\u0003\u0017\u0003\u0017\u0006\u0017Ƃ\n\u0017\r\u0017\u000e\u0017ƃ\u0003\u0017\u0005\u0017Ƈ\n\u0017"
    "\u0003\u0018\u0003\u0018\u0006\u0018Ƌ\n\u0018\r\u0018\u000e\u0018ƌ\u0003\u0018\u0003\u0018\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003\u0019\u0003"
    "\u0019\u0003\u0019\u0005\u0019ǜ\n\u0019\u0003\u001a\u0006\u001aǟ\n\u001a\r\u001a\u000e\u001aǠ\u0003\u001a\u0005"
    "\u001aǤ\n\u001a\u0003\u001a\u0005\u001aǧ\n\u001a\u0003\u001b\u0003\u001b\u0003\u001b\u0006\u001bǬ\n\u001b\r"
    "\u001b\u000e\u001bǭ\u0003\u001b\u0005\u001bǱ\n\u001b\u0003\u001b\u0005\u001bǴ\n\u001b\u0003\u001c\u0006\u001cǷ\n"
    "\u001c\r\u001c\u000e\u001cǸ\u0003\u001d\u0006\u001dǼ\n\u001d\r\u001d\u000e\u001dǽ\u0003\u001e\u0006\u001eȁ\n"
    "\u001e\r\u001e\u000e\u001eȂ\u0003\u001e\u0005\u001eȆ\n\u001e\u0003\u001e\u0006\u001eȉ\n\u001e\r\u001e\u000e"
    "\u001eȊ\u0003\u001e\u0005\u001eȎ\n\u001e\u0003\u001f\u0003\u001f\u0003\u001f\u0003\u001f\u0003\u001f\u0003\u001f"
    "\u0003\u001f\u0003\u001f\u0003\u001f\u0003\u001f\u0003\u001f\u0003\u001f\u0005\u001fȜ\n\u001f\u0003 \u0003 "
    "\u0003!\u0006!ȡ\n!\r!\u000e!Ȣ\u0003\"\u0006\"Ȧ\n\"\r\"\u000e\"ȧ\u0003ȧ\u0002#\u0003\u0002\u0005\u0002\u0007"
    "\u0002\t\u0002\u000b\u0002\r\u0002\u000f\u0002\u0011\u0002\u0013\u0002\u0015\u0004\u0017\u0005\u0019\u0006\u001b"
    "\u0007\u001d\b\u001f\t!\u0002#\u0002%\u0002'\n)\u000b+\f-\r/\u000e1\u000f3\u00105\u00117\u00129\u0013;\u0014"
    "=\u0002?\u0015A\u0016C\u0017\u0003\u0002\u001d\u0003\u00022;\u000b\u0002c|ääééððøøýþġġĳĳšš\u000b\u0002C"
    "\\ÄÄÉÉÐÐØØÝÞĠĠĲĲŠŠ\u0011\u0002C\\c|ÄÄÉÉÐÐØØÝÞääééððøøýþĠġĲĳŠš\u0012\u00022;C\\c|ÄÄÉÉÐÐØØÝÞääééððøøýþĠġĲĳŠš\u0013"
    "\u00022;C\\aac|ÄÄÉÉÐÐØØÝÞääééððøøýþĠġĲĳŠš\u0004\u0002))‛‛\u0006\u0002$$\u00ad\u00ad½½„‟\u0004\u0002\u000b\u000b"
    "\"\"\u0004\u0002\f\f\u000f\u000f\u0003\u000224\u0004\u000200<<\u0003\u000227\u0003\u000225\u0003\u000223\u0003"
    "\u000233\u0003\u00029;\u0003\u000244\u0003\u000222\u0004\u0002--//\u0004\u0002..00\u0004\u0002GGgg\u0019\u0002(("
    "--/;==??AAC]__aac|ÄÄÉÉÐÐØØÝÞääééððøøýþĠġĲĳŠš\u0006\u00022;C\\aac|\u0007\u0002EFKKNOXXZZ\u0010\u0002##&("
    "*1<=?B]`}}\u007f\u007f««°°‚‚\u2028\u2028™™ℤℤ\u0014\u0002\u000b\f\u000f\u000f\"$&1<=?B]`}}\u007f\u007f««\u00ad"
    "\u00ad°°½½‚‛„‟\u2028\u2028™™ℤℤ\u0002ʋ\u0002\u0015\u0003\u0002\u0002\u0002\u0002\u0017\u0003\u0002\u0002\u0002"
    "\u0002\u0019\u0003\u0002\u0002\u0002\u0002\u001b\u0003\u0002\u0002\u0002\u0002\u001d\u0003\u0002\u0002\u0002"
    "\u0002\u001f\u0003\u0002\u0002\u0002\u0002'\u0003\u0002\u0002\u0002\u0002)\u0003\u0002\u0002\u0002\u0002+\u0003"
    "\u0002\u0002\u0002\u0002-\u0003\u0002\u0002\u0002\u0002/\u0003\u0002\u0002\u0002\u00021\u0003\u0002\u0002\u0002"
    "\u00023\u0003\u0002\u0002\u0002\u00025\u0003\u0002\u0002\u0002\u00027\u0003\u0002\u0002\u0002\u00029\u0003\u0002"
    "\u0002\u0002\u0002;\u0003\u0002\u0002\u0002\u0002?\u0003\u0002\u0002\u0002\u0002A\u0003\u0002\u0002\u0002\u0002C"
    "\u0003\u0002\u0002\u0002\u0003E\u0003\u0002\u0002\u0002\u0005G\u0003\u0002\u0002\u0002\u0007I\u0003\u0002\u0002"
    "\u0002\tK\u0003\u0002\u0002\u0002\u000bM\u0003\u0002\u0002\u0002\rO\u0003\u0002\u0002\u0002\u000fQ\u0003\u0002"
    "\u0002\u0002\u0011S\u0003\u0002\u0002\u0002\u0013U\u0003\u0002\u0002\u0002\u0015\\\u0003\u0002\u0002\u0002\u0017"
    "`\u0003\u0002\u0002\u0002\u0019b\u0003\u0002\u0002\u0002\u001b¡\u0003\u0002\u0002\u0002\u001d£\u0003\u0002\u0002"
    "\u0002\u001fç\u0003\u0002\u0002\u0002!ê\u0003\u0002\u0002\u0002#î\u0003\u0002\u0002\u0002%õ\u0003\u0002\u0002"
    "\u0002'Ŕ\u0003\u0002\u0002\u0002)ŗ\u0003\u0002\u0002\u0002+Ŷ\u0003\u0002\u0002\u0002-ſ\u0003\u0002\u0002\u0002"
    "/ƈ\u0003\u0002\u0002\u00021Ǜ\u0003\u0002\u0002\u00023Ǟ\u0003\u0002\u0002\u00025ǫ\u0003\u0002\u0002\u00027Ƕ\u0003"
    "\u0002\u0002\u00029ǻ\u0003\u0002\u0002\u0002;Ȁ\u0003\u0002\u0002\u0002=ț\u0003\u0002\u0002\u0002?ȝ\u0003\u0002"
    "\u0002\u0002AȠ\u0003\u0002\u0002\u0002Cȥ\u0003\u0002\u0002\u0002EF\t\u0002\u0002\u0002F\u0004\u0003\u0002\u0002"
    "\u0002GH\t\u0003\u0002\u0002H\u0006\u0003\u0002\u0002\u0002IJ\t\u0004\u0002\u0002J\b\u0003\u0002\u0002\u0002KL\t"
    "\u0005\u0002\u0002L\n\u0003\u0002\u0002\u0002MN\t\u0006\u0002\u0002N\f\u0003\u0002\u0002\u0002OP\t\u0007\u0002"
    "\u0002P\u000e\u0003\u0002\u0002\u0002QR\t\b\u0002\u0002R\u0010\u0003\u0002\u0002\u0002ST\t\t\u0002\u0002T\u0012"
    "\u0003\u0002\u0002\u0002UW\u0005\u000f\b\u0002VX\u0005\t\u0005\u0002WV\u0003\u0002\u0002\u0002XY\u0003\u0002"
    "\u0002\u0002YW\u0003\u0002\u0002\u0002YZ\u0003\u0002\u0002\u0002Z\u0014\u0003\u0002\u0002\u0002["
    "]\t\n\u0002\u0002\\[\u0003\u0002\u0002\u0002]^\u0003\u0002\u0002\u0002^\\\u0003\u0002\u0002\u0002^_\u0003\u0002"
    "\u0002\u0002_\u0016\u0003\u0002\u0002\u0002`a\t\u000b\u0002\u0002a\u0018\u0003\u0002\u0002\u0002bc\t\f\u0002"
    "\u0002cd\t\u0002\u0002\u0002de\t\r\u0002\u0002ef\t\u000e\u0002\u0002fj\t\u0002\u0002\u0002gh\t\r\u0002\u0002hi\t"
    "\u000e\u0002\u0002ik\t\u0002\u0002\u0002jg\u0003\u0002\u0002\u0002jk\u0003\u0002\u0002\u0002km\u0003\u0002\u0002"
    "\u0002ln\u0005\u0013\n\u0002ml\u0003\u0002\u0002\u0002mn\u0003\u0002\u0002\u0002n\u001a\u0003\u0002\u0002"
    "\u0002oq\t\u000f\u0002\u0002po\u0003\u0002\u0002\u0002pq\u0003\u0002\u0002\u0002qr\u0003\u0002\u0002\u0002rs\t"
    "\u0002\u0002\u0002su\u00070\u0002\u0002tv\t\u0010\u0002\u0002ut\u0003\u0002\u0002\u0002uv\u0003\u0002\u0002"
    "\u0002vw\u0003\u0002\u0002\u0002wx\t\u0002\u0002\u0002x\u0083\u00070\u0002\u0002yz\t\u0011\u0002\u0002z{"
    "\t\u0012\u0002\u0002{|\t\u0002\u0002\u0002|\u0084\t\u0002\u0002\u0002}~\t\u0013\u0002\u0002~\u007f\t\u0014\u0002"
    "\u0002\u007f\u0080\t\u0002\u0002\u0002\u0080\u0084\t\u0002\u0002\u0002\u0081\u0082\t\u0002\u0002\u0002\u0082"
    "\u0084\t\u0002\u0002\u0002\u0083y\u0003\u0002\u0002\u0002\u0083}\u0003\u0002\u0002\u0002\u0083\u0081\u0003\u0002"
    "\u0002\u0002\u0084\u0086\u0003\u0002\u0002\u0002\u0085\u0087\u0005\u0013\n\u0002\u0086\u0085\u0003\u0002\u0002"
    "\u0002\u0086\u0087\u0003\u0002\u0002\u0002\u0087¢\u0003\u0002\u0002\u0002\u0088\u008a\t\u000f\u0002\u0002\u0089"
    "\u0088\u0003\u0002\u0002\u0002\u0089\u008a\u0003\u0002\u0002\u0002\u008a\u008b\u0003\u0002\u0002\u0002\u008b"
    "\u008c\t\u0002\u0002\u0002\u008c\u008e\u00071\u0002\u0002\u008d\u008f\t\u0010\u0002\u0002\u008e\u008d\u0003"
    "\u0002\u0002\u0002\u008e\u008f\u0003\u0002\u0002\u0002\u008f\u0090\u0003\u0002\u0002\u0002\u0090\u0091\t\u0002"
    "\u0002\u0002\u0091\u009c\u00071\u0002\u0002\u0092\u0093\t\u0011\u0002\u0002\u0093\u0094\t\u0012\u0002\u0002"
    "\u0094\u0095\t\u0002\u0002\u0002\u0095\u009d\t\u0002\u0002\u0002\u0096\u0097\t\u0013\u0002\u0002\u0097\u0098\t"
    "\u0014\u0002\u0002\u0098\u0099\t\u0002\u0002\u0002\u0099\u009d\t\u0002\u0002\u0002\u009a\u009b\t\u0002\u0002"
    "\u0002\u009b\u009d\t\u0002\u0002\u0002\u009c\u0092\u0003\u0002\u0002\u0002\u009c\u0096\u0003\u0002\u0002\u0002"
    "\u009c\u009a\u0003\u0002\u0002\u0002\u009d\u009f\u0003\u0002\u0002\u0002\u009e \u0005\u0013\n\u0002\u009f\u009e"
    "\u0003\u0002\u0002\u0002\u009f \u0003\u0002\u0002\u0002 ¢\u0003\u0002\u0002\u0002¡p\u0003\u0002\u0002\u0002"
    "¡\u0089\u0003\u0002\u0002\u0002¢\u001c\u0003\u0002\u0002\u0002£¤\u0007'\u0002\u0002¤¥\u0005\u001f\u0010\u0002"
    "¥\u001e\u0003\u0002\u0002\u0002¦¨\t\u0015\u0002\u0002§¦\u0003\u0002\u0002\u0002§¨\u0003\u0002\u0002\u0002"
    "¨©\u0003\u0002\u0002\u0002©ª\u0005!\u0011\u0002ª«\t\u0016\u0002\u0002«\u00ad\u0005!\u0011\u0002¬®\u0005#\u0012"
    "\u0002\u00ad¬\u0003\u0002\u0002\u0002\u00ad®\u0003\u0002\u0002\u0002®°\u0003\u0002\u0002\u0002¯±\u0005\u0013\n"
    "\u0002°¯\u0003\u0002\u0002\u0002°±\u0003\u0002\u0002\u0002±è\u0003\u0002\u0002\u0002²´\t\u0015\u0002\u0002"
    "³²\u0003\u0002\u0002\u0002³´\u0003\u0002\u0002\u0002´µ\u0003\u0002\u0002\u0002µ¶\u0005!\u0011\u0002¶¸\u0005"
    "#\u0012\u0002·¹\u0005\u0013\n\u0002¸·\u0003\u0002\u0002\u0002¸¹\u0003\u0002\u0002\u0002¹è\u0003\u0002\u0002"
    "\u0002º¼\t\u0015\u0002\u0002»º\u0003\u0002\u0002\u0002»¼\u0003\u0002\u0002\u0002¼½\u0003\u0002\u0002\u0002"
    "½¿\u0005!\u0011\u0002¾À\u0005\u0013\n\u0002¿¾\u0003\u0002\u0002\u0002¿À\u0003\u0002\u0002\u0002Àè\u0003\u0002"
    "\u0002\u0002ÁÃ\t\u0015\u0002\u0002ÂÁ\u0003\u0002\u0002\u0002ÂÃ\u0003\u0002\u0002\u0002ÃÄ\u0003\u0002\u0002\u0002"
    "ÄÅ\u0005!\u0011\u0002ÅÆ\u00071\u0002\u0002ÆÈ\u0005!\u0011\u0002ÇÉ\u0005\u0013\n\u0002ÈÇ\u0003\u0002\u0002\u0002"
    "ÈÉ\u0003\u0002\u0002\u0002Éè\u0003\u0002\u0002\u0002ÊË\u0005!\u0011\u0002ËÌ\u00070\u0002\u0002ÌÎ\u0003\u0002"
    "\u0002\u0002ÍÊ\u0003\u0002\u0002\u0002ÎÏ\u0003\u0002\u0002\u0002ÏÍ\u0003\u0002\u0002\u0002ÏÐ\u0003\u0002\u0002"
    "\u0002ÐÑ\u0003\u0002\u0002\u0002ÑÓ\u0005!\u0011\u0002ÒÔ\u0005\u0013\n\u0002ÓÒ\u0003\u0002\u0002\u0002ÓÔ\u0003"
    "\u0002\u0002\u0002Ôè\u0003\u0002\u0002\u0002ÕÖ\u0005!\u0011\u0002Ö×\u0007.\u0002\u0002×Ù\u0003\u0002\u0002\u0002"
    "ØÕ\u0003\u0002\u0002\u0002ÙÚ\u0003\u0002\u0002\u0002ÚØ\u0003\u0002\u0002\u0002ÚÛ\u0003\u0002\u0002\u0002ÛÜ\u0003"
    "\u0002\u0002\u0002ÜÞ\u0005!\u0011\u0002Ýß\u0005\u0013\n\u0002ÞÝ\u0003\u0002\u0002\u0002Þß\u0003\u0002\u0002"
    "\u0002ßè\u0003\u0002\u0002\u0002àâ\u0005!\u0011\u0002áã\u00070\u0002\u0002âá\u0003\u0002\u0002\u0002âã\u0003"
    "\u0002\u0002\u0002ãå\u0003\u0002\u0002\u0002äæ\u0005\u0013\n\u0002åä\u0003\u0002\u0002\u0002åæ\u0003\u0002\u0002"
    "\u0002æè\u0003\u0002\u0002\u0002ç§\u0003\u0002\u0002\u0002ç³\u0003\u0002\u0002\u0002ç»\u0003\u0002\u0002\u0002"
    "çÂ\u0003\u0002\u0002\u0002çÍ\u0003\u0002\u0002\u0002çØ\u0003\u0002\u0002\u0002çà\u0003\u0002\u0002\u0002è "
    "\u0003\u0002\u0002\u0002éë\u0005\u0003\u0002\u0002êé\u0003\u0002\u0002\u0002ëì\u0003\u0002\u0002\u0002ìê\u0003"
    "\u0002\u0002\u0002ìí\u0003\u0002\u0002\u0002í\"\u0003\u0002\u0002\u0002îð\t\u0017\u0002\u0002ïñ\t\u0015\u0002"
    "\u0002ðï\u0003\u0002\u0002\u0002ðñ\u0003\u0002\u0002\u0002ñò\u0003\u0002\u0002\u0002òó\u0005!\u0011\u0002ó$"
    "\u0003\u0002\u0002\u0002ôö\t\u0018\u0002\u0002õô\u0003\u0002\u0002\u0002ö÷\u0003\u0002\u0002\u0002÷õ\u0003\u0002"
    "\u0002\u0002÷ø\u0003\u0002\u0002\u0002ø&\u0003\u0002\u0002\u0002ùú\u0007j\u0002\u0002úû\u0007v\u0002\u0002"
    "ûü\u0007v\u0002\u0002üý\u0007r\u0002\u0002ýþ\u0007<\u0002\u0002þÿ\u00071\u0002\u0002ÿĉ\u00071\u0002\u0002"
    "Āā\u0007j\u0002\u0002āĂ\u0007v\u0002\u0002Ăă\u0007v\u0002\u0002ăĄ\u0007r\u0002\u0002Ąą\u0007u\u0002\u0002"
    "ąĆ\u0007<\u0002\u0002Ćć\u00071\u0002\u0002ćĉ\u00071\u0002\u0002Ĉù\u0003\u0002\u0002\u0002ĈĀ\u0003\u0002\u0002"
    "\u0002ĉĊ\u0003\u0002\u0002\u0002ĊČ\u0005%\u0013\u0002ċč\u0005\u0013\n\u0002Čċ\u0003\u0002\u0002\u0002Čč\u0003"
    "\u0002\u0002\u0002čŕ\u0003\u0002\u0002\u0002Ďď\u0007j\u0002\u0002ďĐ\u0007v\u0002\u0002Đđ\u0007v\u0002\u0002"
    "đĒ\u0007r\u0002\u0002Ēē\u0007<\u0002\u0002ēĔ\u00071\u0002\u0002ĔĞ\u00071\u0002\u0002ĕĖ\u0007j\u0002\u0002"
    "Ėė\u0007v\u0002\u0002ėĘ\u0007v\u0002\u0002Ęę\u0007r\u0002\u0002ęĚ\u0007u\u0002\u0002Ěě\u0007<\u0002\u0002"
    "ěĜ\u00071\u0002\u0002ĜĞ\u00071\u0002\u0002ĝĎ\u0003\u0002\u0002\u0002ĝĕ\u0003\u0002\u0002\u0002ĝĞ\u0003\u0002"
    "\u0002\u0002Ğğ\u0003\u0002\u0002\u0002ğĠ\u0007y\u0002\u0002Ġġ\u0007y\u0002\u0002ġĢ\u0007y\u0002\u0002Ģģ\u00070"
    "\u0002\u0002ģĤ\u0003\u0002\u0002\u0002ĤĦ\u0005%\u0013\u0002ĥħ\u0005\u0013\n\u0002Ħĥ\u0003\u0002\u0002\u0002"
    "Ħħ\u0003\u0002\u0002\u0002ħŕ\u0003\u0002\u0002\u0002ĨĪ\t\u0019\u0002\u0002ĩĨ\u0003\u0002\u0002\u0002Īī\u0003"
    "\u0002\u0002\u0002īĩ\u0003\u0002\u0002\u0002īĬ\u0003\u0002\u0002\u0002Ĭņ\u0003\u0002\u0002\u0002ĭĮ\u00070\u0002"
    "\u0002Įį\u0007e\u0002\u0002įİ\u0007q\u0002\u0002İŇ\u0007o\u0002\u0002ıĲ\u00070\u0002\u0002Ĳĳ\u0007q\u0002\u0002"
    "ĳĴ\u0007t\u0002\u0002ĴŇ\u0007i\u0002\u0002ĵĶ\u00070\u0002\u0002Ķķ\u0007g\u0002\u0002ķĸ\u0007f\u0002\u0002"
    "ĸŇ\u0007w\u0002\u0002Ĺĺ\u00070\u0002\u0002ĺĻ\u0007i\u0002\u0002Ļļ\u0007q\u0002\u0002ļŇ\u0007x\u0002\u0002"
    "Ľľ\u00070\u0002\u0002ľĿ\u0007p\u0002\u0002Ŀŀ\u0007g\u0002\u0002ŀŇ\u0007v\u0002\u0002Łł\u00070\u0002\u0002"
    "łŃ\u0007k\u0002\u0002Ńń\u0007p\u0002\u0002ńŅ\u0007h\u0002\u0002ŅŇ\u0007q\u0002\u0002ņĭ\u0003\u0002\u0002\u0002"
    "ņı\u0003\u0002\u0002\u0002ņĵ\u0003\u0002\u0002\u0002ņĹ\u0003\u0002\u0002\u0002ņĽ\u0003\u0002\u0002\u0002ņŁ\u0003"
    "\u0002\u0002\u0002Ňŋ\u0003\u0002\u0002\u0002ňŉ\u00070\u0002\u0002ŉŊ\u0007v\u0002\u0002ŊŌ\u0007t\u0002\u0002"
    "ŋň\u0003\u0002\u0002\u0002ŋŌ\u0003\u0002\u0002\u0002Ōŏ\u0003\u0002\u0002\u0002ōŎ\u00071\u0002\u0002ŎŐ\u0005"
    "%\u0013\u0002ŏō\u0003\u0002\u0002\u0002ŏŐ\u0003\u0002\u0002\u0002ŐŒ\u0003\u0002\u0002\u0002őœ\u0005\u0013\n"
    "\u0002Œő\u0003\u0002\u0002\u0002Œœ\u0003\u0002\u0002\u0002œŕ\u0003\u0002\u0002\u0002ŔĈ\u0003\u0002\u0002\u0002"
    "Ŕĝ\u0003\u0002\u0002\u0002Ŕĩ\u0003\u0002\u0002\u0002ŕ("
    "\u0003\u0002\u0002\u0002ŖŘ\u0005\r\u0007\u0002ŗŖ\u0003\u0002\u0002\u0002Řř\u0003\u0002\u0002\u0002řŗ\u0003\u0002"
    "\u0002\u0002řŚ\u0003\u0002\u0002\u0002ŚŜ\u0003\u0002\u0002\u0002śŝ\u00070\u0002\u0002Ŝś\u0003\u0002\u0002\u0002"
    "Ŝŝ\u0003\u0002\u0002\u0002ŝş\u0003\u0002\u0002\u0002ŞŠ\u0005\r\u0007\u0002şŞ\u0003\u0002\u0002\u0002Šš\u0003"
    "\u0002\u0002\u0002šş\u0003\u0002\u0002\u0002šŢ\u0003\u0002\u0002\u0002Ţţ\u0003\u0002\u0002\u0002ţů\u0007B\u0002"
    "\u0002ŤŦ\u0005\r\u0007\u0002ťŤ\u0003\u0002\u0002\u0002Ŧŧ\u0003\u0002\u0002\u0002ŧť\u0003\u0002\u0002\u0002"
    "ŧŨ\u0003\u0002\u0002\u0002Ũũ\u0003\u0002\u0002\u0002ũū\u00070\u0002\u0002ŪŬ\u0005\r\u0007\u0002ūŪ\u0003\u0002"
    "\u0002\u0002Ŭŭ\u0003\u0002\u0002\u0002ŭū\u0003\u0002\u0002\u0002ŭŮ\u0003\u0002\u0002\u0002ŮŰ\u0003\u0002\u0002"
    "\u0002ůť\u0003\u0002\u0002\u0002Űű\u0003\u0002\u0002\u0002űů\u0003\u0002\u0002\u0002űŲ\u0003\u0002\u0002\u0002"
    "ŲŴ\u0003\u0002\u0002\u0002ųŵ\u0005\u0013\n\u0002Ŵų\u0003\u0002\u0002\u0002Ŵŵ\u0003\u0002\u0002\u0002ŵ*\u0003"
    "\u0002\u0002\u0002ŶŸ\u0007%\u0002\u0002ŷŹ\u0005\r\u0007\u0002Ÿŷ\u0003\u0002\u0002\u0002Źź\u0003\u0002\u0002"
    "\u0002źŸ\u0003\u0002\u0002\u0002źŻ\u0003\u0002\u0002\u0002ŻŽ\u0003\u0002\u0002\u0002żž\u0005\u0013\n\u0002"
    "Žż\u0003\u0002\u0002\u0002Žž\u0003\u0002\u0002\u0002ž,"
    "\u0003\u0002\u0002\u0002ſƁ\u0007B\u0002\u0002ƀƂ\u0005\r\u0007\u0002Ɓƀ\u0003\u0002\u0002\u0002Ƃƃ\u0003\u0002"
    "\u0002\u0002ƃƁ\u0003\u0002\u0002\u0002ƃƄ\u0003\u0002\u0002\u0002ƄƆ\u0003\u0002\u0002\u0002ƅƇ\u0005\u0013\n\u0002"
    "Ɔƅ\u0003\u0002\u0002\u0002ƆƇ\u0003\u0002\u0002\u0002Ƈ.\u0003\u0002\u0002\u0002ƈƊ\u0007>\u0002\u0002ƉƋ\u0005\r"
    "\u0007\u0002ƊƉ\u0003\u0002\u0002\u0002Ƌƌ\u0003\u0002\u0002\u0002ƌƊ\u0003\u0002\u0002\u0002ƌƍ\u0003\u0002\u0002"
    "\u0002ƍƎ\u0003\u0002\u0002\u0002ƎƏ\u0007@\u0002\u0002Ə0\u0003\u0002\u0002\u0002ƐƑ\u0007<\u0002\u0002Ƒǜ\u0007"
    "+\u0002\u0002ƒƓ\u0007<\u0002\u0002ƓƔ\u0007/\u0002\u0002Ɣǜ\u0007+\u0002\u0002ƕƖ\u0007<\u0002\u0002ƖƗ\u0007/\u0002"
    "\u0002Ɨǜ\u0007_\u0002\u0002Ƙƙ\u0007<\u0002\u0002ƙǜ\u0007F\u0002\u0002ƚƛ\u0007<\u0002\u0002ƛƜ\u0007/\u0002\u0002"
    "Ɯǜ\u0007F\u0002\u0002Ɲƞ\u0007:\u0002\u0002ƞƟ\u0007/\u0002\u0002Ɵǜ\u0007+\u0002\u0002Ơơ\u0007=\u0002\u0002"
    "ơǜ\u0007+\u0002\u0002Ƣƣ\u0007=\u0002\u0002ƣƤ\u0007–\u0002\u0002Ƥǜ\u0007+\u0002\u0002ƥƦ\u0007<\u0002\u0002"
    "Ʀǜ\u0007*\u0002\u0002Ƨƨ\u0007<\u0002\u0002ƨƩ\u0007/\u0002\u0002Ʃǜ\u0007*\u0002\u0002ƪƫ\u0007<\u0002\u0002"
    "ƫƬ\u0007)\u0002\u0002Ƭǜ\u0007*\u0002\u0002ƭƮ\u0007<\u0002\u0002ƮƯ\u0007)\u0002\u0002Ưǜ\u0007+\u0002\u0002"
    "ưƱ\u0007<\u0002\u0002Ʊǜ\u0007R\u0002\u0002ƲƳ\u0007<\u0002\u0002Ƴǜ\u0007r\u0002\u0002ƴƵ\u0007<\u0002\u0002"
    "Ƶǜ\u0007~\u0002\u0002ƶƷ\u0007?\u0002\u0002Ʒǜ\u0007~\u0002\u0002Ƹƹ\u0007?\u0002\u0002ƹǜ\u0007+\u0002\u0002"
    "ƺƻ\u0007?\u0002\u0002ƻǜ\u0007*\u0002\u0002Ƽƽ\u0007<\u0002\u0002ƽƾ\u0007–\u0002\u0002ƾǜ\u00071\u0002\u0002"
    "ƿǀ\u0007<\u0002\u0002ǀǜ\u00071\u0002\u0002ǁǂ\u0007<\u0002\u0002ǂǃ\u0007`\u0002\u0002ǃǜ\u0007+\u0002\u0002"
    "Ǆǅ\u0007±\u0002\u0002ǅǆ\u0007^\u0002\u0002ǆǇ\u0007a\u0002\u0002Ǉǈ\u0007*\u0002\u0002ǈǉ\u0007テ\u0002\u0002"
    "ǉǊ\u0007+\u0002\u0002Ǌǋ\u0007a\u0002\u0002ǋǌ\u00071\u0002\u0002ǌǜ\u0007±\u0002\u0002Ǎǎ\u0007Q\u0002\u0002"
    "ǎǏ\u0007a\u0002\u0002Ǐǜ\u0007q\u0002\u0002ǐǑ\u0007q\u0002\u0002Ǒǒ\u0007a\u0002\u0002ǒǜ\u0007Q\u0002\u0002"
    "Ǔǔ\u0007Q\u0002\u0002ǔǕ\u0007a\u0002\u0002Ǖǜ\u0007Q\u0002\u0002ǖǗ\u0007^\u0002\u0002Ǘǘ\u0007q\u0002\u0002"
    "ǘǜ\u00071\u0002\u0002Ǚǚ\u0007>\u0002\u0002ǚǜ\u00075\u0002\u0002ǛƐ\u0003\u0002\u0002\u0002Ǜƒ\u0003\u0002\u0002"
    "\u0002Ǜƕ\u0003\u0002\u0002\u0002ǛƘ\u0003\u0002\u0002\u0002Ǜƚ\u0003\u0002\u0002\u0002ǛƝ\u0003\u0002\u0002\u0002"
    "ǛƠ\u0003\u0002\u0002\u0002ǛƢ\u0003\u0002\u0002\u0002Ǜƥ\u0003\u0002\u0002\u0002ǛƧ\u0003\u0002\u0002\u0002Ǜƪ\u0003"
    "\u0002\u0002\u0002Ǜƭ\u0003\u0002\u0002\u0002Ǜư\u0003\u0002\u0002\u0002ǛƲ\u0003\u0002\u0002\u0002Ǜƴ\u0003\u0002"
    "\u0002\u0002Ǜƶ\u0003\u0002\u0002\u0002ǛƸ\u0003\u0002\u0002\u0002Ǜƺ\u0003\u0002\u0002\u0002ǛƼ\u0003\u0002\u0002"
    "\u0002Ǜƿ\u0003\u0002\u0002\u0002Ǜǁ\u0003\u0002\u0002\u0002ǛǄ\u0003\u0002\u0002\u0002ǛǍ\u0003\u0002\u0002\u0002"
    "Ǜǐ\u0003\u0002\u0002\u0002ǛǓ\u0003\u0002\u0002\u0002Ǜǖ\u0003\u0002\u0002\u0002ǛǙ\u0003\u0002\u0002\u0002ǜ2\u0003"
    "\u0002\u0002\u0002ǝǟ\t\u001a\u0002\u0002Ǟǝ\u0003\u0002\u0002\u0002ǟǠ\u0003\u0002\u0002\u0002ǠǞ\u0003\u0002\u0002"
    "\u0002Ǡǡ\u0003\u0002\u0002\u0002ǡǣ\u0003\u0002\u0002\u0002ǢǤ\u00070\u0002\u0002ǣǢ\u0003\u0002\u0002\u0002"
    "ǣǤ\u0003\u0002\u0002\u0002ǤǦ\u0003\u0002\u0002\u0002ǥǧ\u0005\u0013\n\u0002Ǧǥ\u0003\u0002\u0002\u0002Ǧǧ\u0003"
    "\u0002\u0002\u0002ǧ4\u0003\u0002\u0002\u0002Ǩǩ\u0005\u0007\u0004\u0002ǩǪ\u00070\u0002\u0002ǪǬ\u0003\u0002\u0002"
    "\u0002ǫǨ\u0003\u0002\u0002\u0002Ǭǭ\u0003\u0002\u0002\u0002ǭǫ\u0003\u0002\u0002\u0002ǭǮ\u0003\u0002\u0002\u0002"
    "Ǯǰ\u0003\u0002\u0002\u0002ǯǱ\u0005\u0007\u0004\u0002ǰǯ\u0003\u0002\u0002\u0002ǰǱ\u0003\u0002\u0002\u0002Ǳǳ\u0003"
    "\u0002\u0002\u0002ǲǴ\u0005\u0013\n\u0002ǳǲ\u0003\u0002\u0002\u0002ǳǴ\u0003\u0002\u0002\u0002Ǵ6\u0003\u0002\u0002"
    "\u0002ǵǷ\u0005\t\u0005\u0002Ƕǵ\u0003\u0002\u0002\u0002ǷǸ\u0003\u0002\u0002\u0002ǸǶ\u0003\u0002\u0002\u0002"
    "Ǹǹ\u0003\u0002\u0002\u0002ǹ8\u0003\u0002\u0002\u0002ǺǼ\u0005\u000b\u0006\u0002ǻǺ\u0003\u0002\u0002\u0002Ǽǽ\u0003"
    "\u0002\u0002\u0002ǽǻ\u0003\u0002\u0002\u0002ǽǾ\u0003\u0002\u0002\u0002Ǿ:\u0003\u0002\u0002\u0002ǿȁ\u0005\u000b"
    "\u0006\u0002Ȁǿ\u0003\u0002\u0002\u0002ȁȂ\u0003\u0002\u0002\u0002ȂȀ\u0003\u0002\u0002\u0002Ȃȃ\u0003\u0002\u0002"
    "\u0002ȃȅ\u0003\u0002\u0002\u0002ȄȆ\u0007/\u0002\u0002ȅȄ\u0003\u0002\u0002\u0002ȅȆ\u0003\u0002\u0002\u0002"
    "ȆȈ\u0003\u0002\u0002\u0002ȇȉ\u0005\u000b\u0006\u0002Ȉȇ\u0003\u0002\u0002\u0002ȉȊ\u0003\u0002\u0002\u0002ȊȈ\u0003"
    "\u0002\u0002\u0002Ȋȋ\u0003\u0002\u0002\u0002ȋȍ\u0003\u0002\u0002\u0002ȌȎ\u0005\u0013\n\u0002ȍȌ\u0003\u0002\u0002"
    "\u0002ȍȎ\u0003\u0002\u0002\u0002Ȏ<\u0003\u0002\u0002\u0002ȏȜ\u0005\u000f\b\u0002ȐȜ\u0005\u0011\t\u0002ȑȒ\u00070"
    "\u0002\u0002Ȓȓ\u00070\u0002\u0002ȓȜ\u00070\u0002\u0002Ȕȕ\u0007*\u0002\u0002ȕȖ\u0007#\u0002\u0002ȖȜ\u0007+\u0002"
    "\u0002ȗȘ\u0007*\u0002\u0002Șș\u0007A\u0002\u0002șȜ\u0007+\u0002\u0002ȚȜ\t\u001b\u0002\u0002țȏ\u0003\u0002\u0002"
    "\u0002țȐ\u0003\u0002\u0002\u0002țȑ\u0003\u0002\u0002\u0002țȔ\u0003\u0002\u0002\u0002țȗ\u0003\u0002\u0002\u0002"
    "țȚ\u0003\u0002\u0002\u0002Ȝ>\u0003\u0002\u0002\u0002ȝȞ\u0005=\u001f\u0002Ȟ@\u0003\u0002\u0002\u0002ȟȡ\n\u001c"
    "\u0002\u0002Ƞȟ\u0003\u0002\u0002\u0002ȡȢ\u0003\u0002\u0002\u0002ȢȠ\u0003\u0002\u0002\u0002Ȣȣ\u0003\u0002\u0002"
    "\u0002ȣB\u0003\u0002\u0002\u0002ȤȦ\u000b\u0002\u0002\u0002ȥȤ\u0003\u0002\u0002\u0002Ȧȧ\u0003\u0002\u0002\u0002"
    "ȧȨ\u0003\u0002\u0002\u0002ȧȥ\u0003\u0002\u0002\u0002ȨD\u0003\u0002\u0002\u0002I\u0002Y^jmpu\u0083\u0086\u0089"
    "\u008e\u009c\u009f¡§\u00ad°³¸»¿ÂÈÏÓÚÞâåçìð÷ĈČĝĦīņŋŏŒŔřŜšŧŭűŴźŽƃƆƌǛǠǣǦǭǰǳǸǽȂȅȊȍțȢȧ\u0002 "
)

TurkishLexer._decision_to_DFA = [DFA(TurkishLexer._ATN.getDecisionState(i), i) for i in
                                 range(len(TurkishLexer._ATN.decisionToState))]

