# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['qrand']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.19.4,<2.0.0', 'qiskit>=0.23.1,<0.24.0', 'randomgen>=1.19.3,<2.0.0']

setup_kwargs = {
    'name': 'qrand',
    'version': '0.2.0a0',
    'description': 'A quantum random number generator for arbitrary probability distributions',
    'long_description': '[![Unitary Fund](https://img.shields.io/badge/Supported_By-UNITARY_FUND-FFF000.svg?style=flat)](http://unitary.fund)\n[![YouTube](https://img.shields.io/badge/PR-qrand-FF0000.svg?style=flat&logo=YouTube&logoColor=white)](https://youtu.be/CG7BxuWFpME)\n[![PyPI](https://img.shields.io/pypi/v/qrand?label=PyPI&style=flat&color=3776AB&logo=Python&logoColor=white)](https://pypi.org/project/qrand/)\n[![Coverage](https://img.shields.io/badge/Coverage-83%25-green.svg?style=flat)](http://pytest.org)\n[![Apache-2.0 License](https://img.shields.io/github/license/pedrorrivero/qrand?label=License&style=flat&color=1D1D1D)](https://github.com/pedrorrivero/qrand/blob/master/LICENSE)\n\n\n# qrand\n\n> A quantum random number generator for arbitrary probability distributions\n\nRandom numbers are everywhere.\n\nComputer algorithms, data encryption, physical simulations, and even the arts use them all the time. There is one problem though: it turns out that they are actually very difficult to produce in large amounts. Classical computers can only implement mathematical tricks to emulate randomness, while measuring it out of physical processes turns out to be too slow.\n\nLuckily, the probabilistic nature of quantum computers makes these devices particularly useful for the task. Nonetheless, most of the current efforts in producing quantum random numbers have been focused on uniform probability distributions. Despite this fact, many applications actually need to sample from more complex distributions (e.g. gaussian, poisson).\n\nThis software introduces an interface layer between [NumPy](https://numpy.org/) and [Qiskit](https://qiskit.org/), along with some useful functionality that enables the production of quantum random numbers (QRN) for a wide variety of probability distributions. This is ultimately accomplished by transforming uniform probability distributions produced in IBMQ\'s quantum devices, through NumPy\'s random module.\n\n```python3\nfrom qrand import QiskitBitGenerator\nfrom numpy.random import Generator\nfrom qiskit import IBMQ\n\nprovider = IBMQ.load_account()\nbitgen = QiskitBitGenerator(provider)\ngen = Generator(bitgen)\n\nprint(f"Random Raw: {bitgen.random_raw()}")\nprint(f"Random Bitstring: {bitgen.random_bitstring()}")\nprint(f"Random Unsigned Int: {bitgen.random_uint()}")\nprint(f"Random Double: {bitgen.random_double()}")\n\nprint(f"Random Binomial: {gen.binomial(4, 1/4)}")\nprint(f"Random Exponential: {gen.exponential()}")\nprint(f"Random Logistic: {gen.logistic()}")\nprint(f"Random Poisson: {gen.poisson()}")\nprint(f"Random Std. Normal: {gen.standard_normal()}")\nprint(f"Random Triangular: {gen.triangular(-1, 0, 1)}")\n# ...\n```\n\n\n# Documentation\n\n## QiskitBitGenerator\nA quantum random bit-generator based on Qiskit, which can interface with NumPy\'s random module (e.g. to instantiate Generator objects). It implements an efficient strategy to retrieve random bits from IBMQ\'s quantum backends.\n\nOn each request to a backend, it retrieves as many bits as possible and stores them in a cache. This way, the number of internet connections leading to overheads is greatly reduced and, while the cache is loaded, random bits can be retrieved "instantaneously". The user can limit the number of bits to retrieve on each request through the `max_bits_per_request` argument.\n\nAdditionally, it always chooses the least busy backend from the list of available machines. This list can be filtered by the user through the `backend_filter` argument, which defaults to history-enabled non-simulators. If a Qiskit Backend is explicitly passed in as argument, no backend selection will be performed: effectively ignoring any Qiskit Provider object passed. If neither `provider` nor `backend` are passed as inputs, it will default to running Qiskit BasicAer\'s \'qasm_simulator\' locally.\n\n### CLASS ARGUMENTS\n- **provider**: *Optional[Provider] = None* <br/>\n  A Qiskit Provider object to access quantum backends. If `None` it defaults to BasicAer.\n- **backend**: *Optional[Backend] = None* <br/>\n  A Qiskit Backend object to produce random bits. If not `None`, `provider` will be ignored.\n- **backend_filter**: *Optional[BackendFilter] = None* <br/>\n  A Callable that takes in a Qiskit Backend object and returns `True` if it meets certain requirements, `False` otherwise. This is used to filter the list of available backends from which to dynamically choose on each request to the `provider` (if no `backend` is explicitly input). If `None` it defaults to `QiskitBitGenerator.default_backend_filter`.\n- **max_bits_per_request**: *int = 0* <br/>\n  A limit to the number of bits to be retrieved on each request to any Qiskit Backend. If less than one, no bound will be applied and the maximum allowed number of bits will be retrieved.\n- **ISRAW32**: *Final[bool] = False* <br/>\n  Toggle 32-bit BitGenerator mode. If `False` the BitGenerator will be 64-bit. This determines the number of bits returned by NumPy\'s `random_raw()` method, and the default number of bits to output on `random_uint()` and `random_double()`. Once an object is instantiated, this cannot be overridden.\n\n### STATIC / CLASS METHODS\n- **default_backend_filter** *(b: Backend) -> bool*: <br/>\n  Default backend filter Callable.\n  - ARGUMENTS\n    - *b*: Backend <br/>\n      A Qiskit Backend object.\n  - RETURNS\n    - *out*: bool <br/>\n      `True` if `b` is not a simulator and has memory enabled, `False` otherwise.\n- **get_best_backend** *(cls, provider: Provider, backend_filter: Optional[BackendFilter] = None) -> Backend*: <br/>\n  Returns the least busy backend available to an input provider, and according to certain filter(s).\n  - ARGUMENTS\n    - *provider*: Provider <br/>\n      A Qiskit Provider object to access quantum backends.\n    - *backend_filter*: Optional[BackendFilter] = None <br/>\n      A Callable that takes in a Qiskit Backend object and returns `True` if it meets certain requirements, `False` otherwise. If `None` it defaults to `cls.default_backend_filter`.\n  - RETURNS\n    - *out*: Backend <br/>\n      Least busy backend from the filtered list of available backends.\n\n### PUBLIC METHODS\n- **dump_cache** *(self, flush: bool = False) -> str*: <br/>\n  Returns all the contents stored in the cache.\n  - ARGUMENTS\n    - *flush*: bool = False <br/>\n      If `True` erase the cache after dumping.\n  - RETURNS\n    - *out*: str <br/>\n      The bitstring stored in cache.\n- **flush_cache** *(self) -> bool*: <br/>\n  Erase the cache.\n  - RETURNS\n    - *out*: bool <br/>\n      `True` if succeeds, `False` otherwise.\n- **load_cache** *(self, bitstring: str, flush: bool = False) -> bool*: <br/>\n  Load cache contents from bitstring.\n  - ARGUMENTS\n    - *bitstring*: str <br/>\n      The bitstring to load to cache.\n    - *flush*: bool = False <br/>\n      If `True` erase cache before loading.\n  - RETURNS\n    - *out*: bool <br/>\n      `True` if succeeds, `False` otherwise.\n  - RAISES\n    - *TypeError* <br/>\n      If input bitstring is not str\n    - *ValueError* <br/>\n      If input bitstring is not a valid bitstring\n- **random_bitstring** *(self, n_bits: int = 0) -> str*: <br/>\n  Returns a random bitstring of a given lenght.\n  - ARGUMENTS\n    - *n_bits*: int = 0 <br/>\n      Number of bits to retrieve. If less than one it defaults to the raw number of bits for the instance QiskitBitGenerator (i.e. 32 or 64).\n  - RETURNS\n    - *out*: str <br/>\n      Bitstring of lenght `n_bits`.\n- **random_double** *(self, n: float = 1) -> float*: <br/>\n  Returns a random double from a uniform distribution in the range [0,n). Defaults to [0,1).\n  - ARGUMENTS\n    - *n*: float = 1 <br/>\n      Size of the range [0,n) from which to draw the random number.\n  - RETURNS\n    - *out*: float <br/>\n      Random float in the range [0,n).\n- **random_raw** *(self) -> int*: <br/>\n  Returns a random unsigned int of either 32 or 64 bits.\n  - RETURNS\n    - *out*: int <br/>\n      Unsigned int of either 32 or 64 bits.\n- **random_uint** *(self, n_bits: int = 0) -> int*: <br/>\n  Returns a random unsigned int of a given size in bits.\n  - ARGUMENTS\n    - *n_bits*: int = 0 <br/>\n      Number of bits to retrieve. If less than one it defaults to the raw number of bits for the instance QiskitBitGenerator (i.e. 32 or 64).\n  - RETURNS\n    - *out*: int <br/>\n      Unsigned int of `n_bits` bits.\n- **set_state** *(self, provider: Optional[Provider] = None, backend: Optional[Backend] = None, backend_filter: Optional[BackendFilter] = None, max_bits_per_request: Optional[int] = None) -> bool*: <br/>\n  Override constructor arguments. Any change must be explicitly passed as input (i.e. not `None`).\n  - ARGUMENTS\n    - *provider*: Optional[Provider] = None <br/>\n      Same as constructor.\n    - *backend*: Optional[Backend] = None <br/>\n      Same as constructor.\n    - *backend_filter*: Optional[BackendFilter] = None <br/>\n      Same as constructor.\n    - *max_bits_per_request*: Optional[int] = None <br/>\n      Same as constructor.\n  - RETURNS\n    - *out*: bool <br/>\n      `True` if any changes were made, `False` otherwise.\n\n### PUBLIC PROPERTIES\n- **BITS**: *Final[int]* <br/>\n  Either 32 or 64. The number of bits output by NumPy\'s `random_raw()` method. Final, it cannot be modified after instantiation through the ISRAW32 argument.\n- **state**: *dict* <br/>\n  Parsed information about the current state of the QiskitBitGenerator.\n\n\n# Acknowledgements\n\nParts of this software\'s source code have been borrowed from the [qRNG](https://github.com/ozanerhansha/qRNG) project, which is licensed under the [GNU GPLv3](https://github.com/ozanerhansha/qRNG/blob/master/LICENSE) license. Copyright notice and specific changes can be found as a docstring wherever this applies.\n\n---\n(c) Copyright 2021 Pedro Rivero\n',
    'author': 'Pedro Rivero',
    'author_email': 'pedro.rivero.ramirez@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pedrorrivero/qrand',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
