
#ifndef TICK_OPTIM_MODEL_SRC_VARIANTS_HAWKES_LEASTSQ_LIST_H_
#define TICK_OPTIM_MODEL_SRC_VARIANTS_HAWKES_LEASTSQ_LIST_H_

// License: BSD 3 clause

#include "base.h"
#include "../base/hawkes_list.h"
#include "../base/hawkes_single.h"

/** \class ModelHawkesLeastSqList
 * \brief Base class of Hawkes models handling several realizations
 */
class ModelHawkesLeastSqList : public ModelHawkesList {
 protected:
  //! @brief Flag telling if precompations arrays have been allocated or not
  bool weights_allocated;

  //! @bbrief aggregated model used to compute loss, gradient and hessian
  std::unique_ptr<ModelHawkesSingle> aggregated_model;

 public:
  //! @brief Constructor
  //! \param max_n_threads : number of cores to be used for multithreading. If negative,
  //! the number of physical cores will be used
  //! \param optimization_level : 0 corresponds to no optimization and 1 to use of faster (approximated) exponential function
  ModelHawkesLeastSqList(const int max_n_threads = 1,
                  const unsigned int optimization_level = 0);

  void incremental_set_data(const SArrayDoublePtrList1D &timestamps, double end_time);

  /**
   * @brief Precomputations of intermediate values
   * They will be used to compute faster loss and gradient
   */
  void compute_weights();

  /**
   * @brief Compute loss
   * \param coeffs : Point in which loss is computed
   * \return Loss' value
   */
  double loss(const ArrayDouble &coeffs) override;

  /**
   * @brief Compute loss corresponding to sample i (between 0 and rand_max = dim)
   * \param i : selected dimension
   * \param coeffs : Point in which loss is computed
   * \return Loss' value
   */
  double loss_i(const ulong i, const ArrayDouble &coeffs) override;

  /**
   * @brief Compute gradient
   * \param coeffs : Point in which gradient is computed
   * \param out : Array in which the value of the gradient is stored
   */
  void grad(const ArrayDouble &coeffs, ArrayDouble &out) override;

  /**
   * @brief Compute gradient corresponding to sample i (between 0 and rand_max = dim)
   * \param i : selected dimension
   * \param coeffs : Point in which gradient is computed
   * \param out : Array in which the value of the gradient is stored
   */
  void grad_i(const ulong i, const ArrayDouble &coeffs, ArrayDouble &out) override;

 protected:
  //! @brief allocate arrays to store precomputations
  virtual void allocate_weights() {}

  //! @brief synchronize aggregate_model with this instance
  virtual void synchronize_aggregated_model() {}

  virtual void compute_weights_timestamps_list() {}
  virtual void compute_weights_timestamps(const SArrayDoublePtrList1D &timestamps,
                                          double end_time) {}
};

#endif  // TICK_OPTIM_MODEL_SRC_VARIANTS_HAWKES_LEASTSQ_LIST_H_
