"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAccount = void 0;
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * A service account provides an identity for processes that run in a Pod.
 *
 * When you (a human) access the cluster (for example, using kubectl), you are
 * authenticated by the apiserver as a particular User Account (currently this
 * is usually admin, unless your cluster administrator has customized your
 * cluster). Processes in containers inside pods can also contact the apiserver.
 * When they do, they are authenticated as a particular Service Account (for
 * example, default).
 *
 * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account
 */
class ServiceAccount extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, props);
        this._secrets = (_a = props.secrets) !== null && _a !== void 0 ? _a : [];
        this.apiObject = new k8s.ServiceAccount(this, 'Resource', {
            metadata: props.metadata,
            secrets: utils_1.lazy(() => utils_1.undefinedIfEmpty(this._secrets.map(s => ({ name: s.name })))),
        });
    }
    /**
     * Imports a service account from the cluster as a reference.
     * @param name The name of the service account resource.
     */
    static fromServiceAccountName(name) {
        return { name: name };
    }
    /**
     * Allow a secret to be accessed by pods using this service account.
     * @param secret The secret
     */
    addSecret(secret) {
        this._secrets.push(secret);
    }
    /**
     * List of secrets allowed to be used by pods running using this service
     * account.
     *
     * Returns a copy. To add a secret, use `addSecret()`.
     */
    get secrets() {
        return [...this._secrets];
    }
}
exports.ServiceAccount = ServiceAccount;
//# sourceMappingURL=data:application/json;base64,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