"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobSpecDefinition = exports.Job = void 0;
const base_1 = require("./base");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
const utils_1 = require("./utils");
/**
 * A Job creates one or more Pods and ensures that a specified number of them successfully terminate. As pods successfully complete,
 * the Job tracks the successful completions. When a specified number of successful completions is reached, the task (ie, Job) is complete.
 * Deleting a Job will clean up the Pods it created. A simple case is to create one Job object in order to reliably run one Pod to completion.
 * The Job object will start a new Pod if the first Pod fails or is deleted (for example due to a node hardware failure or a node reboot).
 * You can also use a Job to run multiple Pods in parallel.
 */
class Job extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.spec = new JobSpecDefinition(props.spec);
        this.apiObject = new k8s.Job(this, 'Default', {
            metadata: props.metadata,
            spec: utils_1.lazy(() => this.spec._toKube()),
        });
    }
}
exports.Job = Job;
class JobSpecDefinition {
    constructor(props = {}) {
        var _a, _b;
        this.podSpecTemplate = new pod_1.PodSpecDefinition({
            restartPolicy: (_b = (_a = props.podSpecTemplate) === null || _a === void 0 ? void 0 : _a.restartPolicy) !== null && _b !== void 0 ? _b : pod_1.RestartPolicy.NEVER,
            ...props.podSpecTemplate,
        });
        this.podMetadataTemplate = new cdk8s_1.ApiObjectMetadataDefinition(props.podMetadataTemplate);
        this.ttlAfterFinished = props.ttlAfterFinished;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            template: {
                metadata: this.podMetadataTemplate.toJson(),
                spec: this.podSpecTemplate._toKube(),
            },
            ttlSecondsAfterFinished: this.ttlAfterFinished ? this.ttlAfterFinished.toSeconds() : undefined,
        };
    }
}
exports.JobSpecDefinition = JobSpecDefinition;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiam9iLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2pvYi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxpQ0FBaUQ7QUFDakQsaUNBQWtGO0FBR2xGLHFDQUFxQztBQUNyQywrQkFBa0U7QUFFbEUsbUNBQStCO0FBZ0IvQjs7Ozs7O0dBTUc7QUFDSCxNQUFhLEdBQUksU0FBUSxlQUFRO0lBSy9CLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBa0IsRUFBRTtRQUM1RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksaUJBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTlDLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDNUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLElBQUksRUFBRSxZQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztTQUN0QyxDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUFmRCxrQkFlQztBQThCRCxNQUFhLGlCQUFpQjtJQWdCNUIsWUFBWSxRQUFpQixFQUFFOztRQUM3QixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksdUJBQWlCLENBQUM7WUFDM0MsYUFBYSxjQUFFLEtBQUssQ0FBQyxlQUFlLDBDQUFFLGFBQWEsbUNBQUksbUJBQWEsQ0FBQyxLQUFLO1lBQzFFLEdBQUcsS0FBSyxDQUFDLGVBQWU7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLG1CQUFtQixHQUFHLElBQUksbUNBQTJCLENBQUMsS0FBSyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDdEYsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osT0FBTztZQUNMLFFBQVEsRUFBRTtnQkFDUixRQUFRLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sRUFBRTtnQkFDM0MsSUFBSSxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxFQUFFO2FBQ3JDO1lBQ0QsdUJBQXVCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDL0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQXJDRCw4Q0FxQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBSZXNvdXJjZSwgUmVzb3VyY2VQcm9wcyB9IGZyb20gJy4vYmFzZSc7XG5pbXBvcnQgeyBBcGlPYmplY3QsIEFwaU9iamVjdE1ldGFkYXRhLCBBcGlPYmplY3RNZXRhZGF0YURlZmluaXRpb24gfSBmcm9tICdjZGs4cyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuaW1wb3J0ICogYXMgazhzIGZyb20gJy4vaW1wb3J0cy9rOHMnO1xuaW1wb3J0IHsgUmVzdGFydFBvbGljeSwgUG9kU3BlYywgUG9kU3BlY0RlZmluaXRpb24gfSBmcm9tICcuL3BvZCc7XG5pbXBvcnQgeyBEdXJhdGlvbiB9IGZyb20gJy4vZHVyYXRpb24nO1xuaW1wb3J0IHsgbGF6eSB9IGZyb20gJy4vdXRpbHMnO1xuXG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgaW5pdGlhbGl6YXRpb24gb2YgYEpvYmAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iUHJvcHMgZXh0ZW5kcyBSZXNvdXJjZVByb3BzIHtcblxuICAvKipcbiAgICogVGhlIHNwZWMgb2YgdGhlIGpvYi4gVXNlIGBqb2Iuc3BlY2AgdG8gYXBwbHkgcG9zdCBpbnN0YW50aWF0aW9uIG11dGF0aW9ucy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBbiBlbXB0eSBzcGVjIHdpbGwgYmUgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHNwZWM/OiBKb2JTcGVjO1xufVxuXG4vKipcbiAqIEEgSm9iIGNyZWF0ZXMgb25lIG9yIG1vcmUgUG9kcyBhbmQgZW5zdXJlcyB0aGF0IGEgc3BlY2lmaWVkIG51bWJlciBvZiB0aGVtIHN1Y2Nlc3NmdWxseSB0ZXJtaW5hdGUuIEFzIHBvZHMgc3VjY2Vzc2Z1bGx5IGNvbXBsZXRlLFxuICogdGhlIEpvYiB0cmFja3MgdGhlIHN1Y2Nlc3NmdWwgY29tcGxldGlvbnMuIFdoZW4gYSBzcGVjaWZpZWQgbnVtYmVyIG9mIHN1Y2Nlc3NmdWwgY29tcGxldGlvbnMgaXMgcmVhY2hlZCwgdGhlIHRhc2sgKGllLCBKb2IpIGlzIGNvbXBsZXRlLlxuICogRGVsZXRpbmcgYSBKb2Igd2lsbCBjbGVhbiB1cCB0aGUgUG9kcyBpdCBjcmVhdGVkLiBBIHNpbXBsZSBjYXNlIGlzIHRvIGNyZWF0ZSBvbmUgSm9iIG9iamVjdCBpbiBvcmRlciB0byByZWxpYWJseSBydW4gb25lIFBvZCB0byBjb21wbGV0aW9uLlxuICogVGhlIEpvYiBvYmplY3Qgd2lsbCBzdGFydCBhIG5ldyBQb2QgaWYgdGhlIGZpcnN0IFBvZCBmYWlscyBvciBpcyBkZWxldGVkIChmb3IgZXhhbXBsZSBkdWUgdG8gYSBub2RlIGhhcmR3YXJlIGZhaWx1cmUgb3IgYSBub2RlIHJlYm9vdCkuXG4gKiBZb3UgY2FuIGFsc28gdXNlIGEgSm9iIHRvIHJ1biBtdWx0aXBsZSBQb2RzIGluIHBhcmFsbGVsLlxuICovXG5leHBvcnQgY2xhc3MgSm9iIGV4dGVuZHMgUmVzb3VyY2Uge1xuXG4gIHByb3RlY3RlZCByZWFkb25seSBhcGlPYmplY3Q6IEFwaU9iamVjdDtcbiAgcHVibGljIHJlYWRvbmx5IHNwZWM6IEpvYlNwZWNEZWZpbml0aW9uO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBKb2JQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLnNwZWMgPSBuZXcgSm9iU3BlY0RlZmluaXRpb24ocHJvcHMuc3BlYyk7XG5cbiAgICB0aGlzLmFwaU9iamVjdCA9IG5ldyBrOHMuSm9iKHRoaXMsICdEZWZhdWx0Jywge1xuICAgICAgbWV0YWRhdGE6IHByb3BzLm1ldGFkYXRhLFxuICAgICAgc3BlYzogbGF6eSgoKSA9PiB0aGlzLnNwZWMuX3RvS3ViZSgpKSxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGluaXRpYWxpemF0aW9uIG9mIGBKb2JTcGVjYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKb2JTcGVjIHtcbiAgLyoqXG4gICAqIFRoZSBzcGVjIG9mIHBvZHMgY3JlYXRlZCBieSB0aGlzIGpvYi5cbiAgICovXG4gIHJlYWRvbmx5IHBvZFNwZWNUZW1wbGF0ZT86IFBvZFNwZWM7XG5cbiAgLyoqXG4gICAqIFRoZSBtZXRhZGF0YSBvZiBwb2RzIGNyZWF0ZWQgYnkgdGhpcyBqb2IuXG4gICAqL1xuICByZWFkb25seSBwb2RNZXRhZGF0YVRlbXBsYXRlPzogQXBpT2JqZWN0TWV0YWRhdGE7XG5cbiAgLyoqXG4gICAqIExpbWl0cyB0aGUgbGlmZXRpbWUgb2YgYSBKb2IgdGhhdCBoYXMgZmluaXNoZWQgZXhlY3V0aW9uIChlaXRoZXIgQ29tcGxldGVcbiAgICogb3IgRmFpbGVkKS4gSWYgdGhpcyBmaWVsZCBpcyBzZXQsIGFmdGVyIHRoZSBKb2IgZmluaXNoZXMsIGl0IGlzIGVsaWdpYmxlIHRvXG4gICAqIGJlIGF1dG9tYXRpY2FsbHkgZGVsZXRlZC4gV2hlbiB0aGUgSm9iIGlzIGJlaW5nIGRlbGV0ZWQsIGl0cyBsaWZlY3ljbGVcbiAgICogZ3VhcmFudGVlcyAoZS5nLiBmaW5hbGl6ZXJzKSB3aWxsIGJlIGhvbm9yZWQuIElmIHRoaXMgZmllbGQgaXMgc2V0IHRvIHplcm8sXG4gICAqIHRoZSBKb2IgYmVjb21lcyBlbGlnaWJsZSB0byBiZSBkZWxldGVkIGltbWVkaWF0ZWx5IGFmdGVyIGl0IGZpbmlzaGVzLiBUaGlzXG4gICAqIGZpZWxkIGlzIGFscGhhLWxldmVsIGFuZCBpcyBvbmx5IGhvbm9yZWQgYnkgc2VydmVycyB0aGF0IGVuYWJsZSB0aGVcbiAgICogYFRUTEFmdGVyRmluaXNoZWRgIGZlYXR1cmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSWYgdGhpcyBmaWVsZCBpcyB1bnNldCwgdGhlIEpvYiB3b24ndCBiZSBhdXRvbWF0aWNhbGx5IGRlbGV0ZWQuXG4gICAqL1xuICByZWFkb25seSB0dGxBZnRlckZpbmlzaGVkPzogRHVyYXRpb247XG59XG5cbmV4cG9ydCBjbGFzcyBKb2JTcGVjRGVmaW5pdGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgc3BlYyBmb3IgcG9kcyBjcmVhdGVkIGJ5IHRoaXMgam9iLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBvZFNwZWNUZW1wbGF0ZTogUG9kU3BlY0RlZmluaXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBtZXRhZGF0YSBmb3IgcG9kcyBjcmVhdGVkIGJ5IHRoaXMgam9iLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBvZE1ldGFkYXRhVGVtcGxhdGU6IEFwaU9iamVjdE1ldGFkYXRhRGVmaW5pdGlvbjtcblxuICAvKipcbiAgICogVFRMIGJlZm9yZSB0aGUgam9iIGlzIGRlbGV0ZWQgYWZ0ZXIgaXQgaXMgZmluaXNoZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdHRsQWZ0ZXJGaW5pc2hlZD86IER1cmF0aW9uO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBKb2JTcGVjID0ge30pIHtcbiAgICB0aGlzLnBvZFNwZWNUZW1wbGF0ZSA9IG5ldyBQb2RTcGVjRGVmaW5pdGlvbih7XG4gICAgICByZXN0YXJ0UG9saWN5OiBwcm9wcy5wb2RTcGVjVGVtcGxhdGU/LnJlc3RhcnRQb2xpY3kgPz8gUmVzdGFydFBvbGljeS5ORVZFUixcbiAgICAgIC4uLnByb3BzLnBvZFNwZWNUZW1wbGF0ZSxcbiAgICB9KTtcbiAgICB0aGlzLnBvZE1ldGFkYXRhVGVtcGxhdGUgPSBuZXcgQXBpT2JqZWN0TWV0YWRhdGFEZWZpbml0aW9uKHByb3BzLnBvZE1ldGFkYXRhVGVtcGxhdGUpO1xuICAgIHRoaXMudHRsQWZ0ZXJGaW5pc2hlZCA9IHByb3BzLnR0bEFmdGVyRmluaXNoZWQ7XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiBrOHMuSm9iU3BlYyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHRlbXBsYXRlOiB7XG4gICAgICAgIG1ldGFkYXRhOiB0aGlzLnBvZE1ldGFkYXRhVGVtcGxhdGUudG9Kc29uKCksXG4gICAgICAgIHNwZWM6IHRoaXMucG9kU3BlY1RlbXBsYXRlLl90b0t1YmUoKSxcbiAgICAgIH0sXG4gICAgICB0dGxTZWNvbmRzQWZ0ZXJGaW5pc2hlZDogdGhpcy50dGxBZnRlckZpbmlzaGVkID8gdGhpcy50dGxBZnRlckZpbmlzaGVkLnRvU2Vjb25kcygpIDogdW5kZWZpbmVkLFxuICAgIH07XG4gIH1cbn0iXX0=