"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const fs = require("fs");
const path = require("path");
const minimatch = require("minimatch");
const utils_1 = require("./utils");
/**
 * ConfigMap holds configuration data for pods to consume.
 */
class ConfigMap extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, props);
        this._binaryData = {};
        this._data = {};
        this.apiObject = new k8s.ConfigMap(this, 'ConfigMap', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: utils_1.lazy(() => this.synthesizeData()),
            binaryData: utils_1.lazy(() => this.synthesizeBinaryData()),
        });
        for (const [k, v] of Object.entries((_a = props.data) !== null && _a !== void 0 ? _a : {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries((_b = props.binaryData) !== null && _b !== void 0 ? _b : {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     * @param name The name of the config map to import
     */
    static fromConfigMapName(name) {
        return { name };
    }
    /**
     * Adds a data entry to the config map.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map. BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     * @param localFile The path to the local file
     * @param key The ConfigMap key (default to the file name).
     */
    addFile(localFile, key) {
        key = key !== null && key !== void 0 ? key : path.basename(localFile);
        const value = fs.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     * @param localDir A path to a local directory
     * @param options Options
     */
    addDirectory(localDir, options = {}) {
        var _a, _b;
        const exclude = (_a = options.exclude) !== null && _a !== void 0 ? _a : [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                if (minimatch(file, pattern)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = (_b = options.keyPrefix) !== null && _b !== void 0 ? _b : '';
        for (const file of fs.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (fs.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
//# sourceMappingURL=data:application/json;base64,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