# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pytest_drf',
 'pytest_drf.util',
 'tests',
 'tests.pytest_drf',
 'tests.testapp',
 'tests.testapp.views']

package_data = \
{'': ['*']}

modules = \
['pytest', 'LICENSE', 'CHANGELOG']
install_requires = \
['djangorestframework>3',
 'inflection>=0.3.1,<0.4.0',
 'pytest-assert-utils>=0,<1',
 'pytest-common-subject>=1.0,<2.0',
 'pytest-lambda>=1.1,<2.0',
 'pytest>=3.6']

entry_points = \
{'pytest11': ['drf = pytest_drf.plugin']}

setup_kwargs = {
    'name': 'pytest-drf',
    'version': '1.1.0',
    'description': 'A Django REST framework plugin for pytest.',
    'long_description': '# pytest-drf\n\n[![PyPI version](https://badge.fury.io/py/pytest-drf.svg)](https://badge.fury.io/py/pytest-drf)\n[![Build Status](https://travis-ci.org/theY4Kman/pytest-drf.svg?branch=master)](https://travis-ci.org/theY4Kman/pytest-drf)\n\npytest-drf is a [pytest](http://pytest.org) plugin for testing your [Django REST Framework](https://www.django-rest-framework.org/) APIs.\n\n\n# Installation\n\n```bash\npip install pytest-drf\n```\n\n\n# The Spiel\n\npytest-drf aims to shoo away clunky setup code and boilerplate in DRF tests, in favor of declarative scaffolds and configurable fixtures encouraging small, easy-to-follow tests with single responsibilities.\n\nThis is accomplished by performing one request per test, and providing the response as a fixture. All configuration of the request — the URL, the query params, the HTTP method, the POST data, etc — is also done through fixtures. This frees the test methods to contain only assertions about the response or the state of the app after the request completes.\n\nFor example, consider a public API endpoint that responds to a GET request with the JSON string "Hello, World!" and a 200 status code. Such an endpoint might be written like so\n\n```python\n# example/views.py\n\nfrom rest_framework import permissions\nfrom rest_framework.decorators import api_view, permission_classes\nfrom rest_framework.response import Response\n\n@api_view()\n@permission_classes([permissions.AllowAny])\ndef hello_world(request):\n    return Response(\'Hello, World!\')\n```\n\nLet\'s route it to `/hello`, and give it a name, so we can easily generate URLs for it.\n\n```python\n# example/urls.py\n\nfrom django.urls import path\n\nfrom example import views\n\nurlpatterns = [\n    path(\'hello\', views.hello_world, name=\'hello-world\'),\n]\n```\n\nWith pytest-drf, we\'d verify the behavior of our endpoint with something like this\n\n```python\n# tests/test_hello.py\n\nimport pytest\nfrom django.urls import reverse\nfrom pytest_drf import APIViewTest, Returns200, UsesGetMethod\n\nclass TestHelloWorld(\n    APIViewTest,\n    UsesGetMethod,\n    Returns200,\n):\n    @pytest.fixture\n    def url(self):\n        return reverse(\'hello-world\')\n\n    def test_it_returns_hello_world(self, json):\n        expected = \'Hello, World!\'\n        actual = json\n        assert expected == actual\n```\n\nWhen we run pytest, we see two tests run\n\n```\n$ py.test\n\ntests/test_hello.py::TestHelloWorld::test_it_returns_200 <- pytest_drf/status.py PASSED [ 50%]\ntests/test_hello.py::TestHelloWorld::test_it_returns_hello_world PASSED                 [100%]\n```\n\n\n### What happened here? \n\nWell, `APIViewTest` defines a `response` fixture that takes the value of the `url` fixture and makes an HTTP request to it. The `UsesGetMethod` mixin defines the `http_method` as `\'get\'`, making it a GET request. The `response` fixture is defined as `autouse=True`, so it\'s automatically evaluated before the test method is executed.\n\nThe `Returns200` mixin simply adds a test with `assert response.status_code == 200`. This test has its own request/response cycle, independent of `test_it_returns_hello_world` — this way, if the endpoint responds with the wrong status code but the correct response data, one can see what needs fixing after running `py.test` only once!\n\n\n### What\'s with the `json` fixture?\n\nWhy does `test_it_returns_hello_world` use the `json` fixture, instead of `response`?\n\n`json` is another fixture that merely returns `response.json()`. Because the API response is provided as a fixture, we can perform any post-processing transformations on it in another fixture — this lets us keep all those irrelevant implementation details out of the test methods, so they can focus on one thing and one thing only: verifying the data is correct.\n\n\n### Do I have to use this `expected == actual` business?\n\nNo, but it\'s Encouraged™! By explicitly labeling the known value and the value being tested, it\'ll be easier for another reader (including future you) to understand what\'s actually going on.\n\nThe specific order of `expected` and `actual` in the assertion (i.e. whether it\'s `expected == actual` or `actual == expected`) isn\'t as important as maintaining the same order everywhere. This will ease the interpretation of test failures, as, e.g., you\'ll always know which side of a diff contains the incorrect values.\n\nThe order `expected == actual` is used in pytest-drf\'s mixins to serve a left-to-right bias, where `expected == actual` always puts the known value first.\n\n\n### Reducing the bulkiness of fixtures\n\nThough a fixture-based approach like this can be very convenient in terms of configuration and hiding implementation details, it can also consume a lot of valuable screen real estate — especially if the fixtures consist entirely of a return statement.\n\nFor simple, one-line fixtures, [pytest-lambda](https://github.com/they4kman/pytest-lambda) (included as a dependency to pytest-drf) enables expressing them as lambda functions. Here\'s what our test file looks like with a lambda fixture\n\n```python\nfrom django.urls import reverse\nfrom pytest_drf import APIViewTest, Returns200, UsesGetMethod\nfrom pytest_lambda import lambda_fixture\n\nclass TestHelloWorld(\n    APIViewTest,\n    UsesGetMethod,\n    Returns200,\n):\n    url = lambda_fixture(lambda: reverse(\'hello-world\'))\n\n    def test_it_returns_hello_world(self, json):\n        expected = \'Hello, World!\'\n        actual = json\n        assert expected == actual\n```\n\nAdditionally, for constant value fixtures, pytest-lambda provides `static_fixture`\n\n```python\n\n```\n\n\n## What about authentication?\n\nTo make requests as someone other than `AnonymousUser`, pytest-drf provides the `AsUser()` mixin. Simply create / expose a user in a fixture, and include the mixin with your view test\n\n```python\nfrom django.contrib.auth.models import User\nfrom django.urls import reverse\nfrom pytest_drf import APIViewTest, AsUser, Returns200, UsesGetMethod\nfrom pytest_lambda import lambda_fixture\n\nalice = lambda_fixture(\n    lambda: User.objects.create(\n        username=\'alice\',\n        first_name=\'Alice\',\n        last_name=\'Innchains\',\n        email=\'alice@ali.ce\',\n    ))\n\nclass TestAboutMe(\n    APIViewTest,\n    UsesGetMethod,\n    Returns200,\n    AsUser(\'alice\'),\n):\n    url = lambda_fixture(lambda: reverse(\'about-me\'))\n\n    def test_it_returns_profile(self, json):\n        expected = {\n            \'username\': \'alice\',\n            \'first_name\': \'Alice\',\n            \'last_name\': \'Innchains\',\n            \'email\': \'alice@ali.ce\',\n        }\n        actual = json\n        assert expected == actual\n```\n\n\n## But I mainly use ViewSets, not APIViews!\n\npytest-drf offers `ViewSetTest`, along with some mixins and conventions, to aid in testing ViewSets. \n\nConsider this `KeyValue` model. It\'s got an ID, a string key, and a string value \n\n```python\n# kv/models.py\n\nfrom django.db import models\n\nclass KeyValue(models.Model):\n    key = models.CharField(max_length=32, unique=True)\n    value = models.CharField(max_length=32)\n```\n\nWe implement the endpoints with a `ModelViewSet` and `ModelSerializer`\n\n```python\n# kv/views.py\n\nfrom rest_framework import permissions, serializers, viewsets\nfrom rest_framework.pagination import PageNumberPagination\n\nfrom kv.models import KeyValue\n\nclass KeyValueSerializer(serializers.ModelSerializer):\n    class Meta:\n        model = KeyValue\n        fields = (\n            \'id\',\n            \'key\',\n            \'value\',\n        )\n\nclass KeyValueViewSet(viewsets.ModelViewSet):\n    queryset = KeyValue.objects.order_by(\'id\')\n    serializer_class = KeyValueSerializer\n    pagination_class = PageNumberPagination\n    permission_classes = [permissions.AllowAny]\n```\n\nWe route it with a base path of `/kv`, leading to the registration of two routes (list and detail) servicing list, create, retrieve, update, and destroy actions\n\n - `GET /kv` — list KeyValues\n - `POST /kv` — create a KeyValue\n - `GET /kv/<pk>` — retrieve a KeyValue\n - `PATCH /kv/<pk>` — update a KeyValue\n - `DELETE /kv/<pk>` — delete a KeyValue\n\n```python\n# kv/urls.py\n\nfrom django.urls import include, path\nfrom rest_framework import routers\n\nfrom kv import views\n\nrouter = routers.DefaultRouter()\nrouter.register(\'kv\', views.KeyValueViewSet, basename=\'key-values\')\n\nurlpatterns = [\n    path(\'\', include(router.urls)),\n]\n```\n\nNow, using `ViewSetTest`, we scaffold our tests\n\n```python\n# tests/test_kv.py\n\nfrom pytest_drf import (\n    ViewSetTest,\n    Returns200,\n    Returns201,\n    Returns204,\n    UsesGetMethod,\n    UsesDeleteMethod,\n    UsesDetailEndpoint,\n    UsesListEndpoint,\n    UsesPatchMethod,\n    UsesPostMethod,\n)\nfrom pytest_drf.util import url_for\nfrom pytest_lambda import lambda_fixture\n\nclass TestKeyValueViewSet(ViewSetTest):\n    list_url = lambda_fixture(\n        lambda:\n            url_for(\'key-values-list\'))\n\n    detail_url = lambda_fixture(\n        lambda key_value:\n            url_for(\'key-values-detail\', key_value.pk))\n\n    class TestList(\n        UsesGetMethod,\n        UsesListEndpoint,\n        Returns200,\n    ):\n        pass\n\n    class TestCreate(\n        UsesPostMethod,\n        UsesListEndpoint,\n        Returns201,\n    ):\n        pass\n\n    class TestRetrieve(\n        UsesGetMethod,\n        UsesDetailEndpoint,\n        Returns200,\n    ):\n        pass\n\n    class TestUpdate(\n        UsesPatchMethod,\n        UsesDetailEndpoint,\n        Returns200,\n    ):\n        pass\n\n    class TestDestroy(\n        UsesDeleteMethod,\n        UsesDetailEndpoint,\n        Returns204,\n    ):\n        pass\n```\n\nAt the moment, most of these tests will fail, because we haven\'t yet provided an implementation for the `key_value` fixture (though, `TestCreate` fails on serializer validation, as we haven\'t passed any required fields, yet).\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED    [ 20%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py FAILED  [ 40%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py ERROR [ 60%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py ERROR   [ 80%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py ERROR  [100%]\n```\n\nThe skeleton above, which we\'ll fill out soon, is just to show the key components of a `ViewSetTest`, which are:\n\n\n### Implementing `list_url` and `detail_url`\n\nUnlike APIViews, ViewSets often have two separate routes: one for the general set of resources (the list endpoint), and one for a specific resource (the detail endpoint). Since these are the same URLs regardless of the particular ViewSet action, we implement them once at the beginning of the ViewSetTest, and switch between them in nested classes using `UsesListEndpoint` and `UsesDetailEndpoint`.\n\nSince the detail endpoint acts upon a specific instance, `detail_url` acts upon the `key_value` fixture, which we will implement in the Retrieve/Update/Destroy tests by creating a `KeyValue` row.\n\n\n### ViewSet action child test contexts (nested classes)\n\nFor each ViewSet action (i.e. endpoint/http method combination), we\'ll create a separate nested class (AKA test context) with appropriate mixins to denote which endpoint to hit (implemented by `list_url` or `detail_url`), and with which HTTP method. We also add a `Returns2XX` mixin for each endpoint, which quickly lets us run tests that hit the actual ViewSet.\n\nNow, before we get to writing the crux of our CRUD tests, we\'re gonna implement an _expression method_, which will aid us in verifying API responses.\n \n### Expression methods\n\nAn expression method is a simple function that takes a model instance and returns what the API would respond for it. Our serializer has three fields: `id`, `key`, and `value`. So our expression method looks something like this:\n\n```python\ndef express_key_value(kv: KeyValue) -> Dict[str, Any]:\n    return {\n        \'id\': kv.id,\n        \'key\': kv.key,\n        \'value\': kv.value,\n    }\n```\n\nNow, when we make assertions on our API endpoint\'s response, we can easily verify the integrity of the whole data. For list responses, though, it would be nice to pass a list of `KeyValue` instances and get a list of expressions back. For that, we have `pytest_drf.util.pluralized`:\n\n```python\nfrom pytest_drf.util import pluralized\n\nexpress_key_values = pluralized(express_key_value)\n```\n```python\nIn [1]: a = KeyValue.objects.create(key=\'apples\', value=\'oranges\')\n\nIn [2]: b = KeyValue.objects.create(key=\'bananas\', value=\'B A NA NA S\')\n\nIn [3]: express_key_value(a)\nOut[3]: {\'id\': 1, \'key\': \'apples\', \'value\': \'oranges\'}\n\nIn [4]: express_key_value(b)\nOut[4]: {\'id\': 2, \'key\': \'bananas\', \'value\': \'B A NA NA S\'}\n\nIn [5]: express_key_values([a, b])\nOut[5]:\n[{\'id\': 1, \'key\': \'apples\', \'value\': \'oranges\'},\n {\'id\': 2, \'key\': \'bananas\', \'value\': \'B A NA NA S\'}]\n```\n\nNow that we\'re equipped, let\'s get to writing tests for each ViewSet action\n\n\n## `TestList`\n\nTo verify the behavior of the list action, we want to check that it returns the proper structure for each `KeyValue` in the right order. So, let\'s actually create some `KeyValues` for our endpoint to respond with\n\n```python\nclass TestList(\n    UsesGetMethod,\n    UsesListEndpoint,\n    Returns200,\n):\n    key_values = lambda_fixture(\n        lambda: [\n            KeyValue.objects.create(key=key, value=value)\n            for key, value in {\n                \'quay\': \'worth\',\n                \'chi\': \'revenue\',\n                \'umma\': \'gumma\',\n            }.items()\n        ],\n        autouse=True,\n    )\n```\n\nWe use a dictionary to supply the information, and a list comprehension to create the actual rows. Note that we set `autouse=True`, so our `Returns200` test has a substantial response. If we didn\'t set it, the `Returns200` test wouldn\'t have any real rows — which might mask an exception in serialization.\n\nAnd now let\'s verify the endpoint\'s response\n\n```python\ndef test_it_returns_key_values(self, key_values, results):\n    expected = express_key_values(sorted(key_values, key=lambda kv: kv.id))\n    actual = results\n    assert expected == actual\n```\n\nHere we use the `results` fixture, which is equivalent to `response.json()[\'results\']` — it\'s where the actual API response is when using a pagination class. Our endpoint is using `PageNumberPagination`.\n\nFor our `expected` value, we lean on the pluralized version of our expression method. We pass it our source `KeyValue` instances, sorted by ID, because our `ViewSet` orders them the same.\n\nLet\'s run the tests, and see how we\'re coming along\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED     [ 16%]\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_key_values PASSED                      [ 33%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py FAILED   [ 50%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py ERROR  [ 66%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py ERROR    [ 83%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py ERROR   [100%]\n```\n\nAyyy, it passed! Making progress!\n\n\n## `TestCreate`\n\nWhen it comes to the "C" in "CRUD", there are a few behaviors the endpoint should and shouldn\'t have, if we\'re aiming to be thorough. Namely, the endpoint:\n\n - Should create a new `KeyValue`, and _only one_ `KeyValue` (and also shouldn\'t delete any others)\n - Should create a new `KeyValue` with precisely the data we POST\n - Should return the newly-created `KeyValue` in the proper structure\n\nBefore we get to those tests, let\'s setup our POST data, which is super simple:\n\n```python\nclass TestCreate(\n    UsesPostMethod,\n    UsesListEndpoint,\n    Returns201,\n):\n    data = static_fixture({\n        \'key\': \'snakes\',\n        \'value\': \'hissssssss\',\n    })\n```\n\nWe lean on [pytest-lambda](https://github.com/they4kman/pytest-lambda)\'s `static_fixture`, which creates a fixture that returns a constant value. The `data` fixture will be POSTed as `client.post(data=data)`.\n\nNow, to test whether the endpoint creates one and only one `KeyValue`, we must record which `KeyValue` rows exist _before_ the request, and compare them with the rows that exist _after_ the request. For this, we\'ll use `precondition_fixture`, a feature of [pytest-common-subject](https://github.com/theY4Kman/pytest-common-subject), which pytest-drf is built upon. Though the fixture performing each test\'s HTTP request is evaluated later than all others (using the `@pytest.mark.late` mark from [pytest-fixture-order](https://github.com/theY4Kman/pytest-fixture-order)), `precondition_fixture` uses pytest\'s dependency graph to 100% ensure it\'s evaluated before the request.\n\n```python\nfrom pytest_common_subject import precondition_fixture\n\ninitial_key_value_ids = precondition_fixture(\n    lambda:\n        set(KeyValue.objects.values_list(\'id\', flat=True)))\n\ndef test_it_creates_new_key_value(self, initial_key_value_ids, json):\n    expected = initial_key_value_ids | {json[\'id\']}\n    actual = set(KeyValue.objects.values_list(\'id\', flat=True))\n    assert expected == actual\n```\n\nSo, we record a set of all the existing `KeyValue` IDs with our `precondition_fixture`. What we expect to see after the request is that only one new ID has been added: the ID of our newly-created `KeyValue`, which is returned in the API response. Let\'s check our test results now\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED     [ 14%]\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_key_values PASSED                      [ 28%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py PASSED   [ 42%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_creates_new_key_value PASSED                 [ 57%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py ERROR  [ 71%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py ERROR    [ 85%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py ERROR   [100%]\n```\n\nSuccess!\n\nNext up is verifying that the newly-created `KeyValue` has exactly the field values we POSTed.\n\n```python\nfrom pytest_assert_utils import assert_model_attrs\n\ndef test_it_sets_expected_attrs(self, data, json):\n    key_value = KeyValue.objects.get(pk=json[\'id\'])\n\n    expected = data\n    assert_model_attrs(key_value, expected)\n```\n\nHere we lean on `assert_model_attrs` from [pytest-assert-utils](https://github.com/theY4Kman/pytest-assert-utils), which performs a dict-to-dict comparison under the hood, so if the items aren\'t equal, pytest will display a diff. Let\'s run it\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED     [ 12%]\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_key_values PASSED                      [ 25%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py PASSED   [ 37%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_creates_new_key_value PASSED                 [ 50%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_sets_expected_attrs PASSED                   [ 62%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py ERROR  [ 75%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py ERROR    [ 87%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py ERROR   [100%]\n```\n\nNice!\n\nLast but not least, we verify the structure of the API response. \n\n```python\ndef test_it_returns_key_value(self, json):\n    key_value = KeyValue.objects.get(pk=json[\'id\'])\n\n    expected = express_key_value(key_value)\n    actual = json\n    assert expected == actual\n```\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED     [ 11%]\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_key_values PASSED                      [ 22%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py PASSED   [ 33%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_creates_new_key_value PASSED                 [ 44%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_sets_expected_attrs PASSED                   [ 55%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_key_value PASSED                     [ 66%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py ERROR  [ 77%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py ERROR    [ 88%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py ERROR   [100%]\n```\n\nYay! Our create action is covered!\n\n\n## `TestRetrieve`\n\nRetrieval is simple to test: just create a `KeyValue` beforehand, and verify the endpoint returns it in the proper structure. It\'s also the first of the ViewSet actions to use the `detail_url`. We\'re gonna define the `key_value` fixture that `detail_url` requests\n\n```python\nclass TestRetrieve(\n    UsesGetMethod,\n    UsesDetailEndpoint,\n    Returns200,\n):\n    key_value = lambda_fixture(\n        lambda:\n            KeyValue.objects.create(\n                key=\'monty\',\n                value=\'jython\',\n            ))\n```\n\nAnd with our expression method, verifying the API response is a cinch\n\n```python\ndef test_it_returns_key_value(self, key_value, json):\n    expected = express_key_value(key_value)\n    actual = json\n    assert expected == actual\n```\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED     [ 10%]\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_key_values PASSED                      [ 20%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py PASSED   [ 30%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_creates_new_key_value PASSED                 [ 40%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_sets_expected_attrs PASSED                   [ 50%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_key_value PASSED                     [ 60%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py PASSED [ 70%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_key_value PASSED                   [ 80%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py ERROR    [ 90%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py ERROR   [100%]\n```\n\nAlmost done!\n\n\n## `TestUpdate`\n\nThe update endpoint should do two things:\n\n - Change the `KeyValue` row in the database to match _precisely_ what we\'ve POSTed\n - Return the updated `KeyValue` in the proper structure\n\nLike our retrieve tests, we\'ll start by defining the `key_value` fixture, which we\'ll be updating through our request; but we\'ll also declare the data we\'ll be POSTing\n\n```python\nclass TestUpdate(\n    UsesPatchMethod,\n    UsesDetailEndpoint,\n    Returns200,\n):\n    key_value = lambda_fixture(\n        lambda:\n            KeyValue.objects.create(\n                key=\'pipenv\',\n                value=\'was a huge leap forward\',\n            ))\n\n    data = static_fixture({\n        \'key\': \'buuut poetry\',\n        \'value\': \'locks quicker and i like that\',\n    })\n```\n\nSo let\'s test that our POSTed data makes it to the database\n\n```python\ndef test_it_sets_expected_attrs(self, data, key_value):\n    # We must tell Django to grab fresh data from the database, or we\'ll\n    # see our stale initial data and think our endpoint is broken!\n    key_value.refresh_from_db()\n\n    expected = data\n    assert_model_attrs(key_value, expected)\n```\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED     [  9%]\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_key_values PASSED                      [ 18%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py PASSED   [ 27%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_creates_new_key_value PASSED                 [ 36%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_sets_expected_attrs PASSED                   [ 45%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_key_value PASSED                     [ 54%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py PASSED [ 63%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_key_value PASSED                   [ 72%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py PASSED   [ 81%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_sets_expected_attrs PASSED                   [ 90%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py ERROR   [100%]\n```\n\nExcellent.\n\nNow just the standard response structure verification, though this time we\'ll have to do the same `refresh_from_db()` dance as the last test, for the same reason\n\n```python\ndef test_it_returns_key_value(self, key_value, json):\n    key_value.refresh_from_db()\n\n    expected = express_key_value(key_value)\n    actual = json\n    assert expected == actual\n```\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED     [  8%]\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_key_values PASSED                      [ 16%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py PASSED   [ 25%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_creates_new_key_value PASSED                 [ 33%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_sets_expected_attrs PASSED                   [ 41%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_key_value PASSED                     [ 50%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py PASSED [ 58%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_key_value PASSED                   [ 66%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py PASSED   [ 75%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_sets_expected_attrs PASSED                   [ 83%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_key_value PASSED                     [ 91%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py ERROR   [100%]\n```\n\nBoom! Just the destroy action left!\n\n\n## `TestDestroy`\n\nLast but not least (well, maybe in terms of `KeyValue.objects.count()`), we have our deletion endpoint. Its only behavior is to delete the specified `KeyValue`, and _only_ the specified `KeyValue`. Similar to `TestCreate`, we\'ll use a `precondition_fixture` to record the initial set of `KeyValue` IDs — though, this time we\'ll include the `KeyValue` we\'ll be deleting in the list\n\n```python\nclass TestDestroy(\n    UsesDeleteMethod,\n    UsesDetailEndpoint,\n    Returns204,\n):\n    key_value = lambda_fixture(\n        lambda:\n            KeyValue.objects.create(\n                key=\'i love\',\n                value=\'YOU\',\n            ))\n\n    initial_key_value_ids = precondition_fixture(\n        lambda key_value:  # ensure our to-be-deleted KeyValue exists in our set\n            set(KeyValue.objects.values_list(\'id\', flat=True)))\n\n    def test_it_deletes_key_value(self, initial_key_value_ids, key_value):\n        expected = initial_key_value_ids - {key_value.id}\n        actual = set(KeyValue.objects.values_list(\'id\', flat=True))\n        assert expected == actual\n```\n\n```bash\n$ py.test --tb=no\n\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_200 <- pytest_drf/status.py PASSED     [  7%]\ntests/test_kv.py::TestKeyValueViewSet::TestList::test_it_returns_key_values PASSED                      [ 15%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_201 <- pytest_drf/status.py PASSED   [ 23%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_creates_new_key_value PASSED                 [ 30%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_sets_expected_attrs PASSED                   [ 38%]\ntests/test_kv.py::TestKeyValueViewSet::TestCreate::test_it_returns_key_value PASSED                     [ 46%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_200 <- pytest_drf/status.py PASSED [ 53%]\ntests/test_kv.py::TestKeyValueViewSet::TestRetrieve::test_it_returns_key_value PASSED                   [ 61%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_200 <- pytest_drf/status.py PASSED   [ 69%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_sets_expected_attrs PASSED                   [ 76%]\ntests/test_kv.py::TestKeyValueViewSet::TestUpdate::test_it_returns_key_value PASSED                     [ 84%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_returns_204 <- pytest_drf/status.py PASSED  [ 92%]\ntests/test_kv.py::TestKeyValueViewSet::TestDestroy::test_it_deletes_key_value PASSED                    [100%]\n\n============================================= 13 passed in 0.42s ==============================================\n```\n\nYaaaaay!\n\n\n## Putting it all together\n\n```python\n# tests/test_kv.py\n\nfrom typing import Any, Dict\n\nfrom pytest_common_subject import precondition_fixture\nfrom pytest_drf import (\n    ViewSetTest,\n    Returns200,\n    Returns201,\n    Returns204,\n    UsesGetMethod,\n    UsesDeleteMethod,\n    UsesDetailEndpoint,\n    UsesListEndpoint,\n    UsesPatchMethod,\n    UsesPostMethod,\n)\nfrom pytest_drf.util import pluralized, url_for\nfrom pytest_lambda import lambda_fixture, static_fixture\nfrom pytest_assert_utils import assert_model_attrs\n\n\ndef express_key_value(kv: KeyValue) -> Dict[str, Any]:\n    return {\n        \'id\': kv.id,\n        \'key\': kv.key,\n        \'value\': kv.value,\n    }\n\nexpress_key_values = pluralized(express_key_value)\n\n\nclass TestKeyValueViewSet(ViewSetTest):\n    list_url = lambda_fixture(\n        lambda:\n            url_for(\'key-values-list\'))\n\n    detail_url = lambda_fixture(\n        lambda key_value:\n            url_for(\'key-values-detail\', key_value.pk))\n\n    class TestList(\n        UsesGetMethod,\n        UsesListEndpoint,\n        Returns200,\n    ):\n        key_values = lambda_fixture(\n            lambda: [\n                KeyValue.objects.create(key=key, value=value)\n                for key, value in {\n                    \'quay\': \'worth\',\n                    \'chi\': \'revenue\',\n                    \'umma\': \'gumma\',\n                }.items()\n            ],\n            autouse=True,\n        )\n\n        def test_it_returns_key_values(self, key_values, results):\n            expected = express_key_values(sorted(key_values, key=lambda kv: kv.id))\n            actual = results\n            assert expected == actual\n\n    class TestCreate(\n        UsesPostMethod,\n        UsesListEndpoint,\n        Returns201,\n    ):\n        data = static_fixture({\n            \'key\': \'snakes\',\n            \'value\': \'hissssssss\',\n        })\n\n        initial_key_value_ids = precondition_fixture(\n            lambda:\n                set(KeyValue.objects.values_list(\'id\', flat=True)))\n\n        def test_it_creates_new_key_value(self, initial_key_value_ids, json):\n            expected = initial_key_value_ids | {json[\'id\']}\n            actual = set(KeyValue.objects.values_list(\'id\', flat=True))\n            assert expected == actual\n\n        def test_it_sets_expected_attrs(self, data, json):\n            key_value = KeyValue.objects.get(pk=json[\'id\'])\n\n            expected = data\n            assert_model_attrs(key_value, expected)\n\n        def test_it_returns_key_value(self, json):\n            key_value = KeyValue.objects.get(pk=json[\'id\'])\n\n            expected = express_key_value(key_value)\n            actual = json\n            assert expected == actual\n\n\n    class TestRetrieve(\n        UsesGetMethod,\n        UsesDetailEndpoint,\n        Returns200,\n    ):\n        key_value = lambda_fixture(\n            lambda:\n                KeyValue.objects.create(\n                    key=\'monty\',\n                    value=\'jython\',\n                ))\n\n        def test_it_returns_key_value(self, key_value, json):\n            expected = express_key_value(key_value)\n            actual = json\n            assert expected == actual\n\n    class TestUpdate(\n        UsesPatchMethod,\n        UsesDetailEndpoint,\n        Returns200,\n    ):\n        key_value = lambda_fixture(\n            lambda:\n                KeyValue.objects.create(\n                    key=\'pipenv\',\n                    value=\'was a huge leap forward\',\n                ))\n\n        data = static_fixture({\n            \'key\': \'buuut poetry\',\n            \'value\': \'locks quicker and i like that\',\n        })\n\n        def test_it_sets_expected_attrs(self, data, key_value):\n            # We must tell Django to grab fresh data from the database, or we\'ll\n            # see our stale initial data and think our endpoint is broken!\n            key_value.refresh_from_db()\n\n            expected = data\n            assert_model_attrs(key_value, expected)\n\n        def test_it_returns_key_value(self, key_value, json):\n            key_value.refresh_from_db()\n\n            expected = express_key_value(key_value)\n            actual = json\n            assert expected == actual\n\n    class TestDestroy(\n        UsesDeleteMethod,\n        UsesDetailEndpoint,\n        Returns204,\n    ):\n        key_value = lambda_fixture(\n            lambda:\n                KeyValue.objects.create(\n                    key=\'i love\',\n                    value=\'YOU\',\n                ))\n\n        initial_key_value_ids = precondition_fixture(\n            lambda key_value:  # ensure our to-be-deleted KeyValue exists in our set\n                set(KeyValue.objects.values_list(\'id\', flat=True)))\n\n        def test_it_deletes_key_value(self, initial_key_value_ids, key_value):\n            expected = initial_key_value_ids - {key_value.id}\n            actual = set(KeyValue.objects.values_list(\'id\', flat=True))\n            assert expected == actual\n```\n\nIt\'s quite a feat!\n\nNow, we tested an already-existing endpoint here, just for demonstration purposes. But there\'s a bigger advantage to performing one request per test, and having a single responsibility for each test: we can write the tests first and incrementally build the ViewSet. We run the tests on changes, and when they\'re all green, we know the endpoint is done.\n\nThe beauty of the tests-first methodology is that it frees us up to be creative. Because we have a definite end condition, we can experiment with better implementations — more maintainable, easier to read, using best practices, perhaps leaning on a third-party package for heavy lifting.\n\nWell, congratulations if you\'ve made it this far. I hope you may find some value in this library, or even from some conventions in these example tests. Good luck out there, and remember: readability counts — in tests, doubly so.\n\n\n## Bonus: BDD syntax\n\nPersonally, I like to use `DescribeKeyValueViewSet`, and `DescribeList`, `DescribeCreate`, etc for my test classes. If I\'m testing `DescribeCreate` as a particular user, I like to use, e.g., `ContextAsAdmin`. Sometimes `CaseUnauthenticated` hits the spot.\n\nAnd for test methods, I love to omit the `test` in `test_it_does_xyz`, and simply put `it_does_xyz`.\n\nTo appease my leanings toward BDD namings, I use a `pytest.ini` with these options:\n\n```ini\n[pytest]\n# Only search for tests within files matching these patterns\npython_files = tests.py test_*.py\n\n# Discover tests within classes matching these patterns\n# NOTE: the dash represents a word boundary (functionality provided by pytest-camel-collect)\npython_classes = Test-* Describe-* Context-* With-* Without-* For-* When-* If-* Case-*\n\n# Only methods matching these patterns are considered tests\npython_functions = test_* it_* its_*\n```\n\nAbout the dashes in `python_classes`: sometimes I\'ll name a test class `ForAdminUsers`. If I had the pattern `For*`, it would also match a pytest-drf mixin named `ForbidsAnonymousUsers`. [pytest-camel-collect](https://github.com/theY4Kman/pytest-camel-collect) is a little plugin that interprets dashes in `python_classes` as CamelCase word boundaries. However, similar behavior can be had on stock pytest using a pattern like `For[A-Z0-9]*`.\n\nHere\'s what our example `KeyValueViewSet` test would look like with this BDD naming scheme\n\n<details>\n<summary>BDD-esque KeyValueViewSet test</summary>\n\n```python\nfrom typing import Any, Dict\n\nfrom pytest_common_subject import precondition_fixture\nfrom pytest_drf import (\n    ViewSetTest,\n    Returns200,\n    Returns201,\n    Returns204,\n    UsesGetMethod,\n    UsesDeleteMethod,\n    UsesDetailEndpoint,\n    UsesListEndpoint,\n    UsesPatchMethod,\n    UsesPostMethod,\n)\nfrom pytest_drf.util import pluralized, url_for\nfrom pytest_lambda import lambda_fixture, static_fixture\nfrom pytest_assert_utils import assert_model_attrs\n\n\ndef express_key_value(kv: KeyValue) -> Dict[str, Any]:\n    return {\n        \'id\': kv.id,\n        \'key\': kv.key,\n        \'value\': kv.value,\n    }\n\nexpress_key_values = pluralized(express_key_value)\n\n\nclass DescribeKeyValueViewSet(ViewSetTest):\n    list_url = lambda_fixture(\n        lambda:\n            url_for(\'key-values-list\'))\n\n    detail_url = lambda_fixture(\n        lambda key_value:\n            url_for(\'key-values-detail\', key_value.pk))\n\n    class DescribeList(\n        UsesGetMethod,\n        UsesListEndpoint,\n        Returns200,\n    ):\n        key_values = lambda_fixture(\n            lambda: [\n                KeyValue.objects.create(key=key, value=value)\n                for key, value in {\n                    \'quay\': \'worth\',\n                    \'chi\': \'revenue\',\n                    \'umma\': \'gumma\',\n                }.items()\n            ],\n            autouse=True,\n        )\n\n        def it_returns_key_values(self, key_values, results):\n            expected = express_key_values(sorted(key_values, key=lambda kv: kv.id))\n            actual = results\n            assert expected == actual\n\n    class DescribeCreate(\n        UsesPostMethod,\n        UsesListEndpoint,\n        Returns201,\n    ):\n        data = static_fixture({\n            \'key\': \'snakes\',\n            \'value\': \'hissssssss\',\n        })\n\n        initial_key_value_ids = precondition_fixture(\n            lambda:\n                set(KeyValue.objects.values_list(\'id\', flat=True)))\n\n        def it_creates_new_key_value(self, initial_key_value_ids, json):\n            expected = initial_key_value_ids | {json[\'id\']}\n            actual = set(KeyValue.objects.values_list(\'id\', flat=True))\n            assert expected == actual\n\n        def it_sets_expected_attrs(self, data, json):\n            key_value = KeyValue.objects.get(pk=json[\'id\'])\n\n            expected = data\n            assert_model_attrs(key_value, expected)\n\n        def it_returns_key_value(self, json):\n            key_value = KeyValue.objects.get(pk=json[\'id\'])\n\n            expected = express_key_value(key_value)\n            actual = json\n            assert expected == actual\n\n\n    class DescribeRetrieve(\n        UsesGetMethod,\n        UsesDetailEndpoint,\n        Returns200,\n    ):\n        key_value = lambda_fixture(\n            lambda:\n                KeyValue.objects.create(\n                    key=\'monty\',\n                    value=\'jython\',\n                ))\n\n        def it_returns_key_value(self, key_value, json):\n            expected = express_key_value(key_value)\n            actual = json\n            assert expected == actual\n\n    class DescribeUpdate(\n        UsesPatchMethod,\n        UsesDetailEndpoint,\n        Returns200,\n    ):\n        key_value = lambda_fixture(\n            lambda:\n                KeyValue.objects.create(\n                    key=\'pipenv\',\n                    value=\'was a huge leap forward\',\n                ))\n\n        data = static_fixture({\n            \'key\': \'buuut poetry\',\n            \'value\': \'locks quicker and i like that\',\n        })\n\n        def it_sets_expected_attrs(self, data, key_value):\n            # We must tell Django to grab fresh data from the database, or we\'ll\n            # see our stale initial data and think our endpoint is broken!\n            key_value.refresh_from_db()\n\n            expected = data\n            assert_model_attrs(key_value, expected)\n\n        def it_returns_key_value(self, key_value, json):\n            key_value.refresh_from_db()\n\n            expected = express_key_value(key_value)\n            actual = json\n            assert expected == actual\n\n    class DescribeDestroy(\n        UsesDeleteMethod,\n        UsesDetailEndpoint,\n        Returns204,\n    ):\n        key_value = lambda_fixture(\n            lambda:\n                KeyValue.objects.create(\n                    key=\'i love\',\n                    value=\'YOU\',\n                ))\n\n        initial_key_value_ids = precondition_fixture(\n            lambda key_value:  # ensure our to-be-deleted KeyValue exists in our set\n                set(KeyValue.objects.values_list(\'id\', flat=True)))\n\n        def it_deletes_key_value(self, initial_key_value_ids, key_value):\n            expected = initial_key_value_ids - {key_value.id}\n            actual = set(KeyValue.objects.values_list(\'id\', flat=True))\n            assert expected == actual\n```\n\n</details>\n',
    'author': 'Zach "theY4Kman" Kanzler',
    'author_email': 'they4kman@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/theY4Kman/pytest-drf',
    'packages': packages,
    'package_data': package_data,
    'py_modules': modules,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
