# `dag-cbor`: A Python implementation of [DAG-CBOR](https://ipld.io/specs/codecs/dag-cbor/spec/)

[![Generic badge](https://img.shields.io/badge/python-3.7+-green.svg)](https://docs.python.org/3.7/)
![PyPI version shields.io](https://img.shields.io/pypi/v/dag-cbor.svg)
[![PyPI status](https://img.shields.io/pypi/status/dag-cbor.svg)](https://pypi.python.org/pypi/dag-cbor/)
[![Checked with Mypy](http://www.mypy-lang.org/static/mypy_badge.svg)](https://github.com/python/mypy)
[![Python package](https://github.com/hashberg-io/py-dag-cbor/actions/workflows/python-pytest.yml/badge.svg)](https://github.com/hashberg-io/py-dag-cbor/actions/workflows/python-pytest.yml)


This is a fully compliant Python implementation of the [DAG-CBOR codec](https://ipld.io/specs/codecs/dag-cbor/spec/), a subset of the [Concise Binary Object Representation (CBOR)](https://cbor.io/) supporting the [IPLD Data Model](https://ipld.io/docs/data-model/) and enforcing a unique (strict) encoded representation of items.

You can install this package with `pip`:

```
pip install dag-cbor
```

The [documentation](https://hashberg-io.github.io/py-dag-cbor/dag_cbor/index.html) for this package is automatically generated by [pdoc](https://pdoc3.github.io/pdoc/).

## Basic usage

The core functionality of the library is performed by the `encode` and `decode` functions:

```python
>>> import dag_cbor
>>> dag_cbor.encode({'a': 12, 'b': 'hello!'})
b'\xa2aa\x0cabfhello!'
>>> dag_cbor.decode(b'\xa2aa\x0cabfhello!')
{'a': 12, 'b': 'hello!'}
```

A buffered binary stream (i.e. an instance of `io.BufferedIOBase`) can be passed to the `encode` function using the optional keyword argument `stream`, in which case the encoded bytes are written to the stream rather than returned:

```python
>>> from io import BytesIO
>>> mystream = BytesIO()
>>> dag_cbor.encode({'a': 12, 'b': 'hello!'}, stream=mystream)
>>> mystream.getvalue()
b'\xa2aa\x0cabfhello!'
```

 A buffered binary stream can be passed to the decode function instead of a `bytes` object, in which case the contents of the stream are read in their entirety and decoded:

```python
>>> mystream = BytesIO(b'\xa2aa\x0cabfhello!')
>>> dag_cbor.decode(mystream)
{'a': 12, 'b': 'hello!'}
```

## Random data

The `random` module contains functions to generate random data compatible with DAG-CBOR encoding. The functions are named `rand_X`, where `X` is one of `int`, `bytes`, `str`, `bool` (for `False` and `True`), `bool_none` (for `False`, `True` or `None`), `list`, `dict`, or `cid` (for CID data, encoded as an instance of `BaseCID` from the [`py-cid`](https://github.com/ipld/py-cid) package). The function call `rand_X(n)` returns an iterator yielding a stream of `n` random values of type `X`:

```python
>>> import pprint
>>> from dag_cbor.random import rand_options, rand_dict
>>> options = {
... 	"min_codepoint": 0x30,
...     "max_codepoint": 0x7a,
...     "max_chars": 4,
...     "max_bytes": 8,
...		"include_cid": False
... }
>>> with rand_options(**options):
...     for d in rand_dict(3):
...             pprint.pp(d)
...
{'': False,
 'biq': b'\x9b\xf4\xb7oG\x90',
 'ctz': b'{\x94^`_',
 'rg': 1.0563551621961306e+308,
 'y': ['ejdx',
       [None,
        8009056699280396679,
        3.6758824653722026e+307,
        -1.5555617166832836e+308,
        ''],
       1.6466769889093459e+308,
       [-12301485475384051926,
        '',
        b'zH\xe5',
        18321703892302698705,
        8.795874874880185e+307,
        'jwdr']]}
{'': -1.2571678042268863e+308, 'vmwq': -3474837013640509145}
{'h': [['k', False, -14411084192553465082],
       1.6359295422392278e+308,
       'vcad',
       False],
 'nsio': b'*\xa6;\xf9\xab',
 'sdmc': -359404367281466347}
```

For the full list of functions and options, please refer to the [`dag_cbor.random` documentation](https://hashberg-io.github.io/py-dag-cbor/dag_cbor/random.html).

