# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['ec2_spot_price']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.17.12,<2.0.0', 'pandas>=1.2.2,<2.0.0']

entry_points = \
{'console_scripts': ['ec2_spot_price = ec2_spot_price.ec2_spot_price:main']}

setup_kwargs = {
    'name': 'ec2-spot-price',
    'version': '0.1.16',
    'description': 'ec2-spot-price: retrieve Amazon EC2 spot instance price',
    'long_description': '# ec2-spot-price: Retrieve Amazon EC2 spot instance price\n\nThis Python module provides simple functions and command to retrieve [Amazon EC2 spot instance price](https://aws.amazon.com/ec2/spot/pricing/) by AWS API\n\n\n## Install\n\n```sh\npip install ec2-spot-price\n```\n\n## Setup\n\nYou should setup [AWS authentication credentials](https://docs.aws.amazon.com/cli/latest/userguide/cli-configure-files.html) which have permissions to access EC2 [`DescribeSpotPriceHistory`](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html) API. A simple way to do is to [create IAM user](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users_create.html) and attach [`AmazonEC2ReadOnlyAccess`](https://console.aws.amazon.com/iam/home#policies/arn:aws:iam::aws:policy/AmazonEC2ReadOnlyAccess) policy directly. (Or you can use existent credentials which have permissions to access that API)\n\n### Access IAM console\n\nLogin to AWS console and access [IAM console](https://console.aws.amazon.com/iam/home)\n\n### Add new IAM user\n\nAdd new IAM user like the following.\n\n- User name: `myuser`  # whatever you want\n- Access type: Programmatic access\n- Set permissions: Attach existing policies directly\n- Policy name: `AmazonEC2ReadOnlyAccess`\n- Download .csv\n\n\n### Edit ~/.aws/credentials\n\nYou can use "[named profile](https://docs.aws.amazon.com/cli/latest/userguide/cli-configure-profiles.html)" to have multiple credentials settings.\n\n```ini:~/.aws/credentials\n[myprofile]  # whatever you want, or [default]\naws_access_key_id=[copy from csv]\naws_secret_access_key=[copy from csv]\nregion=us-east-2  # wherever you want\n```\n\n## Usage\n\n### Run as command\n\nYou can run `ec2_spot_price` command to retrieve spot instance prices.\n\nIf you use "named profile" in credentials file, you need to specify `AWS_PROFILE` environment variable.\n\n```sh\nexport AWS_PROFILE=myprofile\n```\n\nThen, run command `ec2_spot_price` (or `python /path/to/ec2_spot_price.py`).\n\n```sh\n% ec2_spot_price -h\nusage: ec2_spot_price.py [-h] [-r REGION_NAMES] [-i INSTANCE_TYPES]\n                         [-o OS_TYPES] [-s SEP] [-v]\n\nretrieve Amazon EC2 spot instance price\n\noptional arguments:\n  -h, --help            show this help message and exit\n  -r REGION_NAMES, --region_names REGION_NAMES\n                        filter regions. if "" is specified, retrieve all of\n                        the regions. (default: "us-east-1,us-east-2,us-\n                        west-1,us-west-2")\n  -i INSTANCE_TYPES, --instance_types INSTANCE_TYPES\n                        filter instance types e.g. "g3.4xlarge,p2.xlarge".\n                        (default: retrieve all of the instance types)\n  -o OS_TYPES, --os_types OS_TYPES\n                        filter OS types. (default: "Linux/UNIX")\n  -s SEP, --sep SEP     separator of CSV. (default: ",")\n  -v, --verbose         increase output verbosity\n```\n\nYou can specify region names by `-r`, instance types by `-i` and OS types by `-o`. For example, the following command shows CSV of `us-east-1,us-east-2` regions, `c5.xlarge,c5d.xlarge` instances and `Linux/UNIX` OS (default value).\n\n```sh\n% ec2_spot_price -r "us-east-1,us-east-2" -i "c5.xlarge,c5d.xlarge"\nSpotPrice,AvailabilityZone,InstanceType,ProductDescription,Timestamp\n0.038000,us-east-2a,c5.xlarge,Linux/UNIX,2021-02-21 02:12:51+00:00\n0.038000,us-east-2b,c5.xlarge,Linux/UNIX,2021-02-21 02:12:51+00:00\n0.038000,us-east-2c,c5.xlarge,Linux/UNIX,2021-02-21 02:12:51+00:00\n0.038000,us-east-2c,c5d.xlarge,Linux/UNIX,2021-02-21 08:05:53+00:00\n0.038100,us-east-2a,c5d.xlarge,Linux/UNIX,2021-02-21 04:50:26+00:00\n0.038100,us-east-2b,c5d.xlarge,Linux/UNIX,2021-02-21 03:59:40+00:00\n0.066400,us-east-1f,c5d.xlarge,Linux/UNIX,2021-02-20 20:03:30+00:00\n0.071700,us-east-1b,c5.xlarge,Linux/UNIX,2021-02-21 07:37:58+00:00\n0.071800,us-east-1a,c5d.xlarge,Linux/UNIX,2021-02-20 16:31:38+00:00\n0.073600,us-east-1d,c5.xlarge,Linux/UNIX,2021-02-21 06:21:58+00:00\n0.075000,us-east-1c,c5.xlarge,Linux/UNIX,2021-02-21 08:54:30+00:00\n0.077100,us-east-1a,c5.xlarge,Linux/UNIX,2021-02-21 08:03:27+00:00\n0.081500,us-east-1d,c5d.xlarge,Linux/UNIX,2021-02-21 07:55:25+00:00\n0.082700,us-east-1c,c5d.xlarge,Linux/UNIX,2021-02-21 03:41:22+00:00\n0.091100,us-east-1b,c5d.xlarge,Linux/UNIX,2021-02-21 09:45:08+00:00\n0.103000,us-east-1f,c5.xlarge,Linux/UNIX,2021-02-21 06:55:57+00:00\n```\n\nIn this case, you should use `c5.xlarge` at `us-east-2` region.\n\nAn another example to retrieve all of the spot prices in all regions with verbose option.\n\n```sh\n% ec2_spot_price -r "" -i "" -o "" -v > spot_prices.csv\n2021-02-21 20:21:38,686 INFO get_spot_price: retrieving from eu-north-1...\n2021-02-21 20:21:45,047 INFO get_spot_price: retrieving from eu-north-1...done. 1455 items.\n...\n2021-02-21 20:23:18,585 INFO get_spot_price: retrieving from us-west-2...\n2021-02-21 20:23:26,142 INFO get_spot_price: retrieving from us-west-2...done. 4880 items.\n2021-02-21 20:23:26,145 INFO get_spot_prices: retrieved 49627 items from [\'eu-north-1\', \'ap-south-1\', \'eu-west-3\', \'eu-west-2\', \'eu-west-1\', \'ap-northeast-2\', \'ap-northeast-1\', \'sa-east-1\', \'ca-central-1\', \'ap-southeast-1\', \'ap-southeast-2\', \'eu-central-1\', \'us-east-1\', \'us-east-2\', \'us-west-1\', \'us-west-2\'].\n```\n\nThen open `spot_prices.csv` with spread sheet application like Excel.\n\n\n### Use as module\n\nThere are two functions. `get_spot_prices` retrieves spot prices as list. `spot_prices_to_csv` convert spot prices to CSV.\n\n```python\n% python\n>>> import sys\n>>> import ec2_spot_price\n>>> r = ec2_spot_price.get_spot_prices([\'us-east-1\', \'us-east-2\'], [\'c5.xlarge\', \'c5d.xlarge\'], [\'Linux/UNIX\'])\n>>> len(r)\n16\n>>> ec2_spot_price.spot_prices_to_csv(r, path_or_buf=sys.stdout, index=False, sort=True)\nSpotPrice,AvailabilityZone,InstanceType,ProductDescription,Timestamp\n0.038000,us-east-2a,c5.xlarge,Linux/UNIX,2021-02-21 02:12:51+00:00\n0.038000,us-east-2b,c5.xlarge,Linux/UNIX,2021-02-21 02:12:51+00:00\n0.038000,us-east-2c,c5.xlarge,Linux/UNIX,2021-02-21 02:12:51+00:00\n0.038000,us-east-2c,c5d.xlarge,Linux/UNIX,2021-02-21 08:05:53+00:00\n0.038100,us-east-2a,c5d.xlarge,Linux/UNIX,2021-02-21 04:50:26+00:00\n0.038100,us-east-2b,c5d.xlarge,Linux/UNIX,2021-02-21 03:59:40+00:00\n0.066400,us-east-1f,c5d.xlarge,Linux/UNIX,2021-02-20 20:03:30+00:00\n0.071700,us-east-1b,c5.xlarge,Linux/UNIX,2021-02-21 07:37:58+00:00\n0.071800,us-east-1a,c5d.xlarge,Linux/UNIX,2021-02-20 16:31:38+00:00\n0.073600,us-east-1d,c5.xlarge,Linux/UNIX,2021-02-21 06:21:58+00:00\n0.075000,us-east-1c,c5.xlarge,Linux/UNIX,2021-02-21 08:54:30+00:00\n0.077100,us-east-1a,c5.xlarge,Linux/UNIX,2021-02-21 08:03:27+00:00\n0.081500,us-east-1d,c5d.xlarge,Linux/UNIX,2021-02-21 07:55:25+00:00\n0.082700,us-east-1c,c5d.xlarge,Linux/UNIX,2021-02-21 03:41:22+00:00\n0.091100,us-east-1b,c5d.xlarge,Linux/UNIX,2021-02-21 09:45:08+00:00\n0.103000,us-east-1f,c5.xlarge,Linux/UNIX,2021-02-21 06:55:57+00:00\n```\n\nAn another example to retrieve all of the spot prices in all regions.\nYou can use `pd.DataFrame` for more specific filtering.\n\n```python\n% python\n>>> import pandas as pd\n>>> import ec2_spot_price\n>>> r = ec2_spot_price.get_spot_prices([], [], [])\n>>> len(r)\n49627\n>>> df = pd.DataFrame(r)\n>>> df = df.query(\'ProductDescription == "Linux/UNIX"\')\n>>> df = df.drop([\'Timestamp\', \'ProductDescription\'], axis=1)\n>>> df = df.sort_values(by=[\'SpotPrice\', \'AvailabilityZone\', \'InstanceType\'])\n>>> df = df.query(\'InstanceType.str.match("c5.?\\.xlarge")\')\n>>> df.head(20)\n      AvailabilityZone InstanceType SpotPrice\n41314       us-east-2a    c5.xlarge  0.038000\n40412       us-east-2a   c5d.xlarge  0.038000\n39457       us-east-2a   c5n.xlarge  0.038000\n41313       us-east-2b    c5.xlarge  0.038000\n39456       us-east-2b   c5n.xlarge  0.038000\n41312       us-east-2c    c5.xlarge  0.038000\n40831       us-east-2c   c5d.xlarge  0.038000\n39455       us-east-2c   c5n.xlarge  0.038000\n41169       us-east-2b   c5d.xlarge  0.038100\n39969       us-east-2c   c5a.xlarge  0.043100\n40929       us-east-2b   c5a.xlarge  0.044400\n2501       ap-south-1c   c5a.xlarge  0.052800\n1836       ap-south-1a   c5a.xlarge  0.053300\n2406       ap-south-1b   c5a.xlarge  0.053400\n1926       ap-south-1a   c5n.xlarge  0.054100\n1816       ap-south-1b   c5d.xlarge  0.054100\n2858       ap-south-1a    c5.xlarge  0.054200\n2750       ap-south-1a   c5d.xlarge  0.054200\n1810       ap-south-1b    c5.xlarge  0.054200\n2109       ap-south-1b   c5n.xlarge  0.054200\n```\n\n## Links\n\n- https://aws.amazon.com/ec2/spot/pricing/\n- https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances-history.html\n- https://docs.aws.amazon.com/cli/latest/reference/ec2/describe-spot-price-history.html\n- https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html\n- https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users_create.html\n- https://boto3.amazonaws.com/v1/documentation/api/latest/guide/quickstart.html#configuration\n\n\n## Author\n\nSusumu OTA\n',
    'author': 'Susumu OTA',
    'author_email': '1632335+susumuota@users.noreply.github.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/susumuota/ec2-spot-price',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7.1,<4.0.0',
}


setup(**setup_kwargs)
