import ctypes
import json
import os
from io import BytesIO
from urllib.parse import parse_qs
from werkzeug.datastructures import EnvironHeaders
from django.http import JsonResponse
from django.utils.module_loading import import_string
from django.conf import settings
from . import ffi


class Query:
    def __init__(self, instance, request):
        self.handle = 0
        self.instance = instance

        self.request = request

    def process_request(self, headers):
        resp_input = ctypes.create_string_buffer(self.request)

        output_ptr = ctypes.c_char_p()
        output_len = ctypes.c_int()

        status_ptr = ctypes.c_char_p()
        status_len = ctypes.c_int()

        self.handle = ffi.process_request(self.instance,
                                          ctypes.create_string_buffer(headers), len(headers),
                                          resp_input, len(self.request),
                                          ctypes.byref(output_ptr), ctypes.byref(output_len),
                                          ctypes.byref(status_ptr), ctypes.byref(status_len))

        resp_dict = {}
        req_dict = {}

        if output_len.value:
            resp_dict = json.loads(output_ptr.value[:output_len.value].decode("utf-8"))

        if status_len.value:
            req_dict = json.loads(status_ptr.value[:status_len.value].decode("utf-8"))

        ffi.disposeMemory(ctypes.cast(output_ptr, ctypes.c_void_p))
        ffi.disposeMemory(ctypes.cast(status_ptr, ctypes.c_void_p))

        return resp_dict, req_dict

    def process_response(self, resp_body):
        if self.handle == 0:
            return None

        output_ptr = ctypes.c_char_p()
        output_len = ctypes.c_int()

        ffi.process_response(
            self.instance,
            self.handle,
            resp_body, len(resp_body),
            ctypes.byref(output_ptr), ctypes.byref(output_len)
        )

        output_dict = {}

        if output_len.value:
            output_dict = json.loads(output_ptr.value[:output_len.value].decode("utf-8"))

        ffi.disposeMemory(ctypes.cast(output_ptr, ctypes.c_void_p))
        ffi.disposeHandle(self.handle)

        return output_dict


class DjangoMiddleware:
    def __init__(self, get_response):
        # save response processing fn
        self.get_response = get_response

        self.instance = 0

        if ffi.library is None:
            # library is not found, skip middleware initialization
            return

        # default values
        self.path = '/graphql'

        c = ffi.Config()

        inigo_settings = {}

        if hasattr(settings, 'INIGO'):
            inigo_settings = settings.INIGO

        if inigo_settings.get('ENABLE') is False:
            return

        # process Inigo settings
        if inigo_settings.get('DEBUG'):
            c.debug = inigo_settings.get('DEBUG')
        else:
            # use regular DEBUG setting if specific is not provided
            if hasattr(settings, 'DEBUG'):
                c.debug = settings.DEBUG

        if inigo_settings.get('TOKEN'):
            c.token = str.encode(inigo_settings.get('TOKEN'))

        schema = None
        if inigo_settings.get('GRAPHENE_SCHEMA'):
            schema = import_string(inigo_settings.get('GRAPHENE_SCHEMA'))
        elif inigo_settings.get('SCHEMA_PATH'):
            if os.path.isfile(inigo_settings.get('SCHEMA_PATH')):
                with open(inigo_settings.get('SCHEMA_PATH'), 'r') as f:
                    schema = f.read()
        elif hasattr(settings, 'GRAPHENE') and settings.GRAPHENE.get('SCHEMA'):
            schema = import_string(settings.GRAPHENE.get('SCHEMA'))

        if schema:
            c.schema = str.encode(str(schema))

        if inigo_settings.get('PATH'):
            self.path = inigo_settings.get('PATH')

        # create Inigo instance
        self.instance = ffi.create(ctypes.byref(c))

        error = ffi.check_lasterror()
        if error:
            print("INIGO: " + error.decode('utf-8'))

        if self.instance == 0:
            print("INIGO: error, instance can not be created")

    def __call__(self, request):
        # ignore execution if Inigo is not initialized
        if self.instance == 0:
            return self.get_response(request)

        # 'path' guard -> /graphql
        if request.path != self.path:
            return self.get_response(request)

        # graphiql request
        if request.method == 'GET' and ("text/html" in request.META.get("HTTP_ACCEPT", "*/*")):
            return self.get_response(request)

        # support only POST and GET requests
        if request.method != 'POST' and request.method != 'GET':
            return self.get_response(request)

        # parse request
        gReq: bytes = b''
        if request.method == "POST":
            # read request from body
            gReq = request.body
        elif request.method == "GET":
            # read request from query param
            gReq = str.encode(json.dumps({'query': request.GET.get('query')}))
        q = Query(self.instance, gReq)

        # create inigo context if not present. Should exist before 'headers' call
        if hasattr(request, 'inigo') is False:
            request.inigo = InigoContext()

        # inigo: process request
        resp, req = q.process_request(self.headers(request))

        # introspection query
        if resp:
            return self.respond(resp)

        # modify query if required
        if req:
            if request.method == 'POST':
                body = json.loads(request.body)
                body.update({
                    'query': req.get('query'),
                    'operationName': req.get('operationName'),
                    'variables': req.get('variables'),
                })

                request._body = str.encode(json.dumps(body))
            elif request.method == 'GET':
                params = request.GET.copy()
                params.update({
                    'query': req.get('query')
                })
                request.GET = params

        # forward to request handler
        response = self.get_response(request)

        # inigo: process response
        processed_response = q.process_response(response.content)
        if processed_response:
            return self.respond(processed_response)

        return response

    @staticmethod
    def headers(request):
        headers = {}
        for key, value in request.headers.items():
            headers[key] = value.split(", ")

        return str.encode(json.dumps(headers))

    @staticmethod
    def respond(data):
        response = {
            'data': data.get('data'),
        }

        if data.get('errors'):
            response['errors'] = data.get('errors')

        if data.get('extensions'):
            response['extensions'] = data.get('extensions')

        return JsonResponse(response, status=200)


class FlaskMiddleware:
    def __init__(self, app):
        self.app = app.wsgi_app

        self.instance = 0

        if ffi.library is None:
            # library is not found, skip middleware initialization
            return

        # default values
        self.path = '/graphql'

        c = ffi.Config()

        inigo_settings = {}
        if 'INIGO' in app.config:
            inigo_settings = app.config.get('INIGO')

        if inigo_settings.get('ENABLE') is False:
            return

        # process Inigo settings
        if inigo_settings.get('DEBUG'):
            c.debug = inigo_settings.get('DEBUG')
        else:
            # use regular DEBUG setting if specific is not provided
            if 'DEBUG' in app.config:
                c.debug = app.config.get('DEBUG')

        if inigo_settings.get('TOKEN'):
            c.token = str.encode(inigo_settings.get('TOKEN'))

        schema = None
        if inigo_settings.get('GRAPHENE_SCHEMA'):
            schema = import_string(inigo_settings.get('GRAPHENE_SCHEMA'))
        elif inigo_settings.get('SCHEMA_PATH'):
            if os.path.isfile(inigo_settings.get('SCHEMA_PATH')):
                with open(inigo_settings.get('SCHEMA_PATH'), 'r') as f:
                    schema = f.read()
        elif 'GRAPHENE' in app.config and app.config.get('GRAPHENE').get('SCHEMA'):
            schema = import_string(app.config.get('GRAPHENE').get('SCHEMA'))

        if schema:
            c.schema = str.encode(str(schema))

        if inigo_settings.get('PATH'):
            self.path = inigo_settings.get('PATH')

        # create Inigo instance
        self.instance = ffi.create(ctypes.byref(c))

        error = ffi.check_lasterror()
        if error:
            print("INIGO: " + error.decode('utf-8'))

        if self.instance == 0:
            print("INIGO: error, instance can not be created")

    def __call__(self, environ, start_response):
        # ignore execution if Inigo is not initialized
        if self.instance == 0:
            return self.app(environ, start_response)

        # 'path' guard -> /graphql
        if environ['PATH_INFO'] != self.path:
            return self.app(environ, start_response)

        request_method = environ['REQUEST_METHOD']

        # graphiql request
        if request_method == 'GET' and ("text/html" in environ.get('HTTP_ACCEPT', '*/*')):
            return self.app(environ, start_response)

        # support only POST and GET requests
        if request_method != 'POST' and request_method != 'GET':
            return self.app(environ, start_response)

        # parse request
        g_req: bytes = b''
        if request_method == "POST":

            # Get the request body from the environ as reading from global request object caches it.
            if environ.get('wsgi.input'):
                content_length = environ.get('CONTENT_LENGTH')
                if content_length == '-1':
                    g_req = environ.get('wsgi.input').read(-1)
                else:
                    g_req = environ.get('wsgi.input').read(int(content_length))
                # reset request body for the nested app
                environ['wsgi.input'] = BytesIO(g_req)
        elif request_method == "GET":
            # Returns a dictionary in which the values are lists
            query_params = parse_qs(environ['QUERY_STRING'])
            data = {
                'query': query_params.get('query', [''])[0],
                'operationName': query_params.get('operationName', [''])[0],
                'variables': query_params.get('variables', [''])[0],
            }
            g_req = str.encode(json.dumps(data))

        q = Query(self.instance, g_req)

        # create inigo context if not present. Should exist before 'headers' call
        if environ.get('inigo') is None:
            environ['inigo'] = InigoContext()

        headers = dict(EnvironHeaders(environ).to_wsgi_list())

        # inigo: process request
        resp, req = q.process_request(self.headers(headers))

        # introspection query
        if resp:
            return self.respond(resp, start_response)

        # modify query if required
        if req:
            if request_method == 'GET':
                query_params = parse_qs(environ['QUERY_STRING'])
                query_params['query'] = req.get('query')
                environ['QUERY_STRING'] = '&'.join([f"{k}={v[0]}" for k, v in query_params.items()])
            elif request_method == 'POST':
                content_length = int(environ.get('CONTENT_LENGTH', 0))
                body = environ['wsgi.input'].read(content_length)
                try:
                    payload = json.loads(body)
                except ValueError:
                    payload = {}
                payload.update({
                    'query': req.get('query'),
                    'operationName': req.get('operationName'),
                    'variables': req.get('variables'),
                })
                payload_str = json.dumps(payload).encode('utf-8')
                environ['wsgi.input'] = BytesIO(payload_str)
                environ['CONTENT_LENGTH'] = str(len(payload_str))

        inner_status = None
        inner_headers = []
        inner_exc_info = None

        def start_response_collector(status, headers, exc_info=None):
            # Just collects the inner response headers, to be modified before sending to client
            nonlocal inner_status, inner_headers, inner_exc_info
            inner_status = status
            inner_headers = headers
            inner_exc_info = exc_info
            # Not calling start_response(), as we will modify the headers first.
            return None

        # forward to request handler
        # populates the inner_* vars, as triggers inner call of the collector closure
        response = self.app(environ, start_response_collector)
        content = b"".join(response).decode("utf8")

        # inigo: process response
        processed_response = q.process_response(content.encode("utf8"))
        if processed_response:
            return self.respond(processed_response, start_response)

        start_response(inner_status, inner_headers, inner_exc_info)
        return response

    @staticmethod
    def headers(headers_dict):
        headers = {}
        for key, value in headers_dict.items():
            headers[key] = value.split(", ")

        return str.encode(json.dumps(headers))

    @staticmethod
    def respond(data, start_response):
        response = {
            'data': data.get('data'),
        }

        if data.get('errors'):
            response['errors'] = data.get('errors')

        if data.get('extensions'):
            response['extensions'] = data.get('extensions')

        status = "200 OK"
        headers = [("Content-type", "application/json")]
        start_response(status, headers)

        return [json.dumps(response).encode("utf-8")]


class InigoContext:
    def __init__(self):
        self.__auth = None
        self.__blocked = False

    @property
    def auth(self):
        return self.__auth

    @auth.setter
    def auth(self, value):
        self.__auth = value

    @property
    def blocked(self):
        return self.__blocked

    def _block(self):
        self.__blocked = True
