# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['lightshield', 'lightshield.proxy']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.7.4,<4.0.0', 'aioredis>=1.3.1,<2.0.0']

setup_kwargs = {
    'name': 'lightshield',
    'version': '0.1.18b0',
    'description': 'Library part of the Lightshield tool.',
    'long_description': "\n# Lightshield (LS)\n\nThis tool provides a fully automated system to permanently mirror the official Riot APIs content\n with your local system. It uses a number of microservices to split requests done to different endpoints\n across separated processes and allows scaling each service according to your current data needs. \n This is especially usefull in the early stages of setting up a data repository.\n\nThe setup is meant for tasks using alot of match data while not wanting to set up their own data pipeline structure.\nLightshield does currently not pull **all** match details data but only a select subset (for details\ncheck the SQL schemas in the postgres folder). Changes can be made easily by simply expanding/replacing the service.\n\nLightshield is optimized to not repeat calls unecessarily. This comes at cost of having data added in a less than\nreal time fashion.\n \n\n## What Lightshield does **not** do well\n####*Deliver up to date data*  \nWhile it can get up to date data on user its not set up for this by default\n\n####*Gather data on unranked player*  \nBy default Lightshield pulls data through the league endpoint which requires a user to be ranked.\nAs such getting information over unranked player is not supported by default but can be manually added.\n\n## Structure in Short\n\nLightshield handles data through a distributable scalable network of triangular microservice structures.\nAll data collected is stored in a dedicated postgres database. Task handling and scalability is provided through a \nbuffering redis database.\n\n- Each cluster of services is responsible for a single server and requires an additional proxy for rate limiting and \nhandling of the API key.\n\nEach step of data processing is stored inside the Postgres database from which a single manager service creates tasks.\nWorker services then process the tasks, add data to the db and close the task.\n\n\n## Requirements\nLightshield runs on docker and can either be built through this repository or by pulling the images \n[directly from DockerHub](https://hub.docker.com/u/lightshield).\n\n\n## Setup\n\n### Env variables\nCopy and rename the included secrets_template.env into secrets.env\n\n### I. Network\n\nInitialize the network used to bundle all services together and allow communication: \n```shell script\ndocker network create lightshield\n```\nThe name can be changed but has to be updated in the compose files as well.\nIf you are planning on running it through docker swarm use the appropriate network type.\n\n### II. Database\nSet up a postgres database either locally, in docker (attached to the network) or remotely. The services currently expect\nno password verification as such using a remote postgres library should only be done if you can limit access through other means.\n\nDB Connection details can be configured through a secrets.env file (template file included).\n\nLightshield requires the in the postgres/ folder listed tables to be set up in the specified database under schemas\ncorresponding to the server they will contain data for. E.g. pulling data for NA1 requires setting up a schema `na1` (lower case)\nwith all tables inside said schema as well as a user `na1` which will be used to pull data from said schema.\n\n### III. Centralized Services\nServices included in the `compose-global.yaml` file are meant to be run centralized meaning they run server-agnostic (usually as a one-off).\nCurrently this refers to the proxy service as well as the buffer redis db.\nStart the services by properly refering to the compose file:\n```shell\n# Pull (specify a tag if needed, defaults to latest)\nTAG=latest docker-compose -f compose-global.yaml pull\n# Or build\ndocker-compose -f compose-global.yaml build\n# And then run\ndocker-compose -f compose-global.yaml up -d\n```\n\n### IV. Server Specific Structure\n\n#### For docker-compose\nRun docker compose with default parameters. The services require the selected server to be passed into the container via\nthe environment variable `SERVER`. In addition make sure to use different project names either through `-p [project name]`\nor through the env variable `COMPOSE_PROJECT_NAME`. This will stop multiple server setups from overwriting one another.\n\n```shell script\n# Build from source\ndocker-compose build\n# Or pull from docker hub (specify a tag if needed, defaults to latest)\nTAG=latest docker-compose pull\n# Run either with the -p tag\nSERVER=EUW1 docker-compose -p lightshield_euw1 up -d\n# Or env variable\nSERVER=EUW1 COMPOSE_PROJECT_NAME=lightshield_euw1 docker-compose up -d\n```\n#### For docker-swarm\nFollow the same guidelines explained for docker-compose. The images can either be built or pulled from DockerHub.\n`SERVER` still needs to be passed into the service.\nThe individual project name is passed through the stack name.\n```shell script\nSERVER=EUW1 docker stack deploy -c compose-services.yaml lightshield_euw1\n```\nThe compose file includes the base_image service which is just a unified default image for all other services. As such\nthis isn't an actual runnable service and would usually just immediatly exit when run in docker-compose. Swarm however tries\nto rebuild and restart the service continuously as such you need to manually remove it from the stack to avoid that.\n\n<hr>\n\n## Functionality\n\nServices are structured in general in a triangular form using the persistent postgres DB as source of truth,\na singular non-scalable manager to select new tasks and a scalable amount of microservices to work through those tasks.\n\n2 lists are used to buffer tasks:\n- A list of non-selected tasks that are yet to be worked on.\n- A list of selected tasks that are being or have been work on  \nThe manager service looks at both lists to determine if it should add a new task to the first non-selected task list.\n  Each worker service pulls tasks from the first list and adds them to the secondary list. Tasks are never marked as done\n  as the tasks are by default no longer eligible once inserted in the DB, e.g. if a summonerId has to be pulled then the manager\n  will only select accounts without summonerId as tasks. Once the summonerId is added it will automatically be ineligible.\n  All tasks that are older than `TASK_BLOCKING` (secrets.env) minutes in the secondary selected list are periodically removed by the manager\n  therefore making space for new tasks. As this is the only way tasks are being removed make sure to keep the limit just high enough\n  to assure that tasks currently being worked on are not removed while not having the queue overflow with already done tasks.\n  \n### Services\n\n#### League Ranking\nUses the league-exp endpoint to crawl all ranked user in a continuous circle. This service has no manager and only requires a one-off.\nvia the `UPDATE_INTERVAL=1` variable in the compose file you can limit the delay between cycles in hours. By default after finishing\nwith challenger the service will wait 1 hour to restart on Iron games.\n\n### Summoner ID\nUsing the summoner endpoint pulls the remaining IDs for each account that were not included in the league ranking response.\nThis is a one time tasks for each user but it will take alot of initial requests until all are done.\n\n### Match History\nPull and update the match history of user. Prioritizes user for which no calls have been made so far and, once all user have\nhad their history pulled user that have had the most soloQ matches played since their last update.\nUse the `MIN_MATCHES=20` variable to limit how many new matches a player has to play to even be considered for an update.\nBecause for each match 10 match-history are updated one should consider keeping this number high as to not make 10 match history\ncalls per new match. Setting it to 10 or 20 means that for each match played on average only 1 or .5 calls have to be made.\n\n### Match Details\nPull match details for a buffered match_id. This service pulls match details and adds a select number of attributes to the DB \n(check the SQL files for more info). If more/less data is needed you have to update the service + db schema accordingly.\nMatches are pulled newest to oldest and are not pulled  for matches older than defined through `DETAILS_CUTOFF` (secrets.env).\n\n### Match History\nCurrently not implemented, WIP.\n\n\n## Lightshield Tools\n\nTools and Code-Sections of the Lightshield Framework that were better fit to be provided through dependency\nrather then included in the main project.\n\n#### What currently doesn't work:\n- The keys used to save data in Redis are not linked to the API key, as such multiple keys have to use\nmultiple Redis servers.\n\n### Ratelimiter (WIP)\n\nMulti-Host async ratelimiting service. The clients each sync via a central redis server. \n\nSet up the proxy in an async context with redis connection details.\n```python\nfrom lightshield.proxy import Proxy\nimport aiohttp\n\nasync def run():\n    p = Proxy()\n    # Initiate the redis connector in async context\n    await p.init(host='localhost', port=5432)\n```\n\nMake singular calls directly through the Proxy.\n```python\nasync with aiohttp.ClientSession(headers={'X-Riot-Token': ''}) as session:\n    await p.request('https://euw1.api.riotgames.com/lol/league-exp/v4/entries/RANKED_SOLO_5x5/SILVER/I', session=session)\n```\n\nMake calls directly to one endpoint without having the proxy select the proper limits \nthrough a regex comparison to increase processing speed.\n```python\nasync with aiohttp.ClientSession(headers={'X-Riot-Token': ''}) as session:\n    zone = await p.get_endpoint('https://euw1.api.riotgames.com/lol/league-exp/v4/entries/')\n    for page in range(1, 10):\n        zone.request('https://euw1.api.riotgames.com/lol/league-exp/v4/entries/RANKED_SOLO_5x5/SILVER/I?page=%s' % page, session)\n```\n\n### Settings (WIP)\nThe settings file contains a number of variables that are used across the project.\nVariables can be set through:  \n`ENV > config.json > default`\n```python\nfrom lightshield import settings\n\nheaders = {'X-Riot-Token': settings.API_KEY}\n```\n",
    'author': 'Doctress',
    'author_email': 'lightshielddev@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/LightshieldDotDev/Lightshield',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
