import tomli
import tomli_w
import yaml
import json
import jsonschema
import subprocess
import copy
import uuid
import os
import re
import shutil
import ast
from pathlib import Path
from rich.console import Console
from typing import Union
import collections.abc

from flowui.cli.utils.constants import COLOR_PALETTE
from flowui.client.github_rest_client import GithubRestClient
from flowui.utils import dict_deep_update


console = Console()


class SetEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, set):
            return list(obj)
        return json.JSONEncoder.default(self, obj)


CONFIG_REQUIRED_FIELDS = {}


def validate_github_token(token: str) -> bool:
    """
    Validate GITHUB_ACCESS_TOKEN
    By now it is only accepting the ghp token.

    Args:
        token (str): Github access token (ghp)

    Returns:
        bool: True if token is valid, False otherwise.
    """
    regex = r"ghp_[0-9a-zA-Z]{35,40}"
    pattern = re.compile(regex)
    if pattern.match(token):
        return True
    return False


REQUIRED_ENV_VARS_VALIDATORS = {
    "GITHUB_ACCESS_TOKEN_OPERATORS": {
        "depends": lambda arg: arg.get('OPERATORS_REPOSITORY_SOURCE') == 'github',
        "validator_func": validate_github_token
    },
    "GITHUB_ACCESS_TOKEN_WORKFLOWS": {
        "depends": lambda arg: arg.get('OPERATORS_REPOSITORY_SOURCE') == 'github',
        "validator_func": validate_github_token
    }
}


def set_config_as_env(key: str, value: Union[str, int, float]) -> None:
    """
    Set an ENV variable with the key and value.

    Args:
        key (Union[str, int]): ENV var name
        value (Union[str, int, float]): ENV var value
    """
    key = str(key).strip().upper()
    os.environ[key] = str(value)


def validate_repository_name(name: str) -> None:
    regex = r'^[A-Za-z0-9_]*$'
    pattern = re.compile(regex)
    if pattern.match(name):
        return True
    return False


def validate_env_vars() -> None:
    """
    Validate user environment variables.
    The accepted variables are:
        - GITHUB_ACCESS_TOKEN: Token to access GitHub API.
    """
    # Set AIRFLOW_UID from host user id 
    # https://airflow.apache.org/docs/apache-airflow/stable/start/docker.html#setting-the-right-airflow-user
    # https://airflow.apache.org/docs/apache-airflow/stable/start/docker.html#environment-variables-supported-by-docker-compose
    uid = subprocess.run(["id", "-u"], capture_output=True, text=True)
    set_config_as_env("AIRFLOW_UID", int(uid.stdout)) 

    for var, validator in REQUIRED_ENV_VARS_VALIDATORS.items():
        if 'depends' in validator:
            should_exists = validator.get('depends')(CONFIG_REQUIRED_FIELDS)
        if not should_exists:
            continue
        env_var = os.environ.get(var, None)
        if env_var:
            continue
        console.print(f"{var} is not defined", style=f"bold {COLOR_PALETTE.get('warning')}")
        new_var = input(f"Enter the {var} value: ")
        while not validator.get('validator_func')(new_var):
            new_var = input(f"Wrong {var} format. Enter a new value: ")
        os.environ[var] = new_var


def validate_config(config_dict: dict) -> None:
    """
    Validate user config.toml file and save it to config_dict and as environment variables to be used by the docker-compose file
    """
    required_fields = list(CONFIG_REQUIRED_FIELDS.keys()).copy()
    required_secrets = list()
    sections = config_dict.keys()
    for section in sections:
        for key, value in config_dict.get(section).items():
            # Check if OPERATORS_SECRETS exist in environment
            if section == "repository" and key == "OPERATORS_SECRETS":
                for v in value:
                    if not os.getenv(v, None):
                        required_secrets.append(v)
            # Check if required fields were defined in config file
            elif key in CONFIG_REQUIRED_FIELDS:
                required_fields.remove(key.upper())
                if key in ["VOLUME_MOUNT_PATH_HOST", "FLOWUI_PATH_HOST"]:
                    value = str(Path(value).resolve())
                # Set config as env vars, it will be used by compose
                set_config_as_env(key, value) 
                CONFIG_REQUIRED_FIELDS[key.upper()] = value

    if len(required_fields) > 0:
        console.print("Missing required fields: {}".format(required_fields), style=f"bold {COLOR_PALETTE.get('error')}") 
    if len(required_secrets) > 0:
        missing = '\n'.join(required_secrets)
        console.print(f"Missing required operators secrets. These shold be defined in your ENV: \n{missing}", style=f"bold {COLOR_PALETTE.get('error')}") 


def validate_repository_structure() -> None:
    """
    Validate the Operators repository structure.
    The basic initial structure must contain:
    - config.toml
    - operators/
    - dependencies/
    """
    operators_path = Path(".") / "operators"
    dependencies_path = Path(".") / "dependencies"
    organized_flowui_path = Path(".") / ".flowui/"
    if not organized_flowui_path.is_dir():
        organized_flowui_path.mkdir(parents=True, exist_ok=True)
    
    # Validating config
    config_path = Path(".") / 'config.toml'
    if not config_path.is_file():
        console.print("Missing config file", style=f"bold {COLOR_PALETTE.get('error')}")
        raise FileNotFoundError("Missing config file")

    with open(config_path, "rb") as f:
        config_dict = tomli.load(f)

    validate_config(config_dict)
    validate_env_vars()

    operators_repository = Path(".")
    if not operators_repository.is_dir():
        console.print("Operators repository path does not exist", style=f"bold {COLOR_PALETTE.get('error')}")
        raise Exception("Operators repository path does not exist")

    if not (operators_repository / 'config.toml').is_file():
        console.print("config.toml file does not exist", style=f"bold {COLOR_PALETTE.get('error')}")
        raise Exception("config.toml file does not exist")

    if not (operators_repository / 'operators').is_dir():
        console.print("Operators directory does not exist", style=f"bold {COLOR_PALETTE.get('error')}")
        raise Exception("Operators directory does not exist")
    
    if not (operators_repository / 'dependencies').is_dir():
        console.print("Dependencies directory does not exist", style=f"bold {COLOR_PALETTE.get('error')}")
        raise Exception("Dependencies directory does not exist")

    os.environ['OPERATORS_REPOSITORY_PATH_HOST'] = str(operators_repository)


def validate_operators_folders() -> None:
    """
    Validate the Operators folders from an Operators repository.
    """
    from flowui.schemas.operator_metadata import OperatorMetadata

    operators_path = Path(".") / "operators"
    dependencies_path = Path(".") / "dependencies"
    organized_flowui_path = Path(".") / ".flowui/"
    dependencies_files = [f.name for f in dependencies_path.glob("*")]
    name_errors = list()
    missing_file_errors = list()
    missing_dependencies_errors = list()
    for op_dir in operators_path.glob("*Operator"):
        if op_dir.is_dir():
            # Validate necessary files exist
            files_names = [f.name for f in op_dir.glob("*")]
            if 'models.py' not in files_names:
                missing_file_errors.append(f"missing 'models.py' for {op_dir.name}")
            if 'operator.py' not in files_names:
                missing_file_errors.append(f"missing 'operator.py' for {op_dir.name}")
            if len(missing_file_errors) > 0:
                raise Exception('\n'.join(missing_file_errors))

            # Validate metadata
            if (op_dir / "metadata.json").is_file():
                with open(str(op_dir / "metadata.json"), "r") as f:
                    metadata = json.load(f)
                jsonschema.validate(instance=metadata, schema=OperatorMetadata.schema())

                # Validate Operators name
                if metadata.get("name", None) and not metadata["name"] == op_dir.name:
                    name_errors.append(op_dir.name)
                
                # Validate dependencies exist
                if metadata.get("dependency", None):
                    req_file = metadata["dependency"].get("requirements_file", None)
                    if req_file and req_file != "default" and req_file not in dependencies_files:
                        missing_dependencies_errors.append(f'missing dependency file {req_file} defined for {op_dir.name}')
                    
                    dock_file = metadata["dependency"].get("dockerfile", None)
                    if dock_file and dock_file != "default" and dock_file not in dependencies_files:
                        missing_dependencies_errors.append(f'missing dependency file {dock_file} defined for {op_dir.name}')

    if len(name_errors) > 0:
        raise Exception(f"The following Operators have inconsistent names: {', '.join(name_errors)}")
    if len(missing_dependencies_errors) > 0:
        raise Exception("\n" + "\n".join(missing_dependencies_errors))
    
    
def create_operators_repository(repository_name: str, dockerhub_registry: str) -> None:
    """
    Create a new Operators repository from template, with folder structure and placeholder files.
    """
    while not validate_repository_name(repository_name):
        repository_name = input("\nInvalid repository name. Should have only numbers, letters and underscores. \nEnter a new repository name: ") or f"new_repository_{str(uuid.uuid4())[0:8]}"
    cwd = Path.cwd()
    repository_folder = cwd / repository_name
    if repository_folder.is_dir():
        raise Exception("Repository folder already exists")
    console.print(f"Cloning template Operators repository at: {repository_folder}")
    subprocess.run(["git", "clone", "https://github.com/Tauffer-Consulting/flowui_operators_repository_template.git", repository_name], capture_output=True, text=True)
    shutil.rmtree(f"{repository_name}/.git")

    # Update config
    with open(f"{repository_name}/config.toml", "rb") as f:
        repo_config = tomli.load(f)

    repo_config["repository"]["REPOSITORY_NAME"] = repository_name
    repo_config["dockerhub"]["REGISTRY_NAME"] =  dockerhub_registry if dockerhub_registry else "ENTER-YOUR-DOCKERHUB-REGISTRY-NAME-HERE"

    with open(f"{repository_name}/config.toml", "wb") as f:
        tomli_w.dump(repo_config, f)

    console.print(f"Operators repository successfully create at: {repository_folder}", style=f"bold {COLOR_PALETTE.get('success')}")
    console.print("")


def create_compiled_operators_metadata() -> None:  
    """
    Create compiled metadata from Operators metadata.json files and include input_schema generated from models.py
    """
    from flowui.scripts.load_operator import load_operator_models_from_path
    from flowui.utils.metadata_default import metadata_default

    operators_path = Path(".") / "operators"
    compiled_metadata = dict()
    for op_dir in operators_path.glob("*Operator"):
        if op_dir.is_dir():
            operator_name = op_dir.name

            # Update with user-defined metadata.json
            metadata = copy.deepcopy(metadata_default)
            if (op_dir / "metadata.json").is_file():
                with open(str(op_dir / "metadata.json"), "r") as f:
                    metadata_op = json.load(f)
                dict_deep_update(metadata, metadata_op)
                metadata["name"] = operator_name

            # Add input and output schemas
            input_model_class, output_model_class, secrets_model_class = load_operator_models_from_path(
                operators_folder_path=str(operators_path), 
                operator_name=op_dir.name
            )
            metadata["input_schema"] = input_model_class.schema()
            metadata["output_schema"] = output_model_class.schema()
            metadata["secrets_schema"] = secrets_model_class.schema() if secrets_model_class else None

            # Add to compiled metadata
            compiled_metadata[operator_name] = metadata
    
    # Save compiled_metadata.json file
    organized_flowui_path = Path(".") / ".flowui/"
    with open(str(organized_flowui_path / "compiled_metadata.json"), "w") as f:
        json.dump(compiled_metadata, f, indent=4)


def create_dependencies_map(repo_name: str, save_map_as_file: bool = True) -> None:
    """
    Construct a map between Operators and unique definitions for docker images dependencies

    Args:
        repo_name (str): Name of the repository, as defined in config.toml
        save_map_as_file (bool, optional): Set if dependencies_map will be saved as file. Defaults to True.

    Raises:
        ValueError: Raise if operators is not found in the operators_repository
    """
    organized_flowui_path = Path(".") / ".flowui/"
    with open(organized_flowui_path / "compiled_metadata.json", "r") as f:
        compiled_metadata = json.load(f)

    operators_images_map = {}
    for op_i, (operator_name, operator_metadata) in enumerate(compiled_metadata.items()):
        
        if operator_metadata.get("secrets_schema"):
            operator_secrets = set(operator_metadata.get("secrets_schema")["properties"].keys())
        else:
            operator_secrets = set()

        if op_i == 0:
            operators_images_map = {
                f"{repo_name}_image_0": {
                    "dependency": operator_metadata["dependency"],
                    "operators": [operator_name],
                    "secrets": operator_secrets
                }
            }
        else:
            # Compare with metadata from previous operators to see if a new docker image needs to be built
            existing_keys = operators_images_map.keys()
            skip_new_image = False
            for i, dep_key in enumerate(existing_keys):
                if all([operator_metadata["dependency"][k] == operators_images_map[dep_key]["dependency"][k] for k in operator_metadata["dependency"].keys()]):
                    operators_images_map[dep_key]["operators"].append(operator_name)
                    operators_images_map[dep_key]["secrets"].update(operator_secrets)
                    skip_new_image = True
                    continue
            if not skip_new_image:
                operators_images_map[f"{repo_name}_image_{len(existing_keys)}"] = {
                    "dependency": operator_metadata["dependency"],
                    "operators": [operator_name],
                    "secrets": operator_secrets
                }

    if not operators_images_map:
        raise ValueError("No operators found in the Operators repository")

    if save_map_as_file:
        map_file_path = organized_flowui_path / "dependencies_map.json"
        with open(map_file_path, "w") as outfile:
            json.dump(operators_images_map, outfile, indent=4, cls=SetEncoder)


def build_docker_images(publish_images: bool) -> None:
    """
    Convenience function to build Docker images from the repository dependencies and publish them to Docker Hub
    """
    from flowui.scripts.build_docker_images_operators import build_images_from_operators_repository

    console.print("Building Docker images and generating map file...")
    updated_dependencies_map = build_images_from_operators_repository(publish=publish_images)
    return updated_dependencies_map


def organize_operators_repository(build_images: bool, publish_images: bool) -> None:
    """
    Organize Operator's repository for FlowUI. This will: 
    - validate the folder structure, and create the operators compiled_metadata.json and dependencies_map.json files
    - build Docker images for the Operators
    - publish images at Dockerhub
    """        
    # Validate repository
    console.print("Validating repository structure and files...")
    validate_repository_structure()
    validate_operators_folders()
    console.print("Validation successful!", style=f"bold {COLOR_PALETTE.get('success')}")

    # Load config
    with open("config.toml", "rb") as f:
        repo_config = tomli.load(f)
    repo_name = repo_config["repository"]["REPOSITORY_NAME"]
    
    # Create compiled metadata from Operators metadata.json files and add data input schema
    create_compiled_operators_metadata()
    
    # Generate dependencies_map.json file
    create_dependencies_map(
        repo_name=repo_name,
        save_map_as_file=True
    )
    console.print("Metadata and dependencies organized successfully!", style=f"bold {COLOR_PALETTE.get('success')}")

    # Build and publish the images
    if build_images:
        updated_dependencies_map = build_docker_images(publish_images=publish_images)
        map_file_path = Path(".") / ".flowui/dependencies_map.json"
        with open(map_file_path, "w") as outfile:
            json.dump(updated_dependencies_map, outfile, indent=4)



def create_release():
    """
    Create a new release and tag in the repository for the latest commit.
    """
    token = os.environ.get('GITHUB_TOKEN')
    client = GithubRestClient(token=token)

    # Get version from config.toml
    with open("config.toml", "rb") as f:
        repo_config = tomli.load(f)

    version = repo_config.get("repository").get("VERSION")
    if not version:
        raise ValueError("VERSION not found in config.toml")

    # https://docs.github.com/en/actions/learn-github-actions/contexts#example-printing-context-information-to-the-log
    # Passing from context to env - ${{ github.repository }} - get repository from context 
    repository = os.environ.get('GITHUB_REPOSITORY')

    # Check if tag already exists
    tag = client.get_tag(repo_name=repository, tag_name=version)
    if tag:
        raise ValueError(f'Tag {version} already exists')
    
    # Get latest commit
    latest_commit = client.get_commits(repo_name=repository, number_of_commits=1)[0]
    # Create tag
    release = client.create_release(
        repo_name=repository,
        version=version,
        tag_message=f'Release {version}',
        release_message=f'Release {version}',
        target_commitish=latest_commit.sha,
    )
    console.print(f"Release {version} created successfully!", style=f"bold {COLOR_PALETTE.get('success')}")
    return release


