"""
Provides functions to encrypt and decrypt files using AES cipher.

Tip:
    The name ``encryptor`` or something like that sounds more appropriate for
    the name of the module and the functions, but the damage is done already.

The Header
----------

The header is used to store the important bits of data that will be used to
identify and/or decrypt the encrypted file.

This is the structure of the header of an encrypted file:

+----------------------------------------+
|  Header Format (Big endian; 118 bytes) |
+========================================+
| Magic number (``I``)                   |
+----------------------------------------+
| Mode Value (``H``)                     |
+----------------------------------------+
| Nonce (``16s``)                        |
+----------------------------------------+
| Authentication Tag (``32s``)           |
+----------------------------------------+
| Metadata (``32s``)                     |
+----------------------------------------+
| Key Derivation Function Salt (``32s``) |
+----------------------------------------+

Note:
    The value in the brackets are the corresponding symbols used in ``struct``
    module.

Parts of Header
~~~~~~~~~~~~~~~

The header can be represented as a C struct:

.. code-block:: c

    typedef struct {
        unsigned int magic;
        unsigned short mode;
        char nonce[16];
        char tag[32];
        char metadata[32];
        char salt[32];
    } Header;

- Magic number (``unsigned int magic``):
    A unique number to identify the filetype.

- Mode value (``unsigned short mode``):
    The AES mode used to encrypt the file

- Nonce (``char nonce[16]``):
    The ``nonce`` or ``initialization vector`` used for the AES cipher.

- Authentication Tag (``char tag[32]``):
    The tag generated by the cipher after the encryption is over.

- Metadata (``char metadata[32]``):
    Any binary data. **Only this can be specified by the user**.
    See :func:`lockerf` for more details about the metadata.

- Key Derivation Function Salt (``char salt[32]``):
    The salt used for key derivation.
"""

import os
import struct
import typing
from collections import namedtuple
from functools import partial
from hashlib import pbkdf2_hmac

from .ciphers import exc
from .ciphers.backends import Backends as _Backends
from .ciphers.interfaces import AES
from .ciphers.modes import Modes, aead, special

# magic - mode - nonce - tag - metadata - salt
HEADER_FORMAT = struct.Struct(">I H 16s 32s 32s 32s")

MAGIC = 0xC8E52E4A

PBKDF2_HMAC = partial(pbkdf2_hmac, hash_name="sha256", iterations=150000)

METADATA = b"CREATED BY: PyFLocker"

EXTENSION = ".pyflk"

_Header = namedtuple("_Header", "magic mode nonce tag metadata salt")


def locker(
    file: os.PathLike,
    password: bytes,
    encrypting: typing.Optional[bool] = None,
    remove: bool = True,
    *,
    ext: typing.Optional[str] = None,
    newfile: typing.Optional[str] = None,
    **kwargs,
) -> None:
    """Encrypts or decrypts files with AES algorithm.

    See also:
        :func:`lockerf` function for more details.

    Args:
        file (str):
            The actual location of the file.
        password (bytes, bytearray, memoryview):
            Password to use to encrypt/decrypt the file. See :func:`lockerf`.
        encrypting (bool):
            Whether the file is being locked (encrypted) or not.

            If ``encrypting`` is True, the file is encrypted no matter what
            the extension is.
            If ``encrypting`` is False, the file is decrypted no matter what
            the extension is.

            If ``encrypting`` is None (the default), it is guessed from the
            file extension and the file header instead.

            If encrypting is provided, argument ``ext`` is ignored.
        remove (bool):
            Whether to remove the file after encryption/decryption. Default
            is True.

    Keyword Arguments:
        ext (str):
            The extension to be used for the encrypted file. If None,
            the default value ``.pyflk`` is used.
        newfile (str):
            The name of the file to be written upon. It must not be
            already present. If None is provided (default), the name
            of the ``file`` plus the extension is used.
        **kwargs:
            All other kwargs are passed to ``lockerf`` function.

    Important:
        The removal of file is **NOT** secure, because it uses
        :py:func:`os.remove` to remove the file. With enough expertise and
        time, the original file can be restored. If you want to remove the
        original file securely, consider using ``shred`` or ``srm`` or some
        other secure file deletion tools.
    """
    if newfile and ext:
        raise ValueError("newfile and ext are mutually exclusive")

    ext = ext or EXTENSION

    # guess encrypting if not provided
    if encrypting is None:
        encrypting = not file.endswith(ext)

    # make newfile name if not provided
    if newfile is None:
        if encrypting:
            newfile = file + ext
        else:
            newfile = os.path.splitext(file)[0]

    try:
        with open(file, "rb") as infile, open(newfile, "xb") as outfile:
            lockerf(infile, outfile, password, encrypting, **kwargs)
    except (TypeError, exc.DecryptionError):
        # remove invalid file
        os.remove(newfile)
        raise
    else:
        # remove the original file
        if remove:
            os.remove(file)


def lockerf(
    infile: typing.BinaryIO,
    outfile: typing.BinaryIO,
    password: bytes,
    encrypting: bool,
    *,
    kdf: typing.Callable[[bytes, bytes, int], bytes] = PBKDF2_HMAC,
    aes_mode: Modes = Modes.MODE_GCM,
    blocksize: int = 16 * 1024,
    metadata: bytes = METADATA,
    dklen: int = 32,
    backend: typing.Optional[_Backends] = None,
) -> None:
    """Utility tool for encrypting files.

    This function reads from ``infile`` in blocks, specified by ``blocksize``,
    encrypts or decrypts the data and writes to ``outfile``. By design of
    the cipher wrapper for R/W to files, no intermediate copy of data is
    made during operation.

    Operation details:

    1. Password derivation

       The ``password`` is first derived into a key with PBKDF2-HMAC with
       32 byte salt, 150000 iterations, ``sha256`` as the hash algorithm.
       See ``kdf`` for more details on key derivation functions.

    2. Cipher creation

       The cipher is created with 12 byte nonce if mode is GCM else 16
       byte nonce. The nonce is stored as a part of ``Header`` for identifying
       the file, along with other important values.

       1. Authentication

          Before the operation begins, the authentication data is passed
          to the cipher. The authentication bits are::

              magic, mode, salt, metadata, nonce

          in that order.

    3. Finalization

       After completion of the entire operation, the tag created by the
       authenticatior of the cipher is written to the file as a part of
       ``Header``. If the file is being decrypted, it is read from the
       ``Header`` for verifying the file integrity and correct decryption.

    Note:
        If the cipher mode does not support authentication, HMAC is used.
        refer to the documentation of
        :class:`pyflocker.ciphers.base.BaseAEADCipher`.

    Args:
        infile (filelike):
            The file or file-like object to read from.
        outfile (filelike):
            The file or file-like object to write into.
        password (bytes, bytearray, memoryview):
            The password to use for encrypting the files.
        encrypting (bool):
            Whether the ``infile`` is being encrypted: True if encrypting
            else False.

    Keyword Arguments:
        kdf (callable):
            The key derivation function to use for deriving keys.
            :func:`hashlib.pbkdf2_hmac` is used with 150000 iterations and
            ``sha256`` as the hash algorithm.

            If a custom ``kdf`` is used, the ``kdf`` must accept 3 arguments,
            ``password``, ``salt`` and ``dklen``. It is assumed that the
            other required values are already passed to it. You can use a
            partial function (:py:func:`functools.partial`) for that purpose.
        aes_mode (:class:`~pyflocker.ciphers.modes.Modes`):
            The AES mode to use for encryption/decryption.
            The mode can be any attribute from :any:`Modes` except those
            which are defined in :obj:`pyflocker.ciphers.modes.special`.
            Defaults to :any:`Modes.MODE_GCM`.

            Specifying this value while decrypting has no effect.
        blocksize (int):
            The amount of data to read from ``infile`` in each iteration.
            Defalts to 16384.
        metadata (bytes, bytearray, memoryview):
            The metadata to write to the file. It must be up-to 32 bytes.
        dklen (int):
            The desired key length (in bytes) for passing to the cipher.
            It specifies the strength of AES cipher. Defaults to 32.
        backend (:any:`Backends`):
            The backend to use to instantiate the AES cipher from.
            If None is specified (the default), any available backend
            will be used.

    Returns:
        None

    Raises:
        DecryptionError: if password was invalid or the file was tampered
            with.
        NotImplementedError: if the mode is not supported.
        ValueError:
            If ``infile`` and ``outfile`` are same or key length is invalid.
        OverflowError: If the length of metadata exceeds 32 bytes.
        TypeError:
            It is raised if the header is invalid, or the metadata or magic
            number does not match.
    """
    if os.path.samefile(infile.fileno(), outfile.fileno()):
        raise ValueError("infile and outfile cannot be the same")

    # set defaults
    if aes_mode in special:
        raise NotImplementedError(f"{aes_mode} is not supported.")

    if len(metadata) > 32:
        raise OverflowError("maximum metadata length exceeded (limit: 32).")

    if not encrypting:
        header = _get_header(infile.read(HEADER_FORMAT.size), metadata)
    else:
        salt = os.urandom(32)
        nonce = os.urandom(12) if aes_mode == AES.MODE_GCM else os.urandom(16)
        header = _Header(MAGIC, aes_mode.value, nonce, b"", metadata, salt)
        outfile.write(HEADER_FORMAT.pack(*header))

    cipher = AES.new(
        encrypting,
        kdf(
            password=password,
            salt=header.salt,
            dklen=_check_key_length(dklen),
        ),
        Modes(header.mode),
        header.nonce,
        file=infile,
        backend=backend,
        tag_length=None,
    )
    cipher.authenticate(
        struct.pack(
            ">I H 32s 32s 16s",
            header.magic,  # XXX: MAGIC works just fine; good to be an idiot
            header.mode,
            header.salt,
            header.metadata,
            header.nonce,
        )
    )
    cipher.update_into(outfile, blocksize=blocksize, tag=header.tag)

    if encrypting:
        outfile.seek(struct.calcsize(">I H 16s"))
        outfile.write(cipher.calculate_tag())


def extract_header_from_file(
    path: os.PathLike,
    metadata: bytes = METADATA,
) -> _Header:
    """Extract the header from the file ``file``.

    Args:
        path (pathlike): The path to the encrypted file.
        metadata (bytes):
            The metadata written to the file as a part
            of the header.

    Returns:
        _Header: The header data.
    """
    with open(path, "rb") as file:
        return _get_header(file.read(HEADER_FORMAT.size), metadata)


def _check_key_length(n):
    if n in (128, 192, 256):
        return n // 8
    elif n in (16, 24, 32):
        return n
    else:
        raise ValueError("invalid key length")


def _get_header(data, metadata=METADATA):
    try:
        (
            magic,
            mode,
            nonce,
            tag,
            metadata_h,
            salt,
        ) = HEADER_FORMAT.unpack(data)
    except struct.error as e:
        raise TypeError("The file format is invalid (Header mismatch).") from e

    if magic != MAGIC or metadata != metadata_h[: len(metadata) - 32]:
        raise TypeError(
            "The file format is invalid (Metadata/magic number mismatch)."
        )

    if mode == Modes.MODE_GCM.value:
        nonce = nonce[:12]
    if Modes(mode) in aead:
        tag = tag[:16]
    return _Header(magic, mode, nonce, tag, metadata, salt)
