"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // create the cluster with a default nodegroup capacity
        const cluster = new eks.Cluster(this, 'Cluster', {
            mastersRole,
            defaultCapacity: 2,
            version: '1.16',
        });
        // // fargate profile for resources in the "default" namespace
        cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }],
        });
        // add some capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
        // add bottlerocket nodes
        cluster.addCapacity('BottlerocketNodes', {
            instanceType: new ec2.InstanceType('t3.small'),
            minCapacity: 2,
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
        // spot instances (up to 10)
        cluster.addCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5,
            },
        });
        // add a extra nodegroup
        cluster.addNodegroup('extra-ng', {
            instanceType: new ec2.InstanceType('t3.small'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: cluster.defaultCapacity ? cluster.defaultCapacity.role : undefined,
        });
        // // apply a kubernetes manifest
        cluster.addResource('HelloApp', ...hello.resources);
        // // add two Helm charts to the cluster. This will be the Kubernetes dashboard and the Nginx Ingress Controller
        cluster.addChart('dashboard', { chart: 'kubernetes-dashboard', repository: 'https://kubernetes-charts.storage.googleapis.com' });
        cluster.addChart('nginx-ingress', { chart: 'nginx-ingress', repository: 'https://helm.nginx.com/stable', namespace: 'kube-system' });
        // add a service account connected to a IAM role
        cluster.addServiceAccount('MyServiceAccount');
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterName', { value: cluster.clusterName });
    }
}
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
app.synth();
//# sourceMappingURL=data:application/json;base64,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