"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MainPipeline = void 0;
const targets = require("aws-cdk-lib/aws-events-targets");
const constructs_1 = require("constructs");
const customNodejsFunction_1 = require("./customNodejsFunction");
const path = require("path");
const notificationsTopic_1 = require("./notificationsTopic");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const lodash_1 = require("lodash");
const context_1 = require("../util/context");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appStage_1 = require("./appStage");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
class MainPipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const source = pipelines_1.CodePipelineSource.codeCommit(props.codeCommitRepository, props.repository.defaultBranch);
        const pipeline = new pipelines_1.CodePipeline(this, 'Pipeline', lodash_1.merge({
            pipelineName: aws_cdk_lib_1.Stack.of(this).stackName,
            synth: new pipelines_1.ShellStep('Synth', {
                input: source,
                installCommands: [
                    ...(props.commands.preInstall || []),
                    ...(props.commands.install || []),
                ],
                commands: [
                    ...(props.commands.buildAndTest || []),
                    ...props.commands.synthPipeline,
                ],
                primaryOutputDirectory: props.cdkOutputDirectory,
            }),
            crossAccountKeys: true,
            codeBuildDefaults: props.codeBuild,
        }, props.codePipeline));
        props.pipeline.forEach(step => {
            if (this.isWave(step)) {
                this.addWaveDeployment(pipeline, step, props.stacks);
            }
            else {
                this.addEnvironmentDeployment(pipeline, step, props.stacks);
            }
        });
        pipeline.buildPipeline();
        this.createPipelineBuildNotifications(pipeline, props.repository, props.repositoryTokenParam);
        this.failuresTopic = this.createPipelineFailuresTopic(pipeline);
    }
    isWave(waveOrEnvironment) {
        return 'wave' in waveOrEnvironment;
    }
    addWaveDeployment(pipeline, step, stacks) {
        const wave = pipeline.addWave(step.wave);
        step.environments.forEach(env => {
            const environmentDeployment = {
                ...env,
                pre: [
                    ...(step.preEachEnvironment || []),
                    ...(env.pre || []),
                ],
                post: [
                    ...(env.post || []),
                    ...(step.postEachEnvironment || []),
                ],
            };
            this.addEnvironmentDeployment(wave, environmentDeployment, stacks, `Wave${capitalizeFirstLetter(step.wave)}`, { WAVE_NAME: step.wave });
        });
        if (step.pre && step.pre.length > 0) {
            wave.addPre(new pipelines_1.ShellStep(`PreWave${capitalizeFirstLetter(step.wave)}`, {
                env: { WAVE_NAME: step.wave },
                commands: step.pre,
            }));
        }
        if (step.post && step.post.length > 0) {
            wave.addPost(new pipelines_1.ShellStep(`PostWave${capitalizeFirstLetter(step.wave)}`, {
                env: { WAVE_NAME: step.wave },
                commands: step.post,
            }));
        }
    }
    addEnvironmentDeployment(parent, step, stacks, idPrefix = '', envVariables) {
        const stage = parent.addStage(new appStage_1.AppStage(this, `${idPrefix}DeployEnv${capitalizeFirstLetter(step.environment)}`, {
            envName: step.environment,
            env: context_1.getEnvironmentConfig(this, step.environment),
            stacks,
        }));
        if (step.pre && step.pre.length > 0) {
            stage.addPre(new pipelines_1.ShellStep(`${idPrefix}PreEnv${capitalizeFirstLetter(step.environment)}`, {
                env: {
                    ...envVariables,
                    ENV_NAME: step.environment,
                },
                commands: step.pre,
            }));
        }
        if (step.post && step.post.length > 0) {
            stage.addPost(new pipelines_1.ShellStep(`${idPrefix}PostEnv${capitalizeFirstLetter(step.environment)}`, {
                env: {
                    ...envVariables,
                    ENV_NAME: step.environment,
                },
                commands: step.post,
            }));
        }
    }
    createPipelineFailuresTopic(pipeline) {
        const failuresTopic = new notificationsTopic_1.NotificationsTopic(this, 'PipelineFailuresTopic', {
            projectName: context_1.getProjectName(this),
            notificationName: 'pipelineFailures',
        });
        // for better visibility, use EventBridge Rules instead of CodeStar Notifications that are generated with pipeline.notifyOn()
        pipeline.pipeline.onStateChange('OnPipelineFailure', {
            eventPattern: {
                detail: {
                    state: ['FAILED'],
                },
            },
            target: new targets.SnsTopic(failuresTopic.topic),
        });
        return failuresTopic.topic;
    }
    /**
     * To send CodePipeline build status back to repository:
     * - trigger Lambda function on CodePipeline state change events,
     * - in Lambda:
     *   - get CodePipeline execution details to get commit SHA,
     *   - send custom event to EventBridge including the commit SHA,
     * - use EventBridge to send build status to repository.
     */
    createPipelineBuildNotifications(pipeline, repository, repoTokenParam) {
        const pipelineBuildStatusFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'PipelineBuildStatus', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'pipelineBuildStatus')),
            environment: {
                REPOSITORY_HOST: repository.host,
                REPOSITORY_NAME: repository.name,
                REPOSITORY_TOKEN_PARAM_NAME: repoTokenParam.parameterName,
            },
        });
        repoTokenParam.grantRead(pipelineBuildStatusFunction);
        pipelineBuildStatusFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['codepipeline:GetPipelineExecution'],
            resources: [`arn:aws:codepipeline:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${pipeline.pipeline.pipelineName}`],
        }));
        pipeline.pipeline.onStateChange('OnPipelineStateChange', {
            target: new targets.LambdaFunction(pipelineBuildStatusFunction),
        });
    }
}
exports.MainPipeline = MainPipeline;
const capitalizeFirstLetter = (str) => str.charAt(0).toUpperCase() + str.slice(1);
//# sourceMappingURL=data:application/json;base64,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