"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FeatureBranchBuilds = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const constructs_1 = require("constructs");
const customNodejsFunction_1 = require("./customNodejsFunction");
const path = require("path");
const notificationsTopic_1 = require("./notificationsTopic");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const context_1 = require("../util/context");
class FeatureBranchBuilds extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const source = aws_codebuild_1.Source.codeCommit({ repository: props.codeCommitRepository });
        const deployProject = this.createDeployProject(source, props.codeBuild, props.commands, props.codeCommitRepository, props.repository.defaultBranch);
        this.createDeployNotifications(deployProject, props.repository, props.repositoryTokenParam);
        this.failuresTopic = this.createBuildFailuresTopic(deployProject);
        this.createDestroyProject(source, props.codeBuild, props.commands, props.codeCommitRepository, props.repository.defaultBranch);
    }
    createDeployProject(source, codeBuild, commands, repository, defaultBranch) {
        const deployProject = new aws_codebuild_1.Project(this, 'DeployProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-featureBranch-deploy`,
            source,
            timeout: codeBuild.timeout,
            environment: codeBuild.buildEnvironment,
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'ENV_NAME=$(echo ${BRANCH_NAME} | awk \'{print tolower($0)}\')',
                            ...(commands.preInstall || []),
                            ...(commands.install || []),
                        ],
                    },
                    build: {
                        commands: [
                            ...(commands.buildAndTest || []),
                            ...commands.deployEnvironment,
                        ],
                    },
                },
            }),
        });
        this.grantAssumeCDKRoles(deployProject);
        repository.onCommit('OnBranchCommit', this.createProjectTriggerOptions(deployProject, defaultBranch, true));
        return deployProject;
    }
    createDeployNotifications(deployProject, repository, repoTokenParam) {
        const deployStatusFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'DeployStatus', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'featureBranchDeployStatus')),
            environment: {
                REPOSITORY_HOST: repository.host,
                REPOSITORY_NAME: repository.name,
                REPOSITORY_TOKEN_PARAM_NAME: repoTokenParam.parameterName,
            },
        });
        repoTokenParam.grantRead(deployStatusFunction);
        deployProject.onStateChange('OnDeployStateChange', {
            target: new targets.LambdaFunction(deployStatusFunction),
        });
    }
    createBuildFailuresTopic(deployProject) {
        const failuresTopic = new notificationsTopic_1.NotificationsTopic(this, 'FeatureBranchBuildFailuresTopic', {
            projectName: context_1.getProjectName(this),
            notificationName: 'featureBranchBuildFailures',
        });
        // for better visibility, use EventBridge Rules instead of CodeStar Notifications that are generated with project.notifyOnBuildFailed()
        deployProject.onBuildFailed('OnFeatureBuildFailure', {
            target: new targets.SnsTopic(failuresTopic.topic),
        });
        return failuresTopic.topic;
    }
    createDestroyProject(source, codeBuild, commands, repository, defaultBranch) {
        const destroyProject = new aws_codebuild_1.Project(this, 'DestroyProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-featureBranch-destroy`,
            source,
            timeout: codeBuild.timeout,
            environment: codeBuild.buildEnvironment,
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'ENV_NAME=$(echo ${BRANCH_NAME} | awk \'{print tolower($0)}\')',
                            ...(commands.preInstall || []),
                            ...(commands.install || []),
                        ],
                    },
                    build: {
                        commands: [
                            ...commands.destroyEnvironment,
                        ],
                    },
                },
            }),
        });
        this.grantAssumeCDKRoles(destroyProject);
        repository.onReferenceDeleted('OnBranchRemoval', this.createProjectTriggerOptions(destroyProject, defaultBranch));
        return destroyProject;
    }
    createProjectTriggerOptions(targetProject, defaultBranch, withSourceVersion = false) {
        return {
            eventPattern: {
                detail: {
                    referenceType: ['branch'],
                    referenceName: [
                        { 'anything-but': [defaultBranch] },
                    ],
                    referenceFullName: [
                        { 'anything-but': { prefix: 'refs/remotes/' } },
                    ],
                },
            },
            target: new aws_events_targets_1.CodeBuildProject(targetProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    sourceVersion: withSourceVersion ? aws_events_1.EventField.fromPath('$.detail.commitId') : undefined,
                    environmentVariablesOverride: [
                        {
                            name: 'BRANCH_NAME',
                            value: aws_events_1.EventField.fromPath('$.detail.referenceName'),
                            type: 'PLAINTEXT',
                        },
                    ],
                }),
            }),
        };
    }
    grantAssumeCDKRoles(project) {
        const qualifier = aws_cdk_lib_1.Stack.of(this).synthesizer.bootstrapQualifier || 'hnb659fds';
        project.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-deploy-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-file-publishing-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-image-publishing-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-lookup-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
            ],
        }));
    }
}
exports.FeatureBranchBuilds = FeatureBranchBuilds;
//# sourceMappingURL=data:application/json;base64,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