"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CIStack = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lodash_1 = require("lodash");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const featureBranchBuilds_1 = require("./constructs/featureBranchBuilds");
const mainPipeline_1 = require("./constructs/mainPipeline");
const types_1 = require("./util/types");
const applicationProps_1 = require("./applicationProps");
const aws_chatbot_1 = require("aws-cdk-lib/aws-chatbot");
const notificationsTopic_1 = require("./constructs/notificationsTopic");
const context_1 = require("./util/context");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const mirrorRepository_1 = require("./constructs/mirrorRepository");
const defaultCommands = {
    npm: {
        install: [
            'npm install --location=global aws-cdk@2',
            'npm ci',
        ],
    },
    pnpm: {
        install: [
            'npm install --location=global aws-cdk@2 pnpm',
            'pnpm install --frozen-lockfile',
        ],
    },
};
class CIStack extends aws_cdk_lib_1.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const resolvedProps = this.resolveProps(props);
        const projectName = context_1.getProjectName(this);
        const repositoryTokenParam = aws_ssm_1.StringParameter.fromSecureStringParameterAttributes(this, 'RepositoryTokenParam', {
            parameterName: `/${context_1.getProjectName(this)}/ci/repositoryAccessToken`,
        });
        const mirror = new mirrorRepository_1.MirrorRepository(this, 'MirrorRepository', {
            repoTokenParam: repositoryTokenParam,
            repository: resolvedProps.repository,
        });
        const mainPipeline = new mainPipeline_1.MainPipeline(this, 'MainPipeline', {
            ...resolvedProps,
            codeCommitRepository: mirror.codeCommitRepository,
            repositoryTokenParam,
        });
        const featureBranchBuilds = new featureBranchBuilds_1.FeatureBranchBuilds(this, 'FeatureBranchBuilds', {
            ...resolvedProps,
            codeCommitRepository: mirror.codeCommitRepository,
            repositoryTokenParam,
        });
        if (resolvedProps.slackNotifications.workspaceId && resolvedProps.slackNotifications.channelId) {
            this.createSlackNotifications(projectName, resolvedProps.slackNotifications, mainPipeline.failuresTopic, featureBranchBuilds.failuresTopic);
        }
    }
    resolveProps(props) {
        if (props.packageManager) {
            lodash_1.merge(applicationProps_1.defaultProps, { commands: defaultCommands[props.packageManager] });
        }
        return lodash_1.defaultsDeep(lodash_1.cloneDeep(props), applicationProps_1.defaultProps);
    }
    createSlackNotifications(projectName, config, mainPipelineFailuresTopic, featureBranchFailuresTopic) {
        const alarmsTopic = new notificationsTopic_1.NotificationsTopic(this, 'SlackAlarmsTopic', {
            projectName: projectName,
            notificationName: 'slackAlarms',
        });
        const slack = new aws_chatbot_1.SlackChannelConfiguration(this, 'SlackChannelConfiguration', {
            slackChannelConfigurationName: this.stackName,
            slackWorkspaceId: config.workspaceId,
            slackChannelId: config.channelId,
            notificationTopics: [
                alarmsTopic.topic,
                config.mainPipelineFailures ? mainPipelineFailuresTopic : undefined,
                config.featureBranchFailures ? featureBranchFailuresTopic : undefined,
            ].filter(types_1.notEmpty),
        });
        slack.role?.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchReadOnlyAccess'));
    }
}
exports.CIStack = CIStack;
//# sourceMappingURL=data:application/json;base64,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