import { Duration } from 'aws-cdk-lib';
import { ComputeType } from 'aws-cdk-lib/aws-codebuild';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { CodeBuildOptions, DockerCredential } from 'aws-cdk-lib/pipelines';
import { Construct } from 'constructs';
export interface ApplicationProps {
    /**
     * An object with a create() method to create Stacks for the application.
     *
     * The same Stacks will be deployed with main pipeline, feature-branch builds, and local deployments.
     */
    readonly stacks: IStacksCreation;
    readonly repository: RepositoryProps;
    /**
     * Package manager used in the repository.
     * If provided, the install commands will be set to install dependencies using given package manager.
     */
    readonly packageManager?: 'npm' | 'pnpm';
    /**
     * Commands executed to build and deploy the application.
     */
    readonly commands?: BuildCommands;
    /**
     * The location where CDK outputs synthetized files.
     * Corresponds to the CDK Pipelines ShellStepProps#primaryOutputDirectory.
     * @default cdk.out
     */
    readonly cdkOutputDirectory?: string;
    /**
     * CodePipeline deployment pipeline for the main repository branch.
     *
     * Can contain environments to deploy
     * and waves that deploy multiple environments in parallel.
     *
     * Each environment and wave can have pre and post commands
     * that will be executed before and after the environment or wave deployment.
     */
    readonly pipeline: (WaveDeployment | EnvironmentDeployment)[];
    /**
     * Override CodeBuild properties, used for the main pipeline
     * as well as feature branch ephemeral environments deploys and destroys.
     *
     * @default 30 minutes timeout, compute type MEDIUM with Linux build image Standard 6.0
     */
    readonly codeBuild?: CodeBuildOptions;
    /**
     * Override CodePipeline properties.
     */
    readonly codePipeline?: CodePipelineOverrides;
    /**
     * Configuration for Slack notifications.
     * Requires configuring AWS Chatbot client manually first.
     */
    readonly slackNotifications?: SlackNotifications;
}
/**
 * To provide a method as parameter, jsii requires creating a behavioral interface, prefixed with "I".
 * Mixing structural and behavioral interfaces is not always possible, hence we extract stacks creation
 * to a separate object described by this behavioral interface.
 */
export interface IStacksCreation {
    /**
     * Create Stacks for the application.
     *
     * Use provided scope as stacks' parent (first constructor argument).
     *
     * Stacks must include provided environment name in their names
     * to distinguish them when deploying multiple environments
     * (like feature-branch environments) to the same account.
     */
    create(scope: Construct, projectName: string, envName: string): void;
}
export interface RepositoryProps {
    /**
     * Repository hosting.
     */
    readonly host: 'github' | 'bitbucket';
    /** Like "my-comapny/my-repo". */
    readonly name: string;
    /**
     * Branch to deploy the environments from in the main pipeline.
     * @default main
     */
    readonly defaultBranch?: string;
}
export interface BuildCommands {
    readonly preInstall?: string[];
    readonly install?: string[];
    readonly buildAndTest?: string[];
    readonly synthPipeline?: string[];
    readonly deployEnvironment?: string[];
    readonly destroyEnvironment?: string[];
}
export interface WaveDeployment {
    /**
     * Wave name.
     */
    readonly wave: string;
    /**
     * List of environments to deploy in parallel.
     */
    readonly environments: EnvironmentDeployment[];
    /**
     * Commands to execute before the wave deployment.
     */
    readonly pre?: string[];
    /**
     * Commands to execute after the wave deployment.
     */
    readonly post?: string[];
    /**
     * Commands to execute before each environment deployment.
     *
     * If environment configuration also contains commands to execute pre-deployment,
     * they will be executed after the commands defined here.
     */
    readonly preEachEnvironment?: string[];
    /**
     * Commands to execute after environment deployment.
     *
     * If environment configuration also contains commands to execute post-deployment,
     * they will be executed before the commands defined here.
     */
    readonly postEachEnvironment?: string[];
}
export interface EnvironmentDeployment {
    /**
     * Environment name.
     *
     * Environment will be deployed to AWS account and region
     * defined in cdk.json file `context/environments` properties,
     * falling back to the `default` environment settings if given environment configuration is not found.
     */
    readonly environment: string;
    /**
     * Commands to execute before the environment deployment.
     */
    readonly pre?: string[];
    /**
     * Commands to execute after the environment deployment.
     */
    readonly post?: string[];
}
/**
 * Since jsii does not support Partial or Omit,
 * we have to define all properties from CodePipelineProps that may be overriden manually.
 */
export interface CodePipelineOverrides {
    readonly pipelineName?: string;
    readonly selfMutation?: boolean;
    readonly dockerEnabledForSelfMutation?: boolean;
    readonly dockerEnabledForSynth?: boolean;
    readonly codeBuildDefaults?: CodeBuildOptions;
    readonly synthCodeBuildDefaults?: CodeBuildOptions;
    readonly assetPublishingCodeBuildDefaults?: CodeBuildOptions;
    readonly selfMutationCodeBuildDefaults?: CodeBuildOptions;
    readonly publishAssetsInParallel?: boolean;
    readonly dockerCredentials?: DockerCredential[];
    readonly reuseCrossRegionSupportStacks?: boolean;
    readonly role?: IRole;
    readonly useChangeSets?: boolean;
    readonly enableKeyRotation?: boolean;
}
export interface SlackNotifications {
    readonly workspaceId: string;
    readonly channelId: string;
    /**
     * Whether to notify about main pipeline failures.
     * @default true
     */
    readonly mainPipelineFailures?: boolean;
    /**
     * Whether to notify about feature branch deployment failures.
     * @default false
     */
    readonly featureBranchFailures?: boolean;
}
export declare const defaultProps: {
    repository: {
        defaultBranch: string;
    };
    commands: {
        install: string[];
        synthPipeline: string[];
        deployEnvironment: string[];
        destroyEnvironment: string[];
    };
    codeBuild: {
        timeout: Duration;
        buildEnvironment: {
            computeType: ComputeType;
            buildImage: import("aws-cdk-lib/aws-codebuild").IBuildImage;
        };
    };
    slackNotifications: {
        mainPipelineFailures: boolean;
    };
};
export declare type ResolvedApplicationProps = ApplicationProps & typeof defaultProps;
