# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['pyinterboleto',
 'pyinterboleto.common',
 'pyinterboleto.consulta',
 'pyinterboleto.emissao',
 'pyinterboleto.utils']

package_data = \
{'': ['*']}

install_requires = \
['Unidecode>=1.2.0,<2.0.0',
 'dacite>=1.6.0,<2.0.0',
 'requests>=2.26.0,<3.0.0',
 'tabulate>=0.8.9,<0.9.0']

setup_kwargs = {
    'name': 'pyinterboleto',
    'version': '1.0.1',
    'description': 'Biblioteca para facilitar o manuseio de boletos de contas PJ no Banco Inter.',
    'long_description': '# PyInterBoleto\nBiblioteca para facilitar o manuseio de boletos de contas PJ no Banco Inter.\n\n[![PyPI version](https://badge.fury.io/py/pyinterboleto.svg)](https://badge.fury.io/py/pyinterboleto) [![codecov](https://codecov.io/gh/feslima/pyinterboleto/branch/main/graph/badge.svg?token=T2SJ0P8KPG)](https://codecov.io/gh/feslima/pyinterboleto)\n\n***\n- [PyInterBoleto](#pyinterboleto)\n- [Instalação](#instalação)\n- [Documentação](#documentação)\n- [Usagem básica](#usagem-básica)\n  - [Configuração de autenticação](#configuração-de-autenticação)\n  - [Emissão de boleto](#emissão-de-boleto)\n  - [Consultas](#consultas)\n    - [Consulta detalhada de boletos individuais](#consulta-detalhada-de-boletos-individuais)\n    - [Consulta de coleção de boletos](#consulta-de-coleção-de-boletos)\n    - [Resgate de PDFs de boletos individuais](#resgate-de-pdfs-de-boletos-individuais)\n  - [Baixa de boleto](#baixa-de-boleto)\n- [Testagem](#testagem)\n\n# Instalação\nBasta executar o comando via pip:\n```\npip install pyinterboleto\n```\n\n# Documentação\nA maioria das classes, métodos, funções e estruturas de dados já contém uma documentação habilitada para uso em IDEs no estilo [numpy docstring](https://numpydoc.readthedocs.io/en/latest/format.html).\n\nFoi optado por não fazer uma documentação exclusiva (no estilo readthedocs) devido a ser uma biblioteca relativamente pequena.\n\nSendo assim, o pacote está organizado em três submódulos principais: **emissão** (os dados necessários na emissão são organizados em estruturas menores. São elas dados de: [emissão](src/pyinterboleto/emissao/emissor.py), [pagador](src/pyinterboleto/emissao/pagador.py), [desconto](src/pyinterboleto/emissao/desconto.py), [multa](src/pyinterboleto/emissao/multa.py), [mora](src/pyinterboleto/emissao/mora.py) e [mensagem](src/pyinterboleto/emissao/mensagem.py)), **consulta** ([detalhada](src/pyinterboleto/consulta/detalhado.py), [coleção](src/pyinterboleto/consulta/lista.py) e [PDF](src/pyinterboleto/consulta/pdf.py)) e [**baixa**](src/pyinterboleto/baixa/__init__.py) de boletos.\n\nEm cada um desses links se encontram as funções e objetos com suas respectivas documentações, caso seja preciso mais informações.\n# Usagem básica\nA classe principal que tem todas as funcionalidades requeridas para a API se chama [**`Boleto`**](src/pyinterboleto/boleto.py). Através dela que todas as operações de emissão, consulta e baixa feitas.\n\nNo entanto, como consta na documentação do Banco Inter, para se ter acesso a API é preciso emitir a chave e o certificado de acesso desta. Antes de utilizar este pacote, certifique-se que você já possui estes arquivos.\n\nFeito isto, alguns exemplos de manuseio são mostrados nas seções à seguir.\n***\n\n## Configuração de autenticação\nAntes de fazer qualquer requisição à API do Inter é preciso antes definir o [objeto de configuração](src/pyinterboleto/utils/requests.py) para autenticação e autorização:\n\n```python\n>>> from pathlib import Path\n>>> from datetime import date, timedelta\n>>> from prettyprinter import pprint, install_extras\n>>> from pyinterboleto import RequestConfigs\n>>>\n>>> install_extras()\n>>>\n>>> # definição da configuração de autenticação\n>>> direc = Path(\'caminho/para/pasta/com/certificados\')\n>>> cert = direc / \'Inter API_Certificado.crt\'\n>>> key = direc / \'Inter API_Chave.key\'\n>>> # client_id e client_secret são obtidos de acordo com a documentação do Inter\n>>> client_id = \'valor-do-id-uuid\'\n>>> client_secret = \'valor-do-secret-uuid\'\n>>> scopes = (ScopeEnum.BOLETO_COBRANCA_WRITE,)\n>>> configs = RequestConfigs(client_id=client_id, client_secret=client_secret, scopes=scopes, certificate=cert, key=key)\n```\n\n## Emissão de boleto\n_*Os dados a seguir são fictícios. Não os utilize do jeito que estão!*_\n\n```python\n>>> from pyinterboleto import Boleto, Emissao, Pagador, Beneficiario\n>>> boleto = Boleto(configs) # configs vem da seção configuração\n>>>\n>>> pagador = Pagador(\n...     cpfCnpj="12.345.678/0001-12",\n...     tipoPessoa=TipoPessoa.JURIDICA,\n...     nome="Alguma Empresa LTDA",\n...     endereco="Qulaquer um",\n...     cidade="Também do Brasil",\n...     uf="SP",\n...     cep="12345-678",\n... )\n>>> beneficiario = Beneficiario(\n...     cpfCnpj="123.456.789-01",\n...     tipoPessoa=TipoPessoa.FISICA,\n...     nome="Algum Nome de Pessoa",\n...     endereco="Algum lugar",\n...     bairro="Qualquer",\n...     cidade="Do Brasil",\n...     uf="SP",\n...     cep="12345-678",\n... )\n>>> dados = Emissao(\n...     pagador=pagador,\n...     beneficiario=beneficiario,\n...     seuNumero="000001",\n...     valorNominal=10.01,\n...     dataVencimento="2023-01-01",\n...     numDiasAgenda=25,\n... )\n>>> result = boleto.emitir(emissao)\n>>> print(result)\n{\'seuNumero\': \'00001\', \'nossoNumero\': \'00123456789\',\n \'codigoBarras\': \'00000000000000000000000000000000000000000000\',\n \'linhaDigitavel\': \'00000000000000000000000000000000000000000000000\'}\n>>>\n```\n\n## Consultas\nHá três categorias de consultas disponíveis: detalhamento individual de boletos, coleção de boletos e resgate de arquivos .pdf.\n\n### Consulta detalhada de boletos individuais\nÉ preciso saber o número de identificação do título antes de fazer esta requisição. Este número pode ser obtido quando a emissão do título é bem sucedida (chave `nossoNumero` do dicionário de resposta na seção anterior)\nou quando se faz a filtragem de coleções de boletos.\n\n```python\n>>> boleto = Boleto(configs)\n>>> num_boleto = \'00123456789\' # numero de identificação do título pelo Inter\n>>> detail = boleto.consulta_detalhada(num_boleto)\n>>> pprint(detail)\npyinterboleto.consulta.detalhado.BoletoDetail(\n    nomeBeneficiario=\'NOME DO BENEFICIARIO CONTA PJ\',\n    cnpjCpfBeneficiario=\'00000000000000\',\n    tipoPessoaBeneficiario=\'JURIDICA\',\n    dataHoraSituacao=datetime.datetime(2021, 5, 10),\n    codigoBarras=\'00000000000000000000000000000000000000000000\',\n    linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n    dataVencimento=datetime.date(2021, 5, 11),\n    dataEmissao=datetime.date(2021, 5, 9),\n    seuNumero=\'00001\',\n    valorNominal=0.01,\n    nomePagador=\'Pessoa Ficticia da Silva\',\n    emailPagador=\'\',\n    dddPagador=\'\',\n    telefonePagador=\'\',\n    tipoPessoaPagador=\'FISICA\',\n    cnpjCpfPagador=\'12345678909\',\n    codigoEspecie=\'OUTROS\',\n    dataLimitePagamento=datetime.date(2021, 6, 10),\n    valorAbatimento=0.0,\n    situacao=\'PAGO\',\n    desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n        codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n    ),\n    desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n        codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n    ),\n    desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n        codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n    ),\n    multa=pyinterboleto.common.multa.MultaConsulta(\n        codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n    ),\n    mora=pyinterboleto.common.mora.MoraConsulta(\n        codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n    ),\n    situacaoPagamento=\'BAIXADO\',\n    valorTotalRecebimento=0.01\n)\n>>>\n```\n\n### Consulta de coleção de boletos\nAs datas de início e final da filtragem são obrigatórias, [há outras definições de elementos de filtragem opcionais](src/pyinterboleto/consulta/lista.py).\n\n```python\n>>> from datetime import date, timedelta\n>>> boleto = Boleto(configs)\n>>> inicial = date.today() - timedelta(days=30)\n>>> final = date.today()\n>>> lista = boleto.consulta_lista(inicial, final)\n>>> pprint(lista)\npyinterboleto.consulta.lista.ResponseList(\n    totalPages=1,\n    totalElements=8,\n    numberOfElements=8,\n    last=True,\n    first=True,\n    size=20,\n    summary=pyinterboleto.consulta.lista.Summary(\n        recebidos=pyinterboleto.consulta.lista.SummaryContent(\n            quantidade=1,\n            valor=0.01\n        ),\n        previstos=pyinterboleto.consulta.lista.SummaryContent(\n            quantidade=2,\n            valor=66.2\n        ),\n        baixados=pyinterboleto.consulta.lista.SummaryContent(\n            quantidade=2,\n            valor=0.02\n        ),\n        expirados=pyinterboleto.consulta.lista.SummaryContent(\n            quantidade=3,\n            valor=38.01\n        )\n    ),\n    content=[\n        pyinterboleto.consulta.lista.BoletoItem(\n            nossoNumero=\'00000000000\',\n            seuNumero=\'00001\',\n            cnpjCpfSacado=\'1234567809\',\n            nomeSacado=\'Pessoa Ficticia da Silva\',\n            situacao=\'PAGO\',\n            dataVencimento=datetime.date(2021, 5, 11),\n            valorNominal=0.01,\n            email=\'\',\n            telefone=\'\',\n            dataEmissao=datetime.date(2021, 5, 9),\n            dataLimite=datetime.date(2021, 6, 10),\n            linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n            desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            multa=pyinterboleto.common.multa.MultaConsulta(\n                codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n            ),\n            mora=pyinterboleto.common.mora.MoraConsulta(\n                codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n            ),\n            valorAbatimento=0.0,\n            dataPagtoBaixa=datetime.date(2021, 5, 10),\n            valorTotalRecebimento=0.01\n        ),\n        pyinterboleto.consulta.lista.BoletoItem(\n            nossoNumero=\'00000000000\',\n            seuNumero=\'00002\',\n            cnpjCpfSacado=\'1234567809\',\n            nomeSacado=\'Pessoa Ficticia da Silva\',\n            situacao=\'EXPIRADO\',\n            dataVencimento=datetime.date(2021, 5, 12),\n            valorNominal=0.01,\n            email=\'\',\n            telefone=\'\',\n            dataEmissao=datetime.date(2021, 5, 10),\n            dataLimite=datetime.date(2021, 6, 11),\n            linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n            desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            multa=pyinterboleto.common.multa.MultaConsulta(\n                codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n            ),\n            mora=pyinterboleto.common.mora.MoraConsulta(\n                codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n            ),\n            valorAbatimento=0.0\n        ),\n        pyinterboleto.consulta.lista.BoletoItem(\n            nossoNumero=\'00000000000\',\n            seuNumero=\'00003\',\n            cnpjCpfSacado=\'1234567809\',\n            nomeSacado=\'Pessoa Ficticia da Silva\',\n            situacao=\'BAIXADO\',\n            dataVencimento=datetime.date(2021, 5, 13),\n            valorNominal=0.01,\n            email=\'\',\n            telefone=\'\',\n            dataEmissao=datetime.date(2021, 5, 11),\n            dataLimite=datetime.date(2021, 6, 12),\n            linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n            desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            multa=pyinterboleto.common.multa.MultaConsulta(\n                codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n            ),\n            mora=pyinterboleto.common.mora.MoraConsulta(\n                codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n            ),\n            valorAbatimento=0.0,\n            dataPagtoBaixa=datetime.date(2021, 5, 11)\n        ),\n        pyinterboleto.consulta.lista.BoletoItem(\n            nossoNumero=\'00000000000\',\n            seuNumero=\'00003\',\n            cnpjCpfSacado=\'1234567809\',\n            nomeSacado=\'Pessoa Ficticia da Silva\',\n            situacao=\'BAIXADO\',\n            dataVencimento=datetime.date(2021, 5, 13),\n            valorNominal=0.01,\n            email=\'\',\n            telefone=\'\',\n            dataEmissao=datetime.date(2021, 5, 11),\n            dataLimite=datetime.date(2021, 6, 12),\n            linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n            desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            multa=pyinterboleto.common.multa.MultaConsulta(\n                codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n            ),\n            mora=pyinterboleto.common.mora.MoraConsulta(\n                codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n            ),\n            valorAbatimento=0.0,\n            dataPagtoBaixa=datetime.date(2021, 5, 11)\n        ),\n        pyinterboleto.consulta.lista.BoletoItem(\n            nossoNumero=\'00000000000\',\n            seuNumero=\'00004\',\n            cnpjCpfSacado=\'1234567809\',\n            nomeSacado=\'Pessoa Ficticia da Silva\',\n            situacao=\'EXPIRADO\',\n            dataVencimento=datetime.date(2021, 6, 1),\n            valorNominal=20.0,\n            email=\'\',\n            telefone=\'\',\n            dataEmissao=datetime.date(2021, 5, 30),\n            dataLimite=datetime.date(2021, 7, 1),\n            linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n            desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            multa=pyinterboleto.common.multa.MultaConsulta(\n                codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n            ),\n            mora=pyinterboleto.common.mora.MoraConsulta(\n                codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n            ),\n            valorAbatimento=0.0\n        ),\n        pyinterboleto.consulta.lista.BoletoItem(\n            nossoNumero=\'00000000000\',\n            seuNumero=\'00005\',\n            cnpjCpfSacado=\'1234567809\',\n            nomeSacado=\'Pessoa Ficticia da Silva\',\n            situacao=\'EXPIRADO\',\n            dataVencimento=datetime.date(2021, 7, 9),\n            valorNominal=18.0,\n            email=\'\',\n            telefone=\'\',\n            dataEmissao=datetime.date(2021, 7, 7),\n            dataLimite=datetime.date(2021, 8, 8),\n            linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n            desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            multa=pyinterboleto.common.multa.MultaConsulta(\n                codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n            ),\n            mora=pyinterboleto.common.mora.MoraConsulta(\n                codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n            ),\n            valorAbatimento=0.0\n        ),\n        pyinterboleto.consulta.lista.BoletoItem(\n            nossoNumero=\'00000000000\',\n            seuNumero=\'00006\',\n            cnpjCpfSacado=\'1234567809\',\n            nomeSacado=\'Pessoa Ficticia da Silva\',\n            situacao=\'VENCIDO\',\n            dataVencimento=datetime.date(2021, 8, 10),\n            valorNominal=43.0,\n            email=\'\',\n            telefone=\'\',\n            dataEmissao=datetime.date(2021, 8, 7),\n            dataLimite=datetime.date(2021, 9, 9),\n            linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n            desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            multa=pyinterboleto.common.multa.MultaConsulta(\n                codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n            ),\n            mora=pyinterboleto.common.mora.MoraConsulta(\n                codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n            ),\n            valorAbatimento=0.0\n        ),\n        pyinterboleto.consulta.lista.BoletoItem(\n            nossoNumero=\'00000000000\',\n            seuNumero=\'00007\',\n            cnpjCpfSacado=\'1234567809\',\n            nomeSacado=\'Pessoa Ficticia da Silva\',\n            situacao=\'VENCIDO\',\n            dataVencimento=datetime.date(2021, 8, 10),\n            valorNominal=23.2,\n            email=\'\',\n            telefone=\'\',\n            dataEmissao=datetime.date(2021, 8, 7),\n            dataLimite=datetime.date(2021, 9, 9),\n            linhaDigitavel=\'00000000000000000000000000000000000000000000000\',\n            desconto1=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto2=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            desconto3=pyinterboleto.common.desconto.DescontoConsulta(\n                codigo=pyinterboleto.common.desconto.CodigoDescontoEnum.NTD\n            ),\n            multa=pyinterboleto.common.multa.MultaConsulta(\n                codigo=pyinterboleto.common.multa.CodigoMultaEnum.NTM\n            ),\n            mora=pyinterboleto.common.mora.MoraConsulta(\n                codigo=pyinterboleto.common.mora.CodigoMoraEnum.I\n            ),\n            valorAbatimento=0.0\n        )\n    ]\n)\n>>>\n```\n\n### Resgate de PDFs de boletos individuais\nAssim como na consulta detalhada individual, é preciso saber o número de identificação do título antes de fazer a requisição. Há dois modos de como o PDF é armazendo: em memória ou salvo diretamento em um arquivo especificado.\n\n```python\n>>> from pathlib import Path\n>>> boleto = Boleto(configs)\n>>> num_boleto = \'00123456789\'\n>>> # Armazenado em um buffer de bytes na memória\n>>> pdf = boleto.consulta_pdf(num_boleto)\n>>>\n>>> # salva em um arquivo chamado \'boleto.pdf\' no diretório atual\n>>> filename = Path().resolve() / \'boleto.pdf\'\n>>> boleto.consulta_pdf(num_boleto, filename)\n>>>\n```\n\n***\n\n## Baixa de boleto\nTambém é preciso saber o número de identificação do título. Os tipos de baixas são definidos no enum [`CodigoBaixaEnum`](src/pyinterboleto/baixa/__init__.py).\n\n```python\n>>> from pyinterboleto import CodigoBaixaEnum\n>>> boleto = Boleto(configs)\n>>> num_boleto = \'00123456789\'\n>>> boleto.baixar_boleto(num_boleto, CodigoBaixaEnum.PC)\n>>>\n```\n\n***\n\n# Testagem\n\nComo a API do Inter não possui ambiente de sandboxing, optei por não implementar rotinas de testes para todas operações, apenas as de consulta. Isto é, o Inter fornece uma cota sem custo adicional de 100 boletos emitidos por mês. Acima disto, é preciso pagar mais.\n\nComo é um recurso bem limitado, não faz sentido implementar uma suíte de testes para emissão e baixa de boletos.\n\nPara realizar os testes localmente, clone o repositório e crie um arquivo chamado `inter.env` na raiz do projeto que tem o seguinte formato:\n\n```\nINTER_ACC=\'\'\nINTER_API_KEY=\'-----BEGIN PRIVATE KEY-----\n...\n-----END PRIVATE KEY-----\n\'\nINTER_API_CERTIFICATE=\'-----BEGIN CERTIFICATE-----\n...\n-----END CERTIFICATE-----\'\n```\n\nAs variáveis `INTER_ACC`, `INTER_API_KEY` e `INTER_API_CERTIFICATE` são o número da conta Inter (apenas números), contéudos do arquivo `.key` e `.crt` respectivamente.\n\nInstale as dependências de desenvolvimento:\n```shell\n# pode usar o gerenciador que quiser (e.g. poetry, conda, etc.)\npip install -r requirements-dev.txt\n```\n\nPara rodar os tests:\n```shell\npytest\n```\n',
    'author': 'Felipe Souza Lima',
    'author_email': 'feslima93@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/feslima/pyinterboleto',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
