"""
A script for generating a set of random seeds to use for more reproducible
results.
"""
import argparse
import random
import sys


def generate(number_of_seeds, seed=None):
    """Uses pythons random integer generator to create a set of unique random
    seeds.

    Parameters
    ----------
    number_of_seeds : int
        number of seeds to generate.
    seed : int, optional
        The random seed to intialize the generator with.

    Returns
    -------
        A list of integers.
    """
    if seed is not None:
        if isinstance(seed, int):
            random.seed(seed)
        else:
            raise TypeError('The inital seed is expected to be an integer.')

    random_seeds = set()
    # max_int = sys.maxsize
    max_int = 2**32 - 1
    i = 0
    while i < number_of_seeds:
        random_seed = random.randint(0, max_int)
        if random_seed not in random_seeds:
            random_seeds.add(random_seed)
            i += 1

    return random_seeds


def save_to_file(seeds, filepath):
    """Saves the random seeds as a file at the fiven filepath

    Parameters
    ----------
    seeds : numpy.array
        A 1D numpy.array of random seeds
    """
    with open(filepath, 'w') as f:
        for seed in seeds:
            f.write('%d\n' % seed)


def parse_args():
    parser = argparse.ArgumentParser(
        description='Generate unique random seeds and save to a file.',
    )
    parser.add_argument(
        'number_of_seeds',
        type=int,
        help='The number of seeds to be generated by calling this script.',
    )
    parser.add_argument(
        '-s',
        '--seed',
        default=None,
        type=int,
        help='An integer to serve as the initial seed to the generator.',
    )
    parser.add_argument(
        '-o',
        '--output_filepath',
        default=None,
        help='The filepath to save the random seeds file to.',
    )

    args = parser.parse_args()

    if args.output_filepath is None:
        args.output_filepath = f'random_seeds_count-{args.number_of_seeds}.txt'

    return args


if __name__ == '__main__':
    args = parse_args()
    seeds = generate(args.number_of_seeds, args.seed)
    save_to_file(seeds, args.output_filepath)
