#! /usr/bin/env python3
# Requires Python 3.6+
"""Sphinx extension for generating the rules table document."""

from typing import Dict, List, Union

from docutils import nodes, statemachine
from sphinx.application import Sphinx
from sphinx.util.docutils import SphinxDirective
from sphinx.util.nodes import nested_parse_with_titles

from ansiblelint import __version__
from ansiblelint.constants import DEFAULT_RULESDIR
from ansiblelint.generate_docs import profiles_as_md, rules_as_md
from ansiblelint.rules import RulesCollection


def _nodes_from_md(
    state: statemachine.State,
    md_source: str,
) -> List[nodes.Node]:
    """Turn an MD string into a list of nodes.

    These nodes can be used in the document.
    """
    node = nodes.Element()
    node.document = state.document
    nested_parse_with_titles(
        state=state,
        content=statemachine.ViewList(
            statemachine.string2lines(md_source),
            source="[ansible-lint autogenerated]",
        ),
        node=node,
    )
    return node.children


class AnsibleLintProfilesDirective(SphinxDirective):
    """Directive ``ansible-lint-profile-list`` definition."""

    has_content = False

    def run(self) -> List[nodes.Node]:
        """Generate a node tree in place of the directive."""
        self.env.note_reread()  # rebuild the current doc unconditionally

        md_rules_table = profiles_as_md()

        return _nodes_from_md(state=self.state, md_source=md_rules_table)


class AnsibleLintDefaultRulesDirective(SphinxDirective):
    """Directive ``ansible-lint-default-rules-list`` definition."""

    has_content = False

    def run(self) -> List[nodes.Node]:
        """Generate a node tree in place of the directive."""
        self.env.note_reread()  # rebuild the current doc unconditionally

        default_rules = RulesCollection([DEFAULT_RULESDIR])
        md_rules_table = rules_as_md(default_rules)

        return _nodes_from_md(state=self.state, md_source=md_rules_table)


def setup(app: Sphinx) -> Dict[str, Union[bool, str]]:
    """Initialize the Sphinx extension."""
    app.add_directive(
        "ansible-lint-default-rules-list",
        AnsibleLintDefaultRulesDirective,
    )
    app.add_directive(
        "ansible-lint-profile-list",
        AnsibleLintProfilesDirective,
    )

    return {
        "parallel_read_safe": True,
        "parallel_write_safe": True,
        "version": __version__,
    }
