# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['decoy']

package_data = \
{'': ['*']}

install_requires = \
['mock>=4.0.2,<5.0.0']

setup_kwargs = {
    'name': 'decoy',
    'version': '0.1.0',
    'description': 'Opinionated, typed stubbing and verification library for Python',
    'long_description': '# Decoy\n\n> Opinionated, typed stubbing and verification library for Python\n\nThe Decoy library allows you to create, stub, and verify test double objects for your Python unit tests, so your tests are:\n\n- Easier to fit into the Arrange-Act-Assert pattern\n- Less prone to insufficient tests due to unconditional stubbing\n- Covered by typechecking\n\nThe Decoy API is heavily inspired by / stolen from the excellent [testdouble.js][] and [Mockito][] projects.\n\n[testdouble.js]: https://github.com/testdouble/testdouble.js\n[mockito]: https://site.mockito.org/\n\n## Install\n\n```bash\n# pip\npip install decoy\n\n# poetry\npoetry add --dev decoy\n```\n\n## Usage\n\n### Setup\n\nYou\'ll want to create a test fixture to reset Decoy state between each test run. In [pytest][], you can do this by using a fixture to create a new Decoy instance for every test.\n\nThe examples below assume the following global test fixture:\n\n```python\nimport pytest\nfrom decoy import Decoy\n\n@pytest.fixture\ndef decoy() -> Decoy:\n    return Decoy()\n```\n\nWhy is this important? The `Decoy` container tracks every test double that is created during a test so that you can define assertions using fully-typed rehearsals of your test double. It\'s important to wipe this slate clean for every test so you don\'t leak memory or have any state preservation between tests.\n\n[pytest]: https://docs.pytest.org/en/latest/\n\n### Stubbing\n\nA stub is a an object used in a test that is pre-configured to act in a certain way if called according to a spec, defined by a rehearsal. A "rehearsal" is simply a call to the stub inside of a `decoy.when` wrapper.\n\nBy pre-configuring the stub with specific rehearsals, you get the following benefits:\n\n- Your test double will only return your mock value **if it is called correctly**\n- You avoid separate "set up mock return value" and "assert mock called correctly" steps\n- If you annotate your test double with an actual type, the rehearsal will fail typechecking if called incorrectly\n\n```python\nimport pytest\nfrom typing import cast, Optional\nfrom decoy import Decoy\n\nfrom .database import Database, Model\n\ndef get_item(uid: str, db: Database) -> Optional[Model]:\n  return db.get_by_id(uid)\n\ndef test_get_item(decoy: Decoy):\n    mock_item = cast(Model, { "foo": "bar" })\n    mock_db = decoy.create_decoy(spec=Database)\n\n    # arrange stub using rehearsals\n    decoy.when(mock_db.get_by_id("some-id")).then_return(mock_item)\n\n    # call code under test\n    some_result = get_item("some-id")\n    other_result = get_item("other-id")\n\n    # assert code result\n    assert some_result == mock_item\n    assert other_result is None\n```\n\n### Verification\n\nIf you\'re coming from `unittest.mock`, you\'re probably more used to calling your code under test and _then_ verifying that your test double was called correctly. Asserting on mock call signatures after the fact can be useful, but **should only be used if the dependency is being called solely for its side-effect(s)**.\n\nVerification of decoy calls after they have occurred be considered a last resort, because:\n\n- If you\'re calling a method/function to get its data, then you can more precisely describe that relationship using [stubbing](#stubbing)\n- Side-effects are harder to understand and maintain than pure functions, so in general you should try to side-effect sparingly\n\nStubbing and verification of a decoy are **mutually exclusive** within a test. If you find yourself wanting to both stub and verify the same decoy, then one or more of these is true:\n\n- The assertions are redundant\n- You should re-read the section on stubbing and maybe the [testdouble.js][] and/or [mockito][] documentation\n- Your dependency is doing too much based on its input (e.g. side-effecting _and_ calculating complex data) and should be refactored\n\n```python\nimport pytest\nfrom typing import cast, Optional\nfrom decoy import Decoy, verify\n\nfrom .logger import Logger\n\ndef log_warning(msg: str, logger: Logger) -> None:\n  logger.warn(msg)\n\ndef test_log_warning(decoy: Decoy):\n  logger = decoy.create_decoy(spec=Logger)\n\n  # call code under test\n  some_result = log_warning("oh no!", logger)\n\n  # verify double called correctly\n  decoy.verify(logger.warn("oh no!"))\n```\n\n### Matchers\n\nSometimes, when you\'re stubbing or verifying decoy calls (or really when you\'re doing any sort of equality assertion in a test), you need to loosen a given assertion. For example, you may want to assert that a double is called with a string, but you don\'t care what the full contents of that string is.\n\nDecoy includes a set of matchers, which are simply Python classes with `__eq__` methods defined, that you can use in decoy rehearsals and/or assertions.\n\n```python\nimport pytest\nfrom typing import cast, Optional\nfrom decoy import Decoy, matchers\n\nfrom .logger import Logger\n\ndef log_warning(msg: str, logger: Logger) -> None:\n  logger.warn(msg)\n\ndef test_log_warning(decoy: Decoy):\n  logger = decoy.create_decoy(spec=Logger)\n\n  # call code under test\n  some_result = log_warning(\n    "Oh no, something horrible went wrong with request ID abc123efg456",\n    logger=logger\n  )\n\n  # verify double called correctly\n  decoy.verify(\n    mock_logger.warn(matchers.StringMatching("something went wrong"))\n  )\n```\n',
    'author': 'Mike Cousins',
    'author_email': 'mike@cousins.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://decoy.cousins.io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
