"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/cdk");
const eks = require("../lib");
function testFixture() {
    const stack = new cdk.Stack(undefined, 'Stack', { env: { region: 'us-east-1' } });
    const vpc = new ec2.VpcNetwork(stack, 'VPC');
    return [stack, vpc];
}
module.exports = {
    'a default cluster spans all subnets'(test) {
        // GIVEN
        const [stack, vpc] = testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Cluster', {
            ResourcesVpcConfig: {
                SubnetIds: [
                    { Ref: "VPCPublicSubnet1SubnetB4246D30" },
                    { Ref: "VPCPublicSubnet2Subnet74179F39" },
                    { Ref: "VPCPublicSubnet3Subnet631C5E25" },
                    { Ref: "VPCPrivateSubnet1Subnet8BCA10E0" },
                    { Ref: "VPCPrivateSubnet2SubnetCFCDAA7A" },
                    { Ref: "VPCPrivateSubnet3Subnet3EDCD457" }
                ]
            }
        }));
        test.done();
    },
    'creating a cluster tags the private VPC subnets'(test) {
        // GIVEN
        const [stack, vpc] = testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            Tags: [
                { Key: "Name", Value: "VPC/PrivateSubnet1" },
                { Key: "aws-cdk:subnet-name", Value: "Private" },
                { Key: "aws-cdk:subnet-type", Value: "Private" },
                { Key: "kubernetes.io/role/internal-elb", Value: "1" }
            ]
        }));
        test.done();
    },
    'adding capacity creates an ASG with tags'(test) {
        // GIVEN
        const [stack, vpc] = testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: "Name",
                    PropagateAtLaunch: true,
                    Value: "Cluster/Default"
                },
                {
                    Key: { "Fn::Join": ["", ["kubernetes.io/cluster/", { Ref: "ClusterEB0386A7" }]] },
                    PropagateAtLaunch: true,
                    Value: "owned"
                }
            ]
        }));
        test.done();
    },
    'adding capacity correctly deduces maxPods and adds userdata'(test) {
        // GIVEN
        const [stack, vpc] = testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                "Fn::Base64": {
                    "Fn::Join": [
                        "",
                        [
                            "#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ",
                            { Ref: "ClusterEB0386A7" },
                            " --use-max-pods 17"
                        ]
                    ]
                }
            }
        }));
        test.done();
    },
    'exercise export/import'(test) {
        // GIVEN
        const [stack1, vpc] = testFixture();
        const stack2 = new cdk.Stack();
        const cluster = new eks.Cluster(stack1, 'Cluster', { vpc });
        // WHEN
        const imported = eks.Cluster.fromClusterAttributes(stack2, 'Imported', cluster.export());
        // THEN
        test.deepEqual(stack2.node.resolve(imported.clusterArn), {
            'Fn::ImportValue': 'Stack:ClusterClusterArn00DCA0E0'
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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