import autoscaling = require('@aws-cdk/aws-autoscaling');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import { Construct, IResource, Resource } from '@aws-cdk/cdk';
/**
 * An EKS cluster
 */
export interface ICluster extends IResource, ec2.IConnectable {
    /**
     * The VPC in which this Cluster was created
     */
    readonly vpc: ec2.IVpcNetwork;
    /**
     * The physical name of the Cluster
     * @attribute
     */
    readonly clusterName: string;
    /**
     * The unique ARN assigned to the service by AWS
     * in the form of arn:aws:eks:
     * @attribute
     */
    readonly clusterArn: string;
    /**
     * The API Server endpoint URL
     * @attribute
     */
    readonly clusterEndpoint: string;
    /**
     * The certificate-authority-data for your cluster.
     * @attribute
     */
    readonly clusterCertificateAuthorityData: string;
    /**
     * Export cluster references to use in other stacks
     */
    export(): ClusterAttributes;
}
/**
 * A SecurityGroup Reference, object not created with this template.
 */
declare abstract class ClusterBase extends Resource implements ICluster {
    abstract readonly connections: ec2.Connections;
    abstract readonly vpc: ec2.IVpcNetwork;
    abstract readonly clusterName: string;
    abstract readonly clusterArn: string;
    abstract readonly clusterEndpoint: string;
    abstract readonly clusterCertificateAuthorityData: string;
    /**
     * Export cluster references to use in other stacks
     */
    export(): ClusterAttributes;
    private makeOutput;
}
export interface ClusterAttributes {
    /**
     * The VPC in which this Cluster was created
     */
    readonly vpc: ec2.VpcNetworkImportProps;
    /**
     * The physical name of the Cluster
     */
    readonly clusterName: string;
    /**
     * The unique ARN assigned to the service by AWS
     * in the form of arn:aws:eks:
     */
    readonly clusterArn: string;
    /**
     * The API Server endpoint URL
     */
    readonly clusterEndpoint: string;
    /**
     * The certificate-authority-data for your cluster.
     */
    readonly clusterCertificateAuthorityData: string;
    readonly securityGroups: ec2.SecurityGroupAttributes[];
}
/**
 * Properties to instantiate the Cluster
 */
export interface ClusterProps {
    /**
     * The VPC in which to create the Cluster
     */
    readonly vpc: ec2.IVpcNetwork;
    /**
     * Where to place EKS Control Plane ENIs
     *
     * If you want to create public load balancers, this must include public subnets.
     *
     * For example, to only select private subnets, supply the following:
     *
     * ```ts
     * vpcSubnets: [
     *   { subnetType: ec2.SubnetType.Private }
     * ]
     * ```
     *
     * @default All public and private subnets
     */
    readonly vpcSubnets?: ec2.SubnetSelection[];
    /**
     * Role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf.
     *
     * @default A role is automatically created for you
     */
    readonly role?: iam.IRole;
    /**
     * Name for the cluster.
     *
     * @default Automatically generated name
     */
    readonly clusterName?: string;
    /**
     * Security Group to use for Control Plane ENIs
     *
     * @default A security group is automatically created
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The Kubernetes version to run in the cluster
     *
     * @default If not supplied, will use Amazon default version
     */
    readonly version?: string;
}
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
export declare class Cluster extends ClusterBase {
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope: Construct, id: string, attrs: ClusterAttributes): ICluster;
    /**
     * The VPC in which this Cluster was created
     */
    readonly vpc: ec2.IVpcNetwork;
    /**
     * The Name of the created EKS Cluster
     */
    readonly clusterName: string;
    /**
     * The AWS generated ARN for the Cluster resource
     *
     * @example arn:aws:eks:us-west-2:666666666666:cluster/prod
     */
    readonly clusterArn: string;
    /**
     * The endpoint URL for the Cluster
     *
     * This is the URL inside the kubeconfig file to use with kubectl
     *
     * @example https://5E1D0CEXAMPLEA591B746AFC5AB30262.yl4.us-west-2.eks.amazonaws.com
     */
    readonly clusterEndpoint: string;
    /**
     * The certificate-authority-data for your cluster.
     */
    readonly clusterCertificateAuthorityData: string;
    /**
     * Manages connection rules (Security Group Rules) for the cluster
     *
     * @type {ec2.Connections}
     * @memberof Cluster
     */
    readonly connections: ec2.Connections;
    /**
     * IAM role assumed by the EKS Control Plane
     */
    readonly role: iam.IRole;
    private readonly version;
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope: Construct, id: string, props: ClusterProps);
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     */
    addCapacity(id: string, options: AddWorkerNodesOptions): autoscaling.AutoScalingGroup;
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Prefer to use `addCapacity` if possible, it will automatically configure
     * the right AMI and the `maxPods` number based on your instance type.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     */
    addAutoScalingGroup(autoScalingGroup: autoscaling.AutoScalingGroup, options: AddAutoScalingGroupOptions): void;
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    private tagSubnets;
}
/**
 * Options for adding worker nodes
 */
export interface AddWorkerNodesOptions extends autoscaling.CommonAutoScalingGroupProps {
    /**
     * Instance type of the instances to start
     */
    readonly instanceType: ec2.InstanceType;
}
/**
 * Options for adding an AutoScalingGroup as capacity
 */
export interface AddAutoScalingGroupOptions {
    /**
     * How many pods to allow on this instance.
     *
     * Should be at most equal to the maximum number of IP addresses available to
     * the instance type less one.
     */
    readonly maxPods: number;
}
export {};
