"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
/**
 * Source for EKS optimized AMIs
 */
class EksOptimizedAmi extends ec2.GenericLinuxImage {
    constructor(props = {}) {
        const version = props.kubernetesVersion || LATEST_KUBERNETES_VERSION;
        if (!(version in EKS_AMI)) {
            throw new Error(`We don't have an AMI for kubernetes version ${version}`);
        }
        super(EKS_AMI[version][props.nodeType || NodeType.Normal]);
    }
}
exports.EksOptimizedAmi = EksOptimizedAmi;
const LATEST_KUBERNETES_VERSION = '1.11';
/**
 * Whether the worker nodes should support GPU or just normal instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Normal instances
     */
    NodeType["Normal"] = "Normal";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
function nodeTypeForInstanceType(instanceType) {
    return instanceType.toString().startsWith('p2') || instanceType.toString().startsWith('p3') ? NodeType.GPU : NodeType.Normal;
}
exports.nodeTypeForInstanceType = nodeTypeForInstanceType;
/**
 * Select AMI to use based on the AWS Region being deployed
 *
 * TODO: Create dynamic mappign by searching SSM Store
 *
 * @see https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html
 */
const EKS_AMI = {
    '1.10': parseTable(`
    US West (Oregon) (us-west-2) 	ami-09e1df3bad220af0b 	ami-0ebf0561e61a2be02
    US East (N. Virginia) (us-east-1) 	ami-04358410d28eaab63 	ami-0131c0ca222183def
    US East (Ohio) (us-east-2) 	ami-0b779e8ab57655b4b 	ami-0abfb3be33c196cbf
    EU (Frankfurt) (eu-central-1) 	ami-08eb700778f03ea94 	ami-000622b1016d2a5bf
    EU (Stockholm) (eu-north-1) 	ami-068b8a1efffd30eda 	ami-cc149ab2
    EU (Ireland) (eu-west-1) 	ami-0de10c614955da932 	ami-0dafd3a1dc43781f7
    Asia Pacific (Tokyo) (ap-northeast-1) 	ami-06398bdd37d76571d 	ami-0afc9d14b2fe11ad9
    Asia Pacific (Seoul) (ap-northeast-2) 	ami-08a87e0a7c32fa649 	ami-0d75b9ab57bfc8c9a
    Asia Pacific (Singapore) (ap-southeast-1) 	ami-0ac3510e44b5bf8ef 	ami-0ecce0670cb66d17b
    Asia Pacific (Sydney) (ap-southeast-2) 	ami-0d2c929ace88cfebe 	ami-03b048bd9d3861ce9
    `),
    '1.11': parseTable(`
    US West (Oregon) (us-west-2) 	ami-0a2abab4107669c1b 	ami-0c9e5e2d8caa9fb5e
    US East (N. Virginia) (us-east-1) 	ami-0c24db5df6badc35a 	ami-0ff0241c02b279f50
    US East (Ohio) (us-east-2) 	ami-0c2e8d28b1f854c68 	ami-006a12f54eaafc2b1
    EU (Frankfurt) (eu-central-1) 	ami-010caa98bae9a09e2 	ami-0d6f0554fd4743a9d
    EU (Stockholm) (eu-north-1) 	ami-06ee67302ab7cf838 	ami-0b159b75
    EU (Ireland) (eu-west-1) 	ami-01e08d22b9439c15a 	ami-097978e7acde1fd7c
    Asia Pacific (Tokyo) (ap-northeast-1) 	ami-0f0e8066383e7a2cb 	ami-036b3969c5eb8d3cf
    Asia Pacific (Seoul) (ap-northeast-2) 	ami-0b7baa90de70f683f 	ami-0b7f163f7194396f7
    Asia Pacific (Singapore) (ap-southeast-1) 	ami-019966ed970c18502 	ami-093f742654a955ee6
    Asia Pacific (Sydney) (ap-southeast-2) 	ami-06ade0abbd8eca425 	ami-05e09575123ff498b
    `),
};
/**
 * Helper function which makes it easier to copy/paste the HTML AMI table into this source.
 *
 * I can't stress enough how much of a temporary solution this should be, but until we
 * have a proper discovery mechanism, this is easier than converting the table into
 * nested dicts by hand.
 */
function parseTable(contents) {
    const normalTable = {};
    const gpuTable = {};
    // Last parenthesized expression that looks like a region
    const extractRegion = /\(([a-z]+-[a-z]+-[0-9]+)\)\s*$/;
    for (const line of contents.split('\n')) {
        if (line.trim() === '') {
            continue;
        }
        const parts = line.split('\t');
        if (parts.length !== 3) {
            throw new Error(`Line lost its TABs: "${line}"`);
        }
        const m = extractRegion.exec(parts[0]);
        if (!m) {
            throw new Error(`Like doesn't seem to contain a region: "${line}"`);
        }
        const region = m[1];
        normalTable[region] = parts[1].trim();
        gpuTable[region] = parts[2].trim();
    }
    return {
        [NodeType.GPU]: gpuTable,
        [NodeType.Normal]: normalTable
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1pLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYW1pLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQXlDO0FBcUJ6Qzs7R0FFRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxHQUFHLENBQUMsaUJBQWlCO0lBQ3hELFlBQVksUUFBOEIsRUFBRTtRQUMxQyxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsaUJBQWlCLElBQUkseUJBQXlCLENBQUM7UUFDckUsSUFBSSxDQUFDLENBQUMsT0FBTyxJQUFJLE9BQU8sQ0FBQyxFQUFFO1lBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLE9BQU8sRUFBRSxDQUFDLENBQUM7U0FDM0U7UUFDRCxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7SUFDN0QsQ0FBQztDQUNGO0FBUkQsMENBUUM7QUFFRCxNQUFNLHlCQUF5QixHQUFHLE1BQU0sQ0FBQztBQUV6Qzs7R0FFRztBQUNILElBQVksUUFVWDtBQVZELFdBQVksUUFBUTtJQUNsQjs7T0FFRztJQUNILDZCQUFpQixDQUFBO0lBRWpCOztPQUVHO0lBQ0gsdUJBQVcsQ0FBQTtBQUNiLENBQUMsRUFWVyxRQUFRLEdBQVIsZ0JBQVEsS0FBUixnQkFBUSxRQVVuQjtBQUVELFNBQWdCLHVCQUF1QixDQUFDLFlBQThCO0lBQ3BFLE9BQU8sWUFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDO0FBQy9ILENBQUM7QUFGRCwwREFFQztBQUVEOzs7Ozs7R0FNRztBQUNILE1BQU0sT0FBTyxHQUFzRTtJQUNqRixNQUFNLEVBQUUsVUFBVSxDQUFDOzs7Ozs7Ozs7OztLQVdoQixDQUFDO0lBQ0osTUFBTSxFQUFFLFVBQVUsQ0FBQzs7Ozs7Ozs7Ozs7S0FXaEIsQ0FBQztDQUNMLENBQUM7QUFFRjs7Ozs7O0dBTUc7QUFDSCxTQUFTLFVBQVUsQ0FBQyxRQUFnQjtJQUNsQyxNQUFNLFdBQVcsR0FBK0IsRUFBRSxDQUFDO0lBQ25ELE1BQU0sUUFBUSxHQUErQixFQUFFLENBQUM7SUFFaEQseURBQXlEO0lBQ3pELE1BQU0sYUFBYSxHQUFHLGdDQUFnQyxDQUFDO0lBRXZELEtBQUssTUFBTSxJQUFJLElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRTtRQUN2QyxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLEVBQUU7WUFBRSxTQUFTO1NBQUU7UUFFckMsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvQixJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsd0JBQXdCLElBQUksR0FBRyxDQUFDLENBQUM7U0FDbEQ7UUFFRCxNQUFNLENBQUMsR0FBRyxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxDQUFDLEVBQUU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLDJDQUEyQyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQUU7UUFDaEYsTUFBTSxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXBCLFdBQVcsQ0FBQyxNQUFNLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDdEMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQztLQUNwQztJQUVELE9BQU87UUFDTCxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsRUFBRSxRQUFRO1FBQ3hCLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFLFdBQVc7S0FDL0IsQ0FBQztBQUNKLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgZWMyID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWVjMicpO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIEVrc09wdGltaXplZEFtaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVrc09wdGltaXplZEFtaVByb3BzIHtcbiAgLyoqXG4gICAqIFdoYXQgaW5zdGFuY2UgdHlwZSB0byByZXRyaWV2ZSB0aGUgaW1hZ2UgZm9yIChub3JtYWwgb3IgR1BVLW9wdGltaXplZClcbiAgICpcbiAgICogQGRlZmF1bHQgTm9ybWFsXG4gICAqL1xuICByZWFkb25seSBub2RlVHlwZT86IE5vZGVUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgS3ViZXJuZXRlcyB2ZXJzaW9uIHRvIHVzZVxuICAgKlxuICAgKiBAZGVmYXVsdCBUaGUgbGF0ZXN0IHZlcnNpb25cbiAgICovXG4gIHJlYWRvbmx5IGt1YmVybmV0ZXNWZXJzaW9uPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFNvdXJjZSBmb3IgRUtTIG9wdGltaXplZCBBTUlzXG4gKi9cbmV4cG9ydCBjbGFzcyBFa3NPcHRpbWl6ZWRBbWkgZXh0ZW5kcyBlYzIuR2VuZXJpY0xpbnV4SW1hZ2UgaW1wbGVtZW50cyBlYzIuSU1hY2hpbmVJbWFnZVNvdXJjZSB7XG4gIGNvbnN0cnVjdG9yKHByb3BzOiBFa3NPcHRpbWl6ZWRBbWlQcm9wcyA9IHt9KSB7XG4gICAgY29uc3QgdmVyc2lvbiA9IHByb3BzLmt1YmVybmV0ZXNWZXJzaW9uIHx8IExBVEVTVF9LVUJFUk5FVEVTX1ZFUlNJT047XG4gICAgaWYgKCEodmVyc2lvbiBpbiBFS1NfQU1JKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBXZSBkb24ndCBoYXZlIGFuIEFNSSBmb3Iga3ViZXJuZXRlcyB2ZXJzaW9uICR7dmVyc2lvbn1gKTtcbiAgICB9XG4gICAgc3VwZXIoRUtTX0FNSVt2ZXJzaW9uXVtwcm9wcy5ub2RlVHlwZSB8fCBOb2RlVHlwZS5Ob3JtYWxdKTtcbiAgfVxufVxuXG5jb25zdCBMQVRFU1RfS1VCRVJORVRFU19WRVJTSU9OID0gJzEuMTEnO1xuXG4vKipcbiAqIFdoZXRoZXIgdGhlIHdvcmtlciBub2RlcyBzaG91bGQgc3VwcG9ydCBHUFUgb3IganVzdCBub3JtYWwgaW5zdGFuY2VzXG4gKi9cbmV4cG9ydCBlbnVtIE5vZGVUeXBlIHtcbiAgLyoqXG4gICAqIE5vcm1hbCBpbnN0YW5jZXNcbiAgICovXG4gIE5vcm1hbCA9ICdOb3JtYWwnLFxuXG4gIC8qKlxuICAgKiBHUFUgaW5zdGFuY2VzXG4gICAqL1xuICBHUFUgPSAnR1BVJyxcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIG5vZGVUeXBlRm9ySW5zdGFuY2VUeXBlKGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZSkge1xuICByZXR1cm4gaW5zdGFuY2VUeXBlLnRvU3RyaW5nKCkuc3RhcnRzV2l0aCgncDInKSB8fCBpbnN0YW5jZVR5cGUudG9TdHJpbmcoKS5zdGFydHNXaXRoKCdwMycpID8gTm9kZVR5cGUuR1BVIDogTm9kZVR5cGUuTm9ybWFsO1xufVxuXG4vKipcbiAqIFNlbGVjdCBBTUkgdG8gdXNlIGJhc2VkIG9uIHRoZSBBV1MgUmVnaW9uIGJlaW5nIGRlcGxveWVkXG4gKlxuICogVE9ETzogQ3JlYXRlIGR5bmFtaWMgbWFwcGlnbiBieSBzZWFyY2hpbmcgU1NNIFN0b3JlXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWtzL2xhdGVzdC91c2VyZ3VpZGUvZWtzLW9wdGltaXplZC1hbWkuaHRtbFxuICovXG5jb25zdCBFS1NfQU1JOiB7W3ZlcnNpb246IHN0cmluZ106IHtbdHlwZTogc3RyaW5nXToge1tyZWdpb246IHN0cmluZ106IHN0cmluZ319fSA9IHtcbiAgJzEuMTAnOiBwYXJzZVRhYmxlKGBcbiAgICBVUyBXZXN0IChPcmVnb24pICh1cy13ZXN0LTIpIFx0YW1pLTA5ZTFkZjNiYWQyMjBhZjBiIFx0YW1pLTBlYmYwNTYxZTYxYTJiZTAyXG4gICAgVVMgRWFzdCAoTi4gVmlyZ2luaWEpICh1cy1lYXN0LTEpIFx0YW1pLTA0MzU4NDEwZDI4ZWFhYjYzIFx0YW1pLTAxMzFjMGNhMjIyMTgzZGVmXG4gICAgVVMgRWFzdCAoT2hpbykgKHVzLWVhc3QtMikgXHRhbWktMGI3NzllOGFiNTc2NTViNGIgXHRhbWktMGFiZmIzYmUzM2MxOTZjYmZcbiAgICBFVSAoRnJhbmtmdXJ0KSAoZXUtY2VudHJhbC0xKSBcdGFtaS0wOGViNzAwNzc4ZjAzZWE5NCBcdGFtaS0wMDA2MjJiMTAxNmQyYTViZlxuICAgIEVVIChTdG9ja2hvbG0pIChldS1ub3J0aC0xKSBcdGFtaS0wNjhiOGExZWZmZmQzMGVkYSBcdGFtaS1jYzE0OWFiMlxuICAgIEVVIChJcmVsYW5kKSAoZXUtd2VzdC0xKSBcdGFtaS0wZGUxMGM2MTQ5NTVkYTkzMiBcdGFtaS0wZGFmZDNhMWRjNDM3ODFmN1xuICAgIEFzaWEgUGFjaWZpYyAoVG9reW8pIChhcC1ub3J0aGVhc3QtMSkgXHRhbWktMDYzOThiZGQzN2Q3NjU3MWQgXHRhbWktMGFmYzlkMTRiMmZlMTFhZDlcbiAgICBBc2lhIFBhY2lmaWMgKFNlb3VsKSAoYXAtbm9ydGhlYXN0LTIpIFx0YW1pLTA4YTg3ZTBhN2MzMmZhNjQ5IFx0YW1pLTBkNzViOWFiNTdiZmM4YzlhXG4gICAgQXNpYSBQYWNpZmljIChTaW5nYXBvcmUpIChhcC1zb3V0aGVhc3QtMSkgXHRhbWktMGFjMzUxMGU0NGI1YmY4ZWYgXHRhbWktMGVjY2UwNjcwY2I2NmQxN2JcbiAgICBBc2lhIFBhY2lmaWMgKFN5ZG5leSkgKGFwLXNvdXRoZWFzdC0yKSBcdGFtaS0wZDJjOTI5YWNlODhjZmViZSBcdGFtaS0wM2IwNDhiZDlkMzg2MWNlOVxuICAgIGApLFxuICAnMS4xMSc6IHBhcnNlVGFibGUoYFxuICAgIFVTIFdlc3QgKE9yZWdvbikgKHVzLXdlc3QtMikgXHRhbWktMGEyYWJhYjQxMDc2NjljMWIgXHRhbWktMGM5ZTVlMmQ4Y2FhOWZiNWVcbiAgICBVUyBFYXN0IChOLiBWaXJnaW5pYSkgKHVzLWVhc3QtMSkgXHRhbWktMGMyNGRiNWRmNmJhZGMzNWEgXHRhbWktMGZmMDI0MWMwMmIyNzlmNTBcbiAgICBVUyBFYXN0IChPaGlvKSAodXMtZWFzdC0yKSBcdGFtaS0wYzJlOGQyOGIxZjg1NGM2OCBcdGFtaS0wMDZhMTJmNTRlYWFmYzJiMVxuICAgIEVVIChGcmFua2Z1cnQpIChldS1jZW50cmFsLTEpIFx0YW1pLTAxMGNhYTk4YmFlOWEwOWUyIFx0YW1pLTBkNmYwNTU0ZmQ0NzQzYTlkXG4gICAgRVUgKFN0b2NraG9sbSkgKGV1LW5vcnRoLTEpIFx0YW1pLTA2ZWU2NzMwMmFiN2NmODM4IFx0YW1pLTBiMTU5Yjc1XG4gICAgRVUgKElyZWxhbmQpIChldS13ZXN0LTEpIFx0YW1pLTAxZTA4ZDIyYjk0MzljMTVhIFx0YW1pLTA5Nzk3OGU3YWNkZTFmZDdjXG4gICAgQXNpYSBQYWNpZmljIChUb2t5bykgKGFwLW5vcnRoZWFzdC0xKSBcdGFtaS0wZjBlODA2NjM4M2U3YTJjYiBcdGFtaS0wMzZiMzk2OWM1ZWI4ZDNjZlxuICAgIEFzaWEgUGFjaWZpYyAoU2VvdWwpIChhcC1ub3J0aGVhc3QtMikgXHRhbWktMGI3YmFhOTBkZTcwZjY4M2YgXHRhbWktMGI3ZjE2M2Y3MTk0Mzk2ZjdcbiAgICBBc2lhIFBhY2lmaWMgKFNpbmdhcG9yZSkgKGFwLXNvdXRoZWFzdC0xKSBcdGFtaS0wMTk5NjZlZDk3MGMxODUwMiBcdGFtaS0wOTNmNzQyNjU0YTk1NWVlNlxuICAgIEFzaWEgUGFjaWZpYyAoU3lkbmV5KSAoYXAtc291dGhlYXN0LTIpIFx0YW1pLTA2YWRlMGFiYmQ4ZWNhNDI1IFx0YW1pLTA1ZTA5NTc1MTIzZmY0OThiXG4gICAgYCksXG59O1xuXG4vKipcbiAqIEhlbHBlciBmdW5jdGlvbiB3aGljaCBtYWtlcyBpdCBlYXNpZXIgdG8gY29weS9wYXN0ZSB0aGUgSFRNTCBBTUkgdGFibGUgaW50byB0aGlzIHNvdXJjZS5cbiAqXG4gKiBJIGNhbid0IHN0cmVzcyBlbm91Z2ggaG93IG11Y2ggb2YgYSB0ZW1wb3Jhcnkgc29sdXRpb24gdGhpcyBzaG91bGQgYmUsIGJ1dCB1bnRpbCB3ZVxuICogaGF2ZSBhIHByb3BlciBkaXNjb3ZlcnkgbWVjaGFuaXNtLCB0aGlzIGlzIGVhc2llciB0aGFuIGNvbnZlcnRpbmcgdGhlIHRhYmxlIGludG9cbiAqIG5lc3RlZCBkaWN0cyBieSBoYW5kLlxuICovXG5mdW5jdGlvbiBwYXJzZVRhYmxlKGNvbnRlbnRzOiBzdHJpbmcpOiB7W3R5cGU6IHN0cmluZ106IHtbcmVnaW9uOiBzdHJpbmddOiBzdHJpbmd9fSB7XG4gIGNvbnN0IG5vcm1hbFRhYmxlOiB7W3JlZ2lvbjogc3RyaW5nXTogc3RyaW5nfSA9IHt9O1xuICBjb25zdCBncHVUYWJsZToge1tyZWdpb246IHN0cmluZ106IHN0cmluZ30gPSB7fTtcblxuICAvLyBMYXN0IHBhcmVudGhlc2l6ZWQgZXhwcmVzc2lvbiB0aGF0IGxvb2tzIGxpa2UgYSByZWdpb25cbiAgY29uc3QgZXh0cmFjdFJlZ2lvbiA9IC9cXCgoW2Etel0rLVthLXpdKy1bMC05XSspXFwpXFxzKiQvO1xuXG4gIGZvciAoY29uc3QgbGluZSBvZiBjb250ZW50cy5zcGxpdCgnXFxuJykpIHtcbiAgICBpZiAobGluZS50cmltKCkgPT09ICcnKSB7IGNvbnRpbnVlOyB9XG5cbiAgICBjb25zdCBwYXJ0cyA9IGxpbmUuc3BsaXQoJ1xcdCcpO1xuICAgIGlmIChwYXJ0cy5sZW5ndGggIT09IDMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTGluZSBsb3N0IGl0cyBUQUJzOiBcIiR7bGluZX1cImApO1xuICAgIH1cblxuICAgIGNvbnN0IG0gPSBleHRyYWN0UmVnaW9uLmV4ZWMocGFydHNbMF0pO1xuICAgIGlmICghbSkgeyB0aHJvdyBuZXcgRXJyb3IoYExpa2UgZG9lc24ndCBzZWVtIHRvIGNvbnRhaW4gYSByZWdpb246IFwiJHtsaW5lfVwiYCk7IH1cbiAgICBjb25zdCByZWdpb24gPSBtWzFdO1xuXG4gICAgbm9ybWFsVGFibGVbcmVnaW9uXSA9IHBhcnRzWzFdLnRyaW0oKTtcbiAgICBncHVUYWJsZVtyZWdpb25dID0gcGFydHNbMl0udHJpbSgpO1xuICB9XG5cbiAgcmV0dXJuIHtcbiAgICBbTm9kZVR5cGUuR1BVXTogZ3B1VGFibGUsXG4gICAgW05vZGVUeXBlLk5vcm1hbF06IG5vcm1hbFRhYmxlXG4gIH07XG59XG4iXX0=