"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const api = require("aws-cdk-lib/aws-apigateway");
const kms = require("aws-cdk-lib/aws-kms");
const sqs = require("aws-cdk-lib/aws-sqs");
test('Test deployment w/o DLQ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deployDeadLetterQueue: false
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
test('Test deployment w/o allowReadOperation', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        allowReadOperation: false,
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM"
    });
});
test('Test deployment w/ allowReadOperation', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: true,
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
test('Test properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const pattern = new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deployDeadLetterQueue: true,
        maxReceiveCount: 3
    });
    // Assertion 1
    expect(pattern.apiGateway !== null);
    // Assertion 2
    expect(pattern.sqsQueue !== null);
    // Assertion 3
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
    expect(pattern.deadLetterQueue !== null);
});
test('Test deployment ApiGateway AuthorizationType override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        },
        queueProps: {},
        createRequestTemplate: "{}",
        allowCreateOperation: true,
        allowReadOperation: false,
        allowDeleteOperation: true,
        deployDeadLetterQueue: false
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "NONE"
    });
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "NONE"
    });
});
test('Test deployment for override ApiGateway createRequestTemplate', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        createRequestTemplate: "Action=SendMessage&MessageBody=$util.urlEncode(\"HelloWorld\")",
        allowCreateOperation: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=SendMessage&MessageBody=$util.urlEncode(\"HelloWorld\")"
            }
        }
    });
});
test('Test deployment for override ApiGateway getRequestTemplate', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        readRequestTemplate: "Action=HelloWorld",
        allowReadOperation: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=HelloWorld"
            }
        }
    });
});
test('Test deployment for override ApiGateway deleteRequestTemplate', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deleteRequestTemplate: "Action=HelloWorld",
        allowDeleteOperation: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=HelloWorld"
            }
        }
    });
});
test('Test deployment with existing queue object', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingQueueObj = new sqs.Queue(stack, 'existing-queue', {
        fifo: true
    });
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        existingQueueObj
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        FifoQueue: true
    });
    expect(stack).toCountResources("AWS::SQS::Queue", 1);
});
test('Queue is encrypted with imported CMK when set on encryptionKey prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = new kms.Key(stack, 'cmk');
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        encryptionKey: cmk
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with imported CMK when set on queueProps.encryptionKeyProps prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = new kms.Key(stack, 'cmk');
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        queueProps: {
            encryptionMasterKey: cmk
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with provided encrytionKeyProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        encryptionKeyProps: {
            alias: 'new-key-alias-from-props'
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "apigatewaysqsEncryptionKey4A698F7C",
                "Arn"
            ]
        }
    });
    expect(stack).toHaveResource('AWS::KMS::Alias', {
        AliasName: 'alias/new-key-alias-from-props',
        TargetKeyId: {
            'Fn::GetAtt': [
                'apigatewaysqsEncryptionKey4A698F7C',
                'Arn'
            ]
        }
    });
});
test('Queue is encrypted by default with AWS-managed KMS key when no other encryption properties are set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {});
    expect(stack).toHaveResource('AWS::SQS::Queue', {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
test('Queue is encrypted with customer managed KMS Key when enable encryption flag is true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "apigatewaysqsEncryptionKey4A698F7C",
                "Arn"
            ]
        }
    });
});
test('Construct accepts additional read request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        additionalReadRequestTemplates: {
            'text/plain': 'Hello'
        }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            RequestTemplates: {
                'application/json': 'Action=ReceiveMessage',
                'text/plain': 'Hello'
            }
        }
    });
});
test('Construct accepts additional create request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowCreateOperation: true,
        additionalCreateRequestTemplates: {
            'text/plain': 'Hello'
        }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'application/json': 'Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")',
                'text/plain': 'Hello'
            }
        }
    });
});
test('Construct accepts additional delete request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowDeleteOperation: true,
        additionalDeleteRequestTemplates: {
            'text/plain': 'DeleteMe'
        }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            RequestTemplates: {
                'application/json': `Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))`,
                'text/plain': 'DeleteMe'
            }
        }
    });
});
test('Construct can override default create request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowCreateOperation: true,
        createRequestTemplate: 'Hello'
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'application/json': 'Hello'
            }
        }
    });
});
test('Construct can override default read request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowReadOperation: true,
        readRequestTemplate: 'Hello'
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            RequestTemplates: {
                'application/json': 'Hello'
            }
        }
    });
});
test('Construct can override default delete request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowDeleteOperation: true,
        deleteRequestTemplate: 'Hello'
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            RequestTemplates: {
                'application/json': 'Hello'
            }
        }
    });
});
test('Construct uses default integration responses', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        allowReadOperation: true,
        allowDeleteOperation: true,
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
});
test('Construct uses custom createIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        createIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom readIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        readIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom deleteIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowDeleteOperation: true,
        deleteIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct throws error when createRequestTemplate is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        createRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when additionalCreateRequestTemplates is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        additionalCreateRequestTemplates: {}
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when createIntegrationResponses is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        createIntegrationResponses: []
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when readRequestTemplate is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: false,
        readRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when additionalReadRequestTemplates is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: false,
        additionalReadRequestTemplates: {},
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when readIntegrationResponses is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: false,
        readIntegrationResponses: [],
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when deleteRequestTemplate is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deleteRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
test('Construct throws error when additionalDeleteRequestTemplates is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        additionalDeleteRequestTemplates: {}
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
test('Construct throws error when deleteIntegrationResponses is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deleteIntegrationResponses: []
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
//# sourceMappingURL=data:application/json;base64,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