from typing import Any, Dict, List, Tuple, Union

import logging

from .dataclasses import (
    BuiltinCallable,
    Processor,
    ValidationResult,
    ValidationStep,
    Validator,
    ValidatorError,
    ValidatorResult,
    dump_ValidationResults,
    load_ValidationStep,
    path_to_ValidationStep,
)
from .exceptions import NpValidatorError, ParsingError
from .path_matching import path_matches_pattern

logger = logging.getLogger(__name__)


def run_BuiltinCallable(target: Any, builtin: BuiltinCallable) -> Any:
    """Runs a BuiltinCallable on an arbitrary value."""
    return builtin.resolved_function(
        target,
        **builtin.args,
    )


def run_ValidationStep(filepath: str, step: ValidationStep) -> ValidationResult:
    """Runs a single validation step.

    Args:
        filepath (str): filepath of data to validate.
        step (ValidationStep): validation step to run.

    Returns:
        result (ValidationResult): result of the validation
    """
    if step.processor:
        try:
            processed = run_BuiltinCallable(
                filepath,
                step.processor.builtin,
            )  # type: Union[Any, str]
        except Exception as e:
            return ValidationResult(
                filepath=filepath,
                errors=[
                    ValidatorError(
                        str(e),
                        processor=step.processor,
                    )
                ],
            )
    else:
        processed = filepath

    results = []
    errors = []
    for validator in step.validators:
        try:
            value, passed = run_BuiltinCallable(
                processed,
                validator.builtin,
            )
            results.append(
                ValidatorResult(
                    value=value,
                    passed=passed,
                    validator=validator,
                )
            )
        except Exception as e:
            errors.append(
                ValidatorError(
                    message=str(e),
                    validator=validator,
                )
            )

    if step.processor:
        return ValidationResult(
            filepath=filepath,
            processor=step.processor,
            results=results,
            errors=errors,
        )
    else:
        return ValidationResult(
            filepath=filepath,
            results=results,
            errors=errors,
        )


def run_validation(
    file_list: List[str],
    validation_steps: List[ValidationStep] = [],
    autogen_steps: bool = True,
    project: str = "",
    raise_if_errors: bool = False,
) -> List[ValidationResult]:
    """Validates a list of files.

    Args:
        file_list (:obj:`list` of :obj:`str`): List of filepaths to validate.
        validation_steps: List of validation steps to run.
        autogen_steps: Add additional autogenerated ValidationSteps based on file types in `file_list`.
        project: Project id associated with the validation.
        raise_if_errors: Raise an NpValidatorError if ValidationStep errors occur.

    Returns:
        results: List of validation results.

    Raises:
        NpValidatorError: One or more validation steps had an unexpected error.
    """
    if autogen_steps:
        logger.debug("Autogenerating validation steps")
        autogenerated = autogenerate_validation_steps(project)
        logger.debug("Autogenerated: %s" % autogenerated)
        validation_steps = list(validation_steps)  # cast in case of map
        validation_steps.extend(autogenerated)

    results = []
    for validation_step in list(validation_steps):  # lazy way to accept a map type
        for filepath in file_list:
            if path_matches_pattern(filepath, validation_step.path_suffix):
                results.append(
                    run_ValidationStep(
                        filepath,
                        validation_step,
                    )
                )

    if raise_if_errors:
        for result in results:
            if len(result.errors) > 0:
                raise NpValidatorError("A validation step failed.")

    return results


def autogenerate_validation_steps(
    project: str = "",
) -> List[ValidationStep]:

    if project == "":
        sync_loader = Processor(name="load_sync_dataset")

        cam_sync_line_labels = [
            "beh_cam_frame_readout",
            "face_cam_frame_readout",
            "eye_cam_frame_readout",
            "beh_cam_exposing",
            "face_cam_exposing",
            "eye_cam_exposing",
        ]

        cam_line_validators = [
            Validator(
                name="meets_value_threshold",
                args={
                    "path": [line],
                    "threshold": 60,
                    "tolerance": 1,
                },
            )
            for line in cam_sync_line_labels
        ]

        barcode_validator = [
            Validator(
                name="meets_value_threshold",
                args={
                    "path": ["barcode_ephys"],
                    "threshold": 5,
                },
            )
        ]

        diode_validator = [
            Validator(
                name="meets_value_threshold",
                args={"path": ["stim_photodiode"], "threshold": 0.5, "tolerance": 0.05},
            )
        ]

        vsync_validator = [
            Validator(
                name="meets_value_threshold",
                args={"path": ["vsync_stim"], "threshold": 60, "tolerance": 1},
            )
        ]

        stim_running_validator = [
            Validator(
                name="meets_value_threshold",
                args={
                    "path": ["stim_running"],
                    "threshold": 0.5,
                },
            )
        ]

        sync_validators = (
            cam_line_validators
            + barcode_validator
            + diode_validator
            + vsync_validator
            + stim_running_validator
        )

        SyncValidationStep = [
            ValidationStep(
                path_suffix=".sync", processor=sync_loader, validators=sync_validators
            )
        ]

        unpickler = Processor(
            name="unpickle",
        )

        standard_pkl_suffixes = [".behavior.pkl", ".mapping.pkl", ".stim.pkl"]

        running_pkl_validator = [
            Validator(
                name="meets_wheel_rotation_threshold",
                args={
                    "threshold": 2,
                },
            )
        ]

        runningValidationStep = [
            ValidationStep(
                path_suffix=suffix,
                processor=unpickler,
                validators=running_pkl_validator,
            )
            for suffix in standard_pkl_suffixes
        ]

        fileNotEmptyValidationStep = [
            ValidationStep(
                path_suffix="",
                validators=[
                    Validator(
                        name="meets_filesize_threshold",
                        args={
                            "threshold": 1,
                        },
                    )
                ],
            )
        ]

        validationStepList = (
            SyncValidationStep + runningValidationStep + fileNotEmptyValidationStep
        )
    else:
        validationStepList = []

    return validationStepList
