"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const cdk = require("@aws-cdk/core");
const appmesh = require("../lib");
module.exports = {
    'When creating a Mesh': {
        'with no spec applied': {
            'should defaults to DROP_ALL egress filter'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                new appmesh.Mesh(stack, 'mesh', { meshName: 'test-mesh' });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::Mesh', {
                    Spec: {},
                }));
                test.done();
            },
        },
        'with spec applied': {
            'should take egress filter from props'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                    egressFilter: appmesh.MeshFilterType.ALLOW_ALL,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::Mesh', {
                    Spec: {
                        EgressFilter: {
                            Type: 'ALLOW_ALL',
                        },
                    },
                }));
                test.done();
            },
        },
    },
    'When adding a Virtual Router to existing mesh': {
        'with at least one complete port mappings': {
            'shoulld create proper router'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                mesh.addVirtualRouter('router', {
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    }
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::VirtualRouter', {
                    Spec: {
                        Listeners: [
                            {
                                PortMapping: {
                                    Port: 8080,
                                    Protocol: 'http',
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
    },
    'VirtualService can use CloudMap service'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const mesh = new appmesh.Mesh(stack, 'mesh', {
            meshName: 'test-mesh',
        });
        const vpc = new ec2.Vpc(stack, 'vpc');
        const namespace = new cloudmap.PrivateDnsNamespace(stack, 'test-namespace', {
            vpc,
            name: 'domain.local',
        });
        const service = namespace.createService('Svc');
        // WHEN
        new appmesh.VirtualNode(stack, 'test-node', {
            mesh,
            cloudMapService: service
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
            Spec: {
                ServiceDiscovery: {
                    AWSCloudMap: {
                        NamespaceName: "domain.local",
                        ServiceName: { "Fn::GetAtt": ["testnamespaceSvcB55702EC", "Name"] }
                    }
                }
            }
        }));
        test.done();
    },
    'When adding a VirtualService to a mesh': {
        'with VirtualRouter and VirtualNode as providers': {
            'should throw error'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const testNode = new appmesh.VirtualNode(stack, 'test-node', {
                    mesh,
                    dnsHostName: 'test-node',
                });
                const testRouter = mesh.addVirtualRouter('router', {
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    }
                });
                // THEN
                test.throws(() => {
                    mesh.addVirtualService('service', {
                        virtualServiceName: 'test-service.domain.local',
                        virtualNode: testNode,
                        virtualRouter: testRouter,
                    });
                });
                test.done();
            },
        },
        'with single virtual router provider resource': {
            'should create service'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const testRouter = mesh.addVirtualRouter('test-router', {
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    }
                });
                mesh.addVirtualService('service', {
                    virtualServiceName: 'test-service.domain.local',
                    virtualRouter: testRouter,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::VirtualService', {
                    Spec: {
                        Provider: {
                            VirtualRouter: {
                                VirtualRouterName: {
                                    'Fn::GetAtt': ['meshtestrouterF78D72DD', 'VirtualRouterName'],
                                },
                            },
                        },
                    },
                }));
                test.done();
            },
        },
        'with single virtual node provider resource': {
            'should create service'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const node = mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                });
                mesh.addVirtualService('service2', {
                    virtualServiceName: 'test-service.domain.local',
                    virtualNode: node,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::VirtualService', {
                    Spec: {
                        Provider: {
                            VirtualNode: {
                                VirtualNodeName: {
                                    'Fn::GetAtt': ['meshtestnodeF93946D4', 'VirtualNodeName'],
                                },
                            },
                        },
                    },
                }));
                test.done();
            },
        },
    },
    'When adding a VirtualNode to a mesh': {
        'with empty default listeners and backends': {
            'should create default resource'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::VirtualNode', {
                    MeshName: {
                        'Fn::GetAtt': ['meshACDFE68E', 'MeshName'],
                    },
                    Spec: {
                        // Specifically: no Listeners and Backends
                        Logging: {
                            AccessLog: {
                                File: {
                                    Path: "/dev/stdout"
                                }
                            }
                        },
                        ServiceDiscovery: {
                            DNS: {
                                Hostname: "test.domain.local"
                            }
                        }
                    },
                }));
                test.done();
            },
        },
        'with added listeners': {
            'should create listener resource'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
                    MeshName: {
                        'Fn::GetAtt': ['meshACDFE68E', 'MeshName'],
                    },
                    Spec: {
                        Listeners: [
                            {
                                PortMapping: {
                                    Port: 8080,
                                    Protocol: 'http',
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
        'with added listeners with healthchecks': {
            'should create healthcheck resource'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                        healthCheck: {
                            healthyThreshold: 3,
                            path: '/',
                            interval: cdk.Duration.seconds(5),
                            timeout: cdk.Duration.seconds(2),
                            unhealthyThreshold: 2,
                        },
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
                    MeshName: {
                        'Fn::GetAtt': ['meshACDFE68E', 'MeshName'],
                    },
                    Spec: {
                        Listeners: [
                            {
                                HealthCheck: {
                                    HealthyThreshold: 3,
                                    IntervalMillis: 5000,
                                    Path: '/',
                                    Port: 8080,
                                    Protocol: 'http',
                                    TimeoutMillis: 2000,
                                    UnhealthyThreshold: 2,
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
        'with backends': {
            'should create resource with service backends'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const service1 = new appmesh.VirtualService(stack, 'service-1', {
                    virtualServiceName: 'service1.domain.local',
                    mesh,
                });
                mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                    backends: [
                        service1,
                    ],
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
                    Spec: {
                        Backends: [
                            {
                                VirtualService: {
                                    VirtualServiceName: {
                                        'Fn::GetAtt': ['service1A48078CF', 'VirtualServiceName'],
                                    },
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
    },
    'Can construct a mesh from a name'(test) {
        // WHEN
        const stack2 = new cdk.Stack();
        const mesh2 = appmesh.Mesh.fromMeshName(stack2, 'imported-mesh', 'abc');
        mesh2.addVirtualService('service', {
            virtualServiceName: 'test.domain.local',
        });
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualService', {
            MeshName: 'abc',
            Spec: {},
            VirtualServiceName: 'test.domain.local',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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