"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualNodeBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.backends = new Array();
        this.listeners = new Array();
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackends(...props) {
        for (const s of props) {
            this.backends.push({
                virtualService: {
                    virtualServiceName: s.virtualServiceName,
                },
            });
        }
    }
    /**
     * Utility method to add an inbound listener for this virtual node
     */
    addListeners(...listeners) {
        if (this.listeners.length + listeners.length > 1) {
            throw new Error('VirtualNode may have at most one listener');
        }
        for (const listener of listeners) {
            const portMapping = listener.portMapping || { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP };
            this.listeners.push({
                portMapping,
                healthCheck: renderHealthCheck(listener.healthCheck, portMapping),
            });
        }
    }
}
/**
 * Minimum and maximum thresholds for HeathCheck numeric properties
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_HealthCheckPolicy.html
 */
const HEALTH_CHECK_PROPERTY_THRESHOLDS = {
    healthyThreshold: [2, 10],
    intervalMillis: [5000, 300000],
    port: [1, 65535],
    timeoutMillis: [2000, 60000],
    unhealthyThreshold: [2, 10],
};
function renderHealthCheck(hc, pm) {
    if (hc === undefined) {
        return undefined;
    }
    if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
        throw new Error('The path property cannot be set with Protocol.TCP');
    }
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || (hc.protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
        port: hc.port || pm.port,
        protocol: hc.protocol || pm.protocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    Object.keys(healthCheck)
        .filter((key) => HEALTH_CHECK_PROPERTY_THRESHOLDS[key] &&
        typeof healthCheck[key] === 'number' &&
        !cdk.Token.isUnresolved(healthCheck[key])).map((key) => {
        const [min, max] = HEALTH_CHECK_PROPERTY_THRESHOLDS[key];
        const value = healthCheck[key];
        if (value < min) {
            throw new Error(`The value of '${key}' is below the minimum threshold (expected >=${min}, got ${value})`);
        }
        if (value > max) {
            throw new Error(`The value of '${key}' is above the maximum threshold (expected <=${max}, got ${value})`);
        }
    });
    return healthCheck;
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
class VirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId })
        });
        this.mesh = props.mesh;
        this.addBackends(...props.backends || []);
        this.addListeners(...props.listener ? [props.listener] : []);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                backends: cdk.Lazy.anyValue({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.anyValue({ produce: () => this.listeners }, { omitEmptyArray: true }),
                serviceDiscovery: {
                    dns: props.dnsHostName !== undefined ? { hostname: props.dnsHostName } : undefined,
                    awsCloudMap: props.cloudMapService !== undefined ? {
                        serviceName: props.cloudMapService.serviceName,
                        namespaceName: props.cloudMapService.namespace.namespaceName,
                        attributes: renderAttributes(props.cloudMapServiceInstanceAttributes)
                    } : undefined,
                },
                logging: {
                    accessLog: {
                        file: {
                            path: '/dev/stdout',
                        },
                    },
                },
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new ImportedVirtualNode(scope, id, { virtualNodeArn });
    }
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeName(scope, id, meshName, virtualNodeName) {
        return new ImportedVirtualNode(scope, id, {
            meshName,
            virtualNodeName
        });
    }
}
exports.VirtualNode = VirtualNode;
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
/**
 * Used to import a VirtualNode and read it's properties
 */
class ImportedVirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualNodeArn) {
            this.virtualNodeArn = props.virtualNodeArn;
            this.virtualNodeName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualNodeArn).resourceName));
        }
        else if (props.virtualNodeName && props.meshName) {
            this.virtualNodeName = props.virtualNodeName;
            this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualNode`,
                resourceName: this.virtualNodeName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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