from typing import Iterable

# isort: split
from ansys.meshing.prime.autogen.surfer import Surfer as _Surfer

# isort: split
from ansys.meshing.prime.autogen.coreobject import CoreObject
from ansys.meshing.prime.autogen.surferstructs import (
    LocalSurferParams,
    LocalSurferResults,
    SurferParams,
    SurferResults,
)
from ansys.meshing.prime.core.model import Model


class Surfer(CoreObject):
    __doc__ = _Surfer.__doc__

    def __init__(self, model: Model):
        """__init__(Surfer self, Model model)"""
        self._model = model

    def mesh_topo_faces(
        self, part_id: int, topo_faces: Iterable[int], params: SurferParams
    ) -> SurferResults:
        """Performs meshing on the given topofaces with provided parameters.

        Parameters
        ----------
        part_id : int
            Id of part.
        topo_faces : Iterable[int]
            Ids of topofaces.
        params : SurferParams
            Surfer Parameters.

        Returns
        -------
        SurferResults
            Returns the SurferResults.

        Examples
        --------
        >>> surfer = prime.Surfer(model)
        >>> surfer_params = prime.SurferParams(model)
        >>> results = surfer.mesh_topo_faces(part_id, topo_faces, surfer_params)

        """
        with _Surfer(model=self._model, part_id=part_id) as surfer:
            return surfer.mesh_topo_faces(topo_faces, params)

    def remesh_face_zonelets_locally(
        self,
        part_id: int,
        face_zonelets: Iterable[int],
        register_id: int,
        local_surfer_params: LocalSurferParams,
    ) -> LocalSurferResults:
        """Remesh the given face zonelets locally at the registered faces with provided parameters.

        Parameters
        ----------
        part_id : int
            Id of part.
        face_zonelets : Iterable[int]
            Ids of face zonelets.
        register_id : int
            Register id of the target faces.
        local_surfer_params : LocalSurferParams
            Local surfer Parameters.

        Returns
        -------
        LocalSurferResults
            Returns the LocalSurferResults.

        Examples
        --------
        >>> surfer = prime.Surfer(model)
        >>> local_surfer_params = prime.LocalSurferParams(model)
        >>> results = surfer.remesh_face_zonelets_locally(part_id,
                        face_zonelets, register_id, local_surfer_params)

        """
        with _Surfer(model=self._model, part_id=part_id) as surfer:
            return surfer.remesh_face_zonelets_locally(
                face_zonelets, register_id, local_surfer_params
            )

    def remesh_face_zonelets(
        self,
        part_id: int,
        face_zonelets: Iterable[int],
        edge_zonelets: Iterable[int],
        params: SurferParams,
    ) -> SurferResults:
        """Performs meshing on the given face zonelets with provided parameters.

        Parameters
        ----------
        part_id : int
            Id of part.
        face_zonelets : Iterable[int]
            Ids of face zonelets.
        edge_zonelets : Iterable[int]
            Ids of edge zonelets.
        params : SurferParams
            Surfer parameters.

        Returns
        -------
        SurferResults
            Returns the SurferResults.

        Examples
        --------
        >>> surfer = prime.Surfer(model)
        >>> surfer_params = prime.SurferParams(model)
        >>> results = surfer.remesh_face_zonelets(part_id,
                        face_zonelets, edge_zonelets, surfer_params)

        """
        with _Surfer(model=self._model, part_id=part_id) as surfer:
            return surfer.remesh_face_zonelets(face_zonelets, edge_zonelets, params)

    def initialize_surfer_params_for_wrapper(self) -> SurferParams:
        """Initialize surfer parameters to mesh surfaces generated by wrapper.


        Returns
        -------
        SurferParams
            Returns the SurferParams initialized for wrapper inputs.

        Examples
        --------
        >>> surfer = prime.Surfer(model)
        >>> surfer_params = surfer.initialize_surfer_params_for_wrapper()

        """
        with _Surfer(model=self._model, part_id=-1) as surfer:
            return surfer.initialize_surfer_params_for_wrapper()

    def __enter__(self):
        return self

    def __exit__(self, type, value, traceback):
        pass
