"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const cdk8s = require("cdk8s");
const fs = require("fs");
const path = require("path");
const minimatch = require("minimatch");
const utils_1 = require("./utils");
/**
 * ConfigMap holds configuration data for pods to consume.
 */
class ConfigMap extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, props);
        this._binaryData = {};
        this._data = {};
        this.apiObject = new k8s.ConfigMap(this, 'ConfigMap', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
        });
        for (const [k, v] of Object.entries((_a = props.data) !== null && _a !== void 0 ? _a : {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries((_b = props.binaryData) !== null && _b !== void 0 ? _b : {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     * @param name The name of the config map to import
     */
    static fromConfigMapName(name) {
        return { name };
    }
    /**
     * Adds a data entry to the config map.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map. BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     * @param localFile The path to the local file
     * @param key The ConfigMap key (default to the file name).
     */
    addFile(localFile, key) {
        key = key !== null && key !== void 0 ? key : path.basename(localFile);
        const value = fs.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     * @param localDir A path to a local directory
     * @param options Options
     */
    addDirectory(localDir, options = {}) {
        var _a, _b;
        const exclude = (_a = options.exclude) !== null && _a !== void 0 ? _a : [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                if (minimatch(file, pattern)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = (_b = options.keyPrefix) !== null && _b !== void 0 ? _b : '';
        for (const file of fs.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (fs.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
//# sourceMappingURL=data:application/json;base64,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