# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_gov_notify']

package_data = \
{'': ['*']}

install_requires = \
['django>=2.2.12,<4.0', 'notifications-python-client>=6.1.0,<7.0.0']

setup_kwargs = {
    'name': 'django-gov-notify',
    'version': '0.3.0',
    'description': 'A GOV.UK Notify flavoured Django email backend',
    'long_description': '# django-gov-notify, a GOV.UK Notify flavoured Django email backend\n\n![GitHub workflow](https://github.com/nimasmi/django-gov-notify/actions/workflows/CI-tests.yml/badge.svg)\n\ndjango-gov-notify provides Django integration with the [GOV.UK Notify](https://www.notifications.service.gov.uk/) service for sending emails and SMS messages. [Python Client Documentation](https://docs.notifications.service.gov.uk/python.html).\n\nThis is implemented as a custom Django email backend. It presents a similar internal API to standard Django email backends, but with some restrictions:\n\n- GOV.UK Notify emails are sent to one recipient each. CC: and BCC: fields are not supported.\n- A single email \'message\' with multiple recipients will result in multiple individual API calls to GOV.UK Notify, each message being sent to a single recipient. The backend will still report back `1`, as per Django\'s default behaviour.\n- Attachments are not (at the moment) supported.\n- Custom headers are not supported.\n- To configure a \'reply-to\' address, you must first configure such an address in the GOV.UK Notify admin interface.\n- The \'from\' address field is not supported. This must be configured within the GOV.UK Notify admin interface.\n- Preformatted emails are expected to be configured in the service admin dashboard as Markdown templates with placeholders.\n- The email body is interpreted as very limited Markdown. On testing, it seems that variables are not interpreted as markdown, or maybe mangled, e.g. `_test_` was emailed as `*test*`.\n\n## Compatibility\n\ndjango-gov-notify supports:\n\n- Python 3.6, 3.7, 3.8, and 3.9\n- Django 2.2, 3.1 and 3.2\n\n## Installation\n\nUsing [pip](https://pip.pypa.io/en/stable/):\n\n```bash\n$ pip install django-gov-notify\n```\n\nUsing [Poetry](https://python-poetry.org/)\n\n```bash\n$ poetry add django-gov-notify\n```\n\n## Configuration\n\nIn your Django project\'s settings:\n\n```python\nEMAIL_BACKEND = "django_gov_notify.backends.NotifyEmailBackend"\n```\n\nYou will need at least one email template ID, with a plain template:\n\n> Subject: ((subject))  \n> Body: ((body))\n\nSet the Django settings:\n\n- `GOVUK_NOTIFY_API_KEY` (NB _not_ GOV_UK…)\n- `GOVUK_NOTIFY_PLAIN_EMAIL_TEMPLATE_ID`\n\nThis plain template ID setting, and template IDs passed to the NotifyEmailMessage class, use string representations of the UUID keys.\n\n## Usage\n\n### Sending an email using a template\n\nConfigure the template in the [GOV.UK Notify dashboard](https://www.notifications.service.gov.uk/):\n\n> Subject: Message about ((topic))  \n> Body: Hello ((first name)), your reference is ((ref number))\n\nCreate an email message, supplying the template ID and a `personalisation` dictionary (this should also include any variables defined in the template subject):\n\n```python\nfrom django_gov_notify.message import NotifyEmailMessage\n\nmessage = NotifyEmailMessage(\n    to=["recipient@example.com"],\n    template_id="43573f75-80e7-402f-b308-e5f1066fbd6f",\n    personalisation={\n        "topic": "The Prisoner",\n        "first name": "Patrick",\n        "ref number": "6",\n    },\n)\nmessage.send()\n```\n\nNote that in this case a subject and body are not required, nor permitted, because the personalisation dict won\'t know how to do anything with them.\n\n### Sending an email using the default (blank) template\n\nThis assumes you have configured a blank template with the parameters\n\n> Subject: ((subject))  \n> Body: ((body))\n\n```python\nfrom django_gov_notify.message import NotifyEmailMessage\n\nmessage = NotifyEmailMessage(\n    subject="Test subject", body="Test message content", to=["recipient@example.com"]\n)\nmessage.send()\n```\n\nNote that in this case a subject and body are required, and you must not pass the `template_id` or `personalisation` kwargs.\n\n### Sending an email using the `send_mail` shortcut function\n\nUse it in the normal fashion, including a \'from\' address that will be discarded:\n\n```python\nfrom django.utils.mail import send_mail\n\nsend_mail("Subject", "Message content", "from@example.com", ["recipient@example.com"])\n```\n\nThis will use the blank template ID configured as `settings.GOVUK_NOTIFY_PLAIN_EMAIL_TEMPLATE_ID`. Attachments, custom headers, and BCC recipients are not supported.\n\n## Contributing\n\nTo work on this repository locally:\n\n- install: `poetry install`\n- run tests: `poetry run python runtests.py`\n',
    'author': 'Nick Smith',
    'author_email': 'nick.smith@torchbox.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/nimasmi/django-gov-notify',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<3.10',
}


setup(**setup_kwargs)
