"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SnsPublish = exports.MessageAttributeDataType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The data type set for the SNS message attributes
 *
 * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
 */
var MessageAttributeDataType;
(function (MessageAttributeDataType) {
    /**
     * Strings are Unicode with UTF-8 binary encoding
     */
    MessageAttributeDataType["STRING"] = "String";
    /**
     * An array, formatted as a string
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["STRING_ARRAY"] = "String.Array";
    /**
     * Numbers are positive or negative integers or floating-point numbers
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["NUMBER"] = "Number";
    /**
     * Binary type attributes can store any binary data
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["BINARY"] = "Binary";
})(MessageAttributeDataType = exports.MessageAttributeDataType || (exports.MessageAttributeDataType = {}));
/**
 * A Step Functions Task to publish messages to SNS topic.
 *
 */
class SnsPublish extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _b;
        super(scope, id, props);
        this.props = props;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_SnsPublishProps(props);
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SnsPublish.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.message)) {
                throw new Error('Task Token is required in `message` Use JsonPath.taskToken to set the token.');
            }
        }
        this.taskPolicies = [
            new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [this.props.topic.topicArn],
            }),
        ];
    }
    /**
     * Provides the SNS Publish service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sns', 'publish', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                TopicArn: this.props.topic.topicArn,
                Message: this.props.message.value,
                MessageStructure: this.props.messagePerSubscriptionType ? 'json' : undefined,
                MessageAttributes: renderMessageAttributes(this.props.messageAttributes),
                Subject: this.props.subject,
            }),
        };
    }
}
exports.SnsPublish = SnsPublish;
_a = JSII_RTTI_SYMBOL_1;
SnsPublish[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SnsPublish", version: "1.151.0" };
SnsPublish.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderMessageAttributes(attributes) {
    if (attributes === undefined) {
        return undefined;
    }
    const renderedAttributes = {};
    Object.entries(attributes).map(([key, val]) => {
        renderedAttributes[key] = renderMessageAttributeValue(val);
    });
    return sfn.TaskInput.fromObject(renderedAttributes).value;
}
function renderMessageAttributeValue(attribute) {
    const dataType = attribute.dataType;
    if (attribute.value instanceof sfn.TaskInput) {
        return {
            DataType: dataType !== null && dataType !== void 0 ? dataType : MessageAttributeDataType.STRING,
            StringValue: dataType !== MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
            BinaryValue: dataType === MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
        };
    }
    if (dataType === MessageAttributeDataType.BINARY) {
        return { DataType: dataType, BinaryValue: `${attribute.value}` };
    }
    if (core_1.Token.isUnresolved(attribute.value)) {
        return { DataType: dataType !== null && dataType !== void 0 ? dataType : MessageAttributeDataType.STRING, StringValue: attribute.value };
    }
    validateMessageAttribute(attribute);
    if (Array.isArray(attribute.value)) {
        return { DataType: MessageAttributeDataType.STRING_ARRAY, StringValue: JSON.stringify(attribute.value) };
    }
    const value = attribute.value;
    if (typeof value === 'number') {
        return { DataType: MessageAttributeDataType.NUMBER, StringValue: `${value}` };
    }
    else {
        return { DataType: MessageAttributeDataType.STRING, StringValue: `${value}` };
    }
}
function validateMessageAttribute(attribute) {
    const dataType = attribute.dataType;
    const value = attribute.value;
    if (dataType === undefined) {
        return;
    }
    if (Array.isArray(value)) {
        if (dataType !== MessageAttributeDataType.STRING_ARRAY) {
            throw new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type Array`);
        }
        const validArrayTypes = ['string', 'boolean', 'number'];
        value.forEach((v) => {
            if (v !== null || !validArrayTypes.includes(typeof v)) {
                throw new Error(`Requested SNS message attribute type was ${typeof value} but Array values must be one of ${validArrayTypes}`);
            }
        });
        return;
    }
    const error = new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type ${typeof value}`);
    switch (typeof value) {
        case 'string':
            // trust the user or will default to string
            if (sfn.JsonPath.isEncodedJsonPath(attribute.value)) {
                return;
            }
            if (dataType === MessageAttributeDataType.STRING ||
                dataType === MessageAttributeDataType.BINARY) {
                return;
            }
            throw error;
        case 'number':
            if (dataType === MessageAttributeDataType.NUMBER) {
                return;
            }
            throw error;
        case 'boolean':
            if (dataType === MessageAttributeDataType.STRING) {
                return;
            }
            throw error;
        default:
            throw error;
    }
}
//# sourceMappingURL=data:application/json;base64,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