import pydicom
import os
import csv
from copy import deepcopy
from abc import ABC, abstractmethod
from typing import Dict, Optional
from idiscore.core import Core, Profile
from idiscore.defaults import get_dicom_rule_sets


class Pseudonymizer(ABC):
    @abstractmethod
    def pseudonimize(self, ds: pydicom.Dataset) -> pydicom.Dataset:
        """Pseudonymize a DICOM dataset.

        Args
        ----
        dataset
            The DICOM dataset to pseudonymize

        Returns
        -------
        Dataset
            The pseudonymized dataset
        """
        pass


class InferredPseudonymizer(Pseudonymizer):
    def __init__(self, mapping: Dict[str, str], output_dir: Optional[str] = None):
        """Pseudonymizer that infers rules from examples.

        Attributes
        ----------
        mapping: Dict
            Mapping of original IDs to pseudonymized IDs
        output_dir: str
            Directory containing pseudonymized files
        """
        self.mapping = mapping
        self.output_dir = output_dir

    def pseudonimize(self, ds: pydicom.Dataset) -> pydicom.Dataset:
        """Return the pre-pseudonymized version of the dataset based on the mapping.

        Args
        ----
        ds: pydicom.Dataset
            Original DICOM dataset

        Returns
        -------
        pydicom.Dataset
            Pre-pseudonymized DICOM dataset from the mapped file
        """
        # Get the SOPInstanceUID from the dataset
        try:
            sop_instance_uid = ds.SOPInstanceUID
        except AttributeError as err:
            raise ValueError("Dataset does not contain SOPInstanceUID tag") from err

        if sop_instance_uid not in self.mapping:
            raise ValueError(
                f"SOPInstanceUID {sop_instance_uid} not found in pseudonymizer mapping"
            )

        target_path = self.mapping[sop_instance_uid]

        # If output_dir is provided and target_path is relative, join them
        if self.output_dir and not os.path.isabs(target_path):
            target_path = os.path.join(self.output_dir, target_path)

        if not os.path.exists(target_path):
            raise FileNotFoundError(
                f"Mapped pseudonymized file does not exist: {target_path}. "
                f"Check if the output_dir parameter is set correctly: {self.output_dir}"
            )

        return pydicom.dcmread(target_path)

    @classmethod
    def from_csv(
        cls,
        csv_path: str,
        output_dir: Optional[str] = None,
        id_col: str = "SOPInstanceUID",
        target_col: str = "path_deidentified",
    ) -> "InferredPseudonymizer":
        """Create pseudonymizer from a CSV mapping file.

        Args
        ----
        csv_file: str
            Path to CSV file with mappings
        output_dir: str, optional
            Directory containing pseudonymized files

        Returns
        -------
        InferredPseudonymizer
            New pseudonymizer instance
        """
        mapping = {}

        with open(csv_path, "r") as csvfile:
            reader = csv.DictReader(csvfile)
            for row in reader:
                sop_instance_uid = row[id_col].strip()
                target_path = row[target_col].strip()
                mapping[sop_instance_uid] = target_path

        return cls(mapping, output_dir)


class IDISPseudonymizer(Pseudonymizer):

    def pseudonimize(self, ds: pydicom.Dataset) -> pydicom.Dataset:
        """Implementation of the IDIS pseudonymization algorithm.

        Attributes
        ----------
        uid_map: Dict
            Mapping of original UIDs to pseudonymized UIDs
        """
        sets = get_dicom_rule_sets()
        profile = Profile(rule_sets=[sets.basic_profile])
        core = Core(profile)
        ds_copy = deepcopy(ds)
        pseudonymized_ds = core.deidentify(ds_copy)
        return pseudonymized_ds
