# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pard']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pard',
    'version': '0.1.7',
    'description': '',
    'long_description': '## PARD (Physicochemical Amino acid Replacement Distances)\n[![DOI](https://zenodo.org/badge/526882024.svg)](https://zenodo.org/badge/latestdoi/526882024)\n![Coverage](\nhttps://raw.githubusercontent.com/MICS-Lab/pard/a0633c0b5e95bff565c6b9f92811b8b7bc0c8a6b/readme_resources/coverage.svg\n)\n\n### Overview\nAmino acid replacements (also referred to as substitutions) are changes from one amino acid to a different one in a\nprotein - and there are different ways to assess the difference between the two amino acids that come into play in an\namino acid replacement.\n\nOne can look at one or several properties\' discrepancy between two amino acids.\nTo list only a few: polarity\n[[1]](https://www.science.org/doi/10.1126/science.185.4154.862) [[3]](https://www.nature.com/articles/215355a0) \n[[4]](https://link.springer.com/article/10.1007/BF01732340),\nmolecular volume / relative sizes of amino-acid side chains \n[[1]](https://www.science.org/doi/10.1126/science.185.4154.862) [[3]](https://www.nature.com/articles/215355a0)\n[[4]](https://link.springer.com/article/10.1007/BF01732340), \nmean number of lone pair electrons on the side-chain\n[[2]](https://www.sciencedirect.com/science/article/abs/pii/0022519366901123),\nmaximum moment of inertia for rotation at the α―β bond\n[[2]](https://www.sciencedirect.com/science/article/abs/pii/0022519366901123)\nor at the β―γ bond [[2]](https://www.sciencedirect.com/science/article/abs/pii/0022519366901123)\nor at the γ―δ bond [[2]](https://www.sciencedirect.com/science/article/abs/pii/0022519366901123), \npresence of a pyrrolidine ring (proline (P))\n[[2]](https://www.sciencedirect.com/science/article/abs/pii/0022519366901123), \nexperimental protein activity and stability after exchanging one amino acid into another\n[[5]](https://pubmed.ncbi.nlm.nih.gov/15944362/),\netc.\n\n`pard` is a package designed to make those **P**hysicochemical **A**mino acid **R**eplacement **D**istances calculations\nstraightforward with Python.\n\nOne typical use is to establish a \'distance\' between a wild-type protein and its mutated version.\n\n\n### Getting started\n#### Install from PyPI (recommended)\nTo use `pard`, run `pip install pard` in your terminal.\n\n#### Usage\n```py\nimport pard\n\namino_acid_1: str = "W"  # Tryptophan\namino_acid_2: str = "L"  # Leucine\ndistance_grantham: int = pard.grantham.grantham(amino_acid_1, amino_acid_2)\ndistance_sneath: int   = pard.sneath.sneath(amino_acid_1, amino_acid_2)\n\nprint(distance_grantham)\nprint(distance_sneath)\n```\nor equivalently\n```py\nfrom pard.grantham import grantham\nfrom pard.sneath import sneath\n\namino_acid_1: str = "W"  # Tryptophan\namino_acid_2: str = "L"  # Leucine\ndistance_grantham: int = grantham(amino_acid_1, amino_acid_2)\ndistance_sneath: int   = sneath(amino_acid_1, amino_acid_2)\n\nprint(distance_grantham)\nprint(distance_sneath)\n```\nwill output:\n```\n>>> 61\n>>> 30\n```\nwhich are the correct values \n[[1]](https://www.science.org/doi/10.1126/science.185.4154.862)\n[[2]](https://www.sciencedirect.com/science/article/abs/pii/0022519366901123).\n\n#### Unit tests\n```\nName                                         Stmts   Miss  Cover\n----------------------------------------------------------------\npard\\__init__.py                                 3      0   100%\npard\\epstein.py                                  5      0   100%\npard\\experimental_exchangeability.py             8      1    88%\npard\\grantham.py                                 3      0   100%\npard\\miyata.py                                   3      0   100%\npard\\raw_python_dictionaries.py                 35      0   100%\npard\\sneath.py                                   3      0   100%\ntests\\__init__.py                                0      0   100%\ntests\\test_epstein.py                           34      0   100%\ntests\\test_experimental_exchangeability.py      22      0   100%\ntests\\test_grantham.py                          18      0   100%\ntests\\test_miyata.py                            18      0   100%\ntests\\test_pard.py                               3      0   100%\ntests\\test_raw_python_dictionaries.py           12      0   100%\ntests\\test_sneath.py                            18      0   100%\n----------------------------------------------------------------\nTOTAL                                          185      1    99%\n```\n\n\n### About the source code\n- Follows [PEP8](https://peps.python.org/pep-0008/) Style Guidelines.\n- All variables are correctly type-hinted, reviewed with [static type checker](https://mypy.readthedocs.io/en/stable/)\n`mypy`.\n\n\n### Useful links:\n- [Corresponding GitHub repository](https://github.com/MICS-Lab/pard)\n- [Corresponding PyPI page](https://pypi.org/project/pard/)\n\n\n### Citation\nIf you use this software, please cite it as below.\n\n- APA:\n\n`Lhotte, R. (2022). Physicochemical Amino acid Replacement Distances (PARD) package (Version 0.1.7) [Computer software].\nhttps://doi.org/10.5281/zenodo.7013169`\n\n- BibTeX:\n\n```\n@software{lhotte_pard_2022,\nauthor = {Lhotte, Romain},\ndoi = {10.5281/zenodo.7013169},\nmonth = {8},\ntitle = {{Physicochemical Amino acid Replacement Distances (PARD) package}},\nversion = {0.1.7},\nyear = {2022}\n}\n```\n\n\n### References\n- [1] Grantham, R., 1974. Amino acid difference formula to help explain protein evolution. science, 185(4154), \npp.862-864.\n- [2] Sneath, P.H.A., 1966. Relations between chemical structure and biological activity in peptides. Journal of\ntheoretical biology, 12(2), pp.157-195.\n- [3] Epstein, C.J., 1967. Non-randomness of ammo-acid changes in the evolution of homologous proteins. Nature,\n215(5099), pp.355-359.\n- [4] Miyata, T., Miyazawa, S. and Yasunaga, T., 1979. Two types of amino acid substitutions in protein evolution. \nJournal of molecular evolution, 12(3), pp.219-236.\n- [5] Yampolsky, L.Y. and Stoltzfus, A., 2005. The exchangeability of amino acids in proteins. Genetics, 170(4), \npp.1459-1472.\n',
    'author': 'JasonMendoza2008',
    'author_email': 'lhotteromain@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
