"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const cognito = require("@aws-cdk/aws-cognito");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('snapshot test buildUserPool default params', () => {
    const stack = new core_1.Stack();
    defaults.buildUserPool(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('snapshot test buildUserPoolClient default params', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    defaults.buildUserPoolClient(stack, userpool);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test override for buildUserPool', () => {
    const stack = new core_1.Stack();
    const userpoolProps = {
        userPoolName: 'test',
        signInType: cognito.SignInType.EMAIL_OR_PHONE
    };
    defaults.buildUserPool(stack, userpoolProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPool', {
        LambdaConfig: {},
        UsernameAttributes: [
            "email",
            "phone_number"
        ],
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test"
    });
});
test('Test override for buildUserPoolClient', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclientProps = {
        userPoolClientName: 'test',
        userPool: userpool
    };
    defaults.buildUserPoolClient(stack, userpool, userpoolclientProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
        UserPoolId: {
            Ref: "CognitoUserPool53E37E69"
        },
        ClientName: "test"
    });
});
test('Test override for buildIdentityPool', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    defaults.buildIdentityPool(stack, userpool, userpoolclient, {
        allowUnauthenticatedIdentities: true
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPool', {
        AllowUnauthenticatedIdentities: true,
        CognitoIdentityProviders: [
            {
                ClientId: {
                    Ref: "CognitoUserPoolClient5AB59AE4"
                },
                ProviderName: {
                    "Fn::GetAtt": [
                        "CognitoUserPool53E37E69",
                        "ProviderName"
                    ]
                },
                ServerSideTokenCheck: true
            }
        ]
    });
});
test('Test setupCognitoForElasticSearch', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    defaults.setupCognitoForElasticSearch(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain"
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPoolRoleAttachment', {
        IdentityPoolId: {
            Ref: "CognitoIdentityPool"
        },
        Roles: {
            authenticated: {
                "Fn::GetAtt": [
                    "CognitoAuthorizedRole14E74FE0",
                    "Arn"
                ]
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRoleWithWebIdentity",
                    Condition: {
                        "StringEquals": {
                            "cognito-identity.amazonaws.com:aud": {
                                Ref: "CognitoIdentityPool"
                            }
                        },
                        "ForAnyValue:StringLike": {
                            "cognito-identity.amazonaws.com:amr": "authenticated"
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        Federated: "cognito-identity.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "es:ESHttp*",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:es:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":domain/test-domain/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "CognitoAccessPolicy"
            }
        ]
    });
});
//# sourceMappingURL=data:application/json;base64,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