"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
const s3_bucket_helper_1 = require("../lib/s3-bucket-helper");
require("@aws-cdk/assert/jest");
test('cloudfront distribution with default params', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {
        deployBucket: true
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check bucket policy metadata', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {
        deployBucket: true
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResource('AWS::S3::BucketPolicy', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "F16",
                        reason: "Public website bucket policy requires a wildcard principal"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('check bucket metadata', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {
        deployBucket: true
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W35",
                        reason: "This S3 bucket is used as the access logging bucket for CloudFront Distribution"
                    },
                    {
                        id: "W51",
                        reason: "This S3 bucket is used as the access logging bucket for CloudFront Distribution"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('test cloudfront check bucket policy', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {
        deployBucket: true
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResourceLike("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "s3:GetObject*",
                        "s3:GetBucket*",
                        "s3:List*"
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::cloudfront:user/CloudFront Origin Access Identity ",
                                    {
                                        Ref: "CloudFrontOriginAccessIdentity"
                                    }
                                ]
                            ]
                        }
                    },
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "S3Bucket07682993",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "S3Bucket07682993",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                },
                {
                    Action: "s3:GetObject",
                    Effect: "Allow",
                    Principal: {
                        CanonicalUser: {
                            "Fn::GetAtt": [
                                "CloudFrontOriginAccessIdentity",
                                "S3CanonicalUserId"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::GetAtt": [
                                        "S3Bucket07682993",
                                        "Arn"
                                    ]
                                },
                                "/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('test cloudfront override properties', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {
        deployBucket: true
    });
    // Create CloudFront Origin Access Identity User
    const cfnOrigAccessId = new cloudfront.CfnCloudFrontOriginAccessIdentity(stack, 'CloudFrontOriginAccessIdentity1', {
        cloudFrontOriginAccessIdentityConfig: {
            comment: 'Access S3 bucket content only through CloudFront'
        }
    });
    const oaiImported = cloudfront.OriginAccessIdentity.fromOriginAccessIdentityName(stack, 'OAIImported1', cfnOrigAccessId.ref);
    const props = {
        originConfigs: [{
                s3OriginSource: {
                    s3BucketSource: sourceBucket,
                    originAccessIdentity: oaiImported
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        allowedMethods: cloudfront.CloudFrontAllowedMethods.ALL,
                        cachedMethods: cloudfront.CloudFrontAllowedCachedMethods.GET_HEAD_OPTIONS
                    }]
            }]
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, props);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "DELETE",
                    "GET",
                    "HEAD",
                    "OPTIONS",
                    "PATCH",
                    "POST",
                    "PUT"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                },
                IncludeCookies: false
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "origin1",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontOriginAccessIdentity1"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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