"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployRegionalApiGateway(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    return defaults.RegionalLambdaRestApi(stack, fn);
}
test('snapshot test RegionalApiGateway default params', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test override for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        description: 'Hello World'
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            Description: "Hello World",
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "RestApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test override for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn, {
        handler: fn,
        restApiName: "HelloWorld"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "EDGE"
                ]
            },
            Name: "HelloWorld"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test ApiGateway::Account resource for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test ApiGateway::Account resource for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test default RestApi deployment w/o ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test default RestApi deployment w/ ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack, {
        restApiName: "customRestApi"
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Name: "customRestApi"
    });
});
function setupRestApi(stack, apiProps) {
    const restApi = defaults.GlobalRestApi(stack, apiProps);
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    // Setup the API Gateway Integration
    const apiGatewayIntegration = new api.AwsIntegration({
        service: "sqs",
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            requestParameters: {
                "integration.request.header.Content-Type": "'application/x-www-form-urlencoded'"
            },
            requestTemplates: {
                "application/x-www-form-urlencoded": "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")&MessageAttribute.1.Name=queryParam1&MessageAttribute.1.Value.StringValue=$input.params(\"query_param_1\")&MessageAttribute.1.Value.DataType=String"
            },
            integrationResponses: [
                {
                    statusCode: "200",
                    responseTemplates: {
                        "text/html": "Success"
                    }
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        },
        path: '12345678' + "/" + 'thisqueuequeueName'
    });
    // Setup the API Gateway method(s)
    apiGatewayResource.addMethod('POST', apiGatewayIntegration, {
        requestParameters: {
            "method.request.querystring.query_param_1": true
        },
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ]
    });
}
//# sourceMappingURL=data:application/json;base64,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