"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    // If deployLambda == false
    if (props.hasOwnProperty('deployLambda') && props.deployLambda === false) {
        if (props.existingLambdaObj) {
            return props.existingLambdaObj;
        }
        else {
            throw Error('Missing existingObj from props for deployLambda = false');
        }
        // If deployLambda == true
    }
    else {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps);
        }
        else {
            throw Error('Missing lambdaFunctionProps from props for deployLambda = true');
        }
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps) {
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, 'LambdaFunctionServiceRole', {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:aws:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let _lambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_10_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_12_X) {
        _lambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionPropsForNodeJS(lambdaServiceRole), lambdaFunctionProps);
    }
    const lambdafunction = new lambda.Function(scope, 'LambdaFunction', _lambdaFunctionProps);
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    cfnLambdafunction.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with more tighter permissions.`
                }]
        }
    };
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGFtYmRhLWhlbHBlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImxhbWJkYS1oZWxwZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILDhDQUE4QztBQUM5Qyx3Q0FBd0M7QUFDeEMsdURBQW9HO0FBQ3BHLHFDQUFxQztBQUNyQyxtQ0FBd0M7QUEwQnhDLFNBQWdCLG1CQUFtQixDQUFDLEtBQW9CLEVBQUUsS0FBK0I7SUFDdkYsdUNBQXVDO0lBQ3ZDLDJCQUEyQjtJQUMzQixJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsY0FBYyxDQUFDLElBQUksS0FBSyxDQUFDLFlBQVksS0FBSyxLQUFLLEVBQUU7UUFDeEUsSUFBSSxLQUFLLENBQUMsaUJBQWlCLEVBQUU7WUFDM0IsT0FBTyxLQUFLLENBQUMsaUJBQWlCLENBQUM7U0FDaEM7YUFBTTtZQUNMLE1BQU0sS0FBSyxDQUFDLHlEQUF5RCxDQUFDLENBQUM7U0FDeEU7UUFDRCwwQkFBMEI7S0FDM0I7U0FBTTtRQUNMLElBQUksS0FBSyxDQUFDLG1CQUFtQixFQUFFO1lBQzdCLE9BQU8sb0JBQW9CLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1NBQy9EO2FBQU07WUFDTCxNQUFNLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO1NBQy9FO0tBQ0Y7QUFDSCxDQUFDO0FBakJELGtEQWlCQztBQUVELFNBQWdCLG9CQUFvQixDQUFDLEtBQW9CLEVBQUUsbUJBQXlDO0lBQ2xHLHdDQUF3QztJQUN4QyxNQUFNLGlCQUFpQixHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsMkJBQTJCLEVBQUU7UUFDekUsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1FBQzNELGNBQWMsRUFBRTtZQUNkLCtCQUErQixFQUFFLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQztnQkFDdEQsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUNuQyxPQUFPLEVBQUU7NEJBQ1AscUJBQXFCOzRCQUNyQixzQkFBc0I7NEJBQ3RCLG1CQUFtQjt5QkFDcEI7d0JBQ0QsU0FBUyxFQUFFLENBQUMsZ0JBQWdCLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSwwQkFBMEIsQ0FBQztxQkFDNUYsQ0FBQyxDQUFDO2FBQ0osQ0FBQztTQUNIO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsNEVBQTRFO0lBQzVFLElBQUksb0JBQW9CLEdBQUcscUJBQWEsQ0FBQyw0Q0FBMEIsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLG1CQUFtQixDQUFDLENBQUM7SUFFN0csSUFBSSxtQkFBbUIsQ0FBQyxPQUFPLEtBQUssTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQzVELG1CQUFtQixDQUFDLE9BQU8sS0FBSyxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRTtRQUMxRCxvQkFBb0IsR0FBRyxxQkFBYSxDQUFDLHFEQUFtQyxDQUFDLGlCQUFpQixDQUFDLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztLQUNySDtJQUVELE1BQU0sY0FBYyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztJQUUxRixNQUFNLGlCQUFpQixHQUFHLGNBQWMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBdUIsQ0FBQztJQUUxRixpQkFBaUIsQ0FBQyxVQUFVLENBQUMsUUFBUSxHQUFHO1FBQ3RDLE9BQU8sRUFBRTtZQUNMLGlCQUFpQixFQUFFLENBQUM7b0JBQ2hCLEVBQUUsRUFBRSxLQUFLO29CQUNULE1BQU0sRUFBRSx5TUFBeU07aUJBQ3BOLENBQUM7U0FDTDtLQUNGLENBQUM7SUFFRixPQUFPLGNBQWMsQ0FBQztBQUN4QixDQUFDO0FBeENELG9EQXdDQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDE5IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgRGVmYXVsdExhbWJkYUZ1bmN0aW9uUHJvcHMsIERlZmF1bHRMYW1iZGFGdW5jdGlvblByb3BzRm9yTm9kZUpTIH0gZnJvbSAnLi9sYW1iZGEtZGVmYXVsdHMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgb3ZlcnJpZGVQcm9wcyB9IGZyb20gJy4vdXRpbHMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkTGFtYmRhRnVuY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGNyZWF0ZSBhIG5ldyBMYW1iZGEgZnVuY3Rpb24gb3IgdXNlIGFuIGV4aXN0aW5nIExhbWJkYSBmdW5jdGlvbi5cbiAgICogSWYgc2V0IHRvIGZhbHNlLCB5b3UgbXVzdCBwcm92aWRlIGEgbGFtYmRhIGZ1bmN0aW9uIG9iamVjdCBhcyBgZXhpc3RpbmdMYW1iZGFPYmpgXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95TGFtYmRhOiBib29sZWFuLFxuICAvKipcbiAgICogRXhpc3RpbmcgaW5zdGFuY2Ugb2YgTGFtYmRhIEZ1bmN0aW9uIG9iamVjdC5cbiAgICogSWYgYGRlcGxveWAgaXMgc2V0IHRvIGZhbHNlIG9ubHkgdGhlbiB0aGlzIHByb3BlcnR5IGlzIHJlcXVpcmVkXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdMYW1iZGFPYmo/OiBsYW1iZGEuRnVuY3Rpb24sXG4gIC8qKlxuICAgKiBPcHRpb25hbCB1c2VyIHByb3ZpZGVkIHByb3BzIHRvIG92ZXJyaWRlIHRoZSBkZWZhdWx0IHByb3BzIGZvciB0aGUgTGFtYmRhIGZ1bmN0aW9uLlxuICAgKiBJZiBgZGVwbG95YCBpcyBzZXQgdG8gdHJ1ZSBvbmx5IHRoZW4gdGhpcyBwcm9wZXJ0eSBpcyByZXF1aXJlZFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHQgcHJvcHMgYXJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IGxhbWJkYUZ1bmN0aW9uUHJvcHM/OiBsYW1iZGEuRnVuY3Rpb25Qcm9wc1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRMYW1iZGFGdW5jdGlvbihzY29wZTogY2RrLkNvbnN0cnVjdCwgcHJvcHM6IEJ1aWxkTGFtYmRhRnVuY3Rpb25Qcm9wcyk6IGxhbWJkYS5GdW5jdGlvbiB7XG4gIC8vIENvbmRpdGlvbmFsIGxhbWJkYSBmdW5jdGlvbiBjcmVhdGlvblxuICAvLyBJZiBkZXBsb3lMYW1iZGEgPT0gZmFsc2VcbiAgaWYgKHByb3BzLmhhc093blByb3BlcnR5KCdkZXBsb3lMYW1iZGEnKSAmJiBwcm9wcy5kZXBsb3lMYW1iZGEgPT09IGZhbHNlKSB7XG4gICAgaWYgKHByb3BzLmV4aXN0aW5nTGFtYmRhT2JqKSB7XG4gICAgICByZXR1cm4gcHJvcHMuZXhpc3RpbmdMYW1iZGFPYmo7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IEVycm9yKCdNaXNzaW5nIGV4aXN0aW5nT2JqIGZyb20gcHJvcHMgZm9yIGRlcGxveUxhbWJkYSA9IGZhbHNlJyk7XG4gICAgfVxuICAgIC8vIElmIGRlcGxveUxhbWJkYSA9PSB0cnVlXG4gIH0gZWxzZSB7XG4gICAgaWYgKHByb3BzLmxhbWJkYUZ1bmN0aW9uUHJvcHMpIHtcbiAgICAgIHJldHVybiBkZXBsb3lMYW1iZGFGdW5jdGlvbihzY29wZSwgcHJvcHMubGFtYmRhRnVuY3Rpb25Qcm9wcyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IEVycm9yKCdNaXNzaW5nIGxhbWJkYUZ1bmN0aW9uUHJvcHMgZnJvbSBwcm9wcyBmb3IgZGVwbG95TGFtYmRhID0gdHJ1ZScpO1xuICAgIH1cbiAgfVxufVxuXG5leHBvcnQgZnVuY3Rpb24gZGVwbG95TGFtYmRhRnVuY3Rpb24oc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzKTogbGFtYmRhLkZ1bmN0aW9uIHtcbiAgLy8gU2V0dXAgdGhlIElBTSBSb2xlIGZvciBMYW1iZGEgU2VydmljZVxuICBjb25zdCBsYW1iZGFTZXJ2aWNlUm9sZSA9IG5ldyBpYW0uUm9sZShzY29wZSwgJ0xhbWJkYUZ1bmN0aW9uU2VydmljZVJvbGUnLCB7XG4gICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgIExhbWJkYUZ1bmN0aW9uU2VydmljZVJvbGVQb2xpY3k6IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICBzdGF0ZW1lbnRzOiBbbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsXG4gICAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnXG4gICAgICAgICAgXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtgYXJuOmF3czpsb2dzOiR7Y2RrLkF3cy5SRUdJT059OiR7Y2RrLkF3cy5BQ0NPVU5UX0lEfTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvKmBdXG4gICAgICAgIH0pXVxuICAgICAgfSlcbiAgICB9XG4gIH0pO1xuXG4gIC8vIE92ZXJyaWRlIHRoZSBEZWZhdWx0RnVuY3Rpb25Qcm9wcyB3aXRoIHVzZXIgcHJvdmlkZWQgIGxhbWJkYUZ1bmN0aW9uUHJvcHNcbiAgbGV0IF9sYW1iZGFGdW5jdGlvblByb3BzID0gb3ZlcnJpZGVQcm9wcyhEZWZhdWx0TGFtYmRhRnVuY3Rpb25Qcm9wcyhsYW1iZGFTZXJ2aWNlUm9sZSksIGxhbWJkYUZ1bmN0aW9uUHJvcHMpO1xuXG4gIGlmIChsYW1iZGFGdW5jdGlvblByb3BzLnJ1bnRpbWUgPT09IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMF9YIHx8XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wcy5ydW50aW1lID09PSBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTJfWCkge1xuICAgICAgX2xhbWJkYUZ1bmN0aW9uUHJvcHMgPSBvdmVycmlkZVByb3BzKERlZmF1bHRMYW1iZGFGdW5jdGlvblByb3BzRm9yTm9kZUpTKGxhbWJkYVNlcnZpY2VSb2xlKSwgbGFtYmRhRnVuY3Rpb25Qcm9wcyk7XG4gIH1cblxuICBjb25zdCBsYW1iZGFmdW5jdGlvbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc2NvcGUsICdMYW1iZGFGdW5jdGlvbicsIF9sYW1iZGFGdW5jdGlvblByb3BzKTtcblxuICBjb25zdCBjZm5MYW1iZGFmdW5jdGlvbiA9IGxhbWJkYWZ1bmN0aW9uLm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIGxhbWJkYS5DZm5GdW5jdGlvbjtcblxuICBjZm5MYW1iZGFmdW5jdGlvbi5jZm5PcHRpb25zLm1ldGFkYXRhID0ge1xuICAgIGNmbl9uYWc6IHtcbiAgICAgICAgcnVsZXNfdG9fc3VwcHJlc3M6IFt7XG4gICAgICAgICAgICBpZDogJ1c1OCcsXG4gICAgICAgICAgICByZWFzb246IGBMYW1iZGEgZnVuY3Rpb25zIGhhcyB0aGUgcmVxdWlyZWQgcGVybWlzc2lvbiB0byB3cml0ZSBDbG91ZFdhdGNoIExvZ3MuIEl0IHVzZXMgY3VzdG9tIHBvbGljeSBpbnN0ZWFkIG9mIGFybjphd3M6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUgd2l0aCBtb3JlIHRpZ2h0ZXIgcGVybWlzc2lvbnMuYFxuICAgICAgICB9XVxuICAgIH1cbiAgfTtcblxuICByZXR1cm4gbGFtYmRhZnVuY3Rpb247XG59XG4iXX0=