"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const elasticsearch = require("@aws-cdk/aws-elasticsearch");
const elasticsearch_defaults_1 = require("./elasticsearch-defaults");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildElasticSearch(scope, domainName, options, cfnDomainProps) {
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = new iam.Role(scope, 'CognitoKibanaConfigureRole', {
        assumedBy: new iam.ServicePrincipal('es.amazonaws.com')
    });
    const cognitoKibanaConfigureRolePolicy = new iam.Policy(scope, 'CognitoKibanaConfigureRolePolicy', {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateElasticsearchDomainConfig"
                ],
                resources: [
                    options.userpool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${options.identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`
                ]
            }),
            new iam.PolicyStatement({
                actions: [
                    "iam:PassRole"
                ],
                conditions: {
                    StringLike: { 'iam:PassedToService': 'cognito-identity.amazonaws.com' }
                },
                resources: [
                    cognitoKibanaConfigureRole.roleArn
                ]
            })
        ]
    });
    cognitoKibanaConfigureRolePolicy.attachToRole(cognitoKibanaConfigureRole);
    let _cfnDomainProps = elasticsearch_defaults_1.DefaultCfnDomainProps(domainName, cognitoKibanaConfigureRole, options);
    if (cfnDomainProps) {
        _cfnDomainProps = utils_1.overrideProps(_cfnDomainProps, cfnDomainProps);
    }
    const esDomain = new elasticsearch.CfnDomain(scope, 'ElasticsearchDomain', _cfnDomainProps);
    esDomain.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W28',
                    reason: `The ES Domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific ES instance only`
                }]
        }
    };
    return esDomain;
}
exports.buildElasticSearch = buildElasticSearch;
function buildElasticSearchCWAlarms(scope) {
    // Setup CW Alarms for ES
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace'
        }),
        threshold: 2000,
        evaluationPeriods: 1,
        statistic: 'Minimum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization'
        }),
        threshold: 80,
        evaluationPeriods: 3,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure'
        }),
        threshold: 80,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization'
        }),
        threshold: 50,
        evaluationPeriods: 3,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure'
        }),
        threshold: 50,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildElasticSearchCWAlarms = buildElasticSearchCWAlarms;
//# sourceMappingURL=data:application/json;base64,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