"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cognito = require("@aws-cdk/aws-cognito");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const cognito_defaults_1 = require("./cognito-defaults");
function buildUserPool(scope, userPoolProps) {
    let cognitoUserPoolProps;
    if (userPoolProps) {
        cognitoUserPoolProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolProps, userPoolProps);
    }
    else {
        cognitoUserPoolProps = cognito_defaults_1.DefaultUserPoolProps;
    }
    const userPool = new cognito.UserPool(scope, 'CognitoUserPool', cognitoUserPoolProps);
    // Set the advancedSecurityMode to ENFORCED
    const cfnUserPool = userPool.node.findChild('Resource');
    cfnUserPool.userPoolAddOns = {
        advancedSecurityMode: 'ENFORCED'
    };
    return userPool;
}
exports.buildUserPool = buildUserPool;
function buildUserPoolClient(scope, userPool, cognitoUserPoolClientProps) {
    let userPoolClientProps;
    if (cognitoUserPoolClientProps) {
        userPoolClientProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolClientProps(userPool), cognitoUserPoolClientProps);
    }
    else {
        userPoolClientProps = cognito_defaults_1.DefaultUserPoolClientProps(userPool);
    }
    return new cognito.UserPoolClient(scope, 'CognitoUserPoolClient', userPoolClientProps);
}
exports.buildUserPoolClient = buildUserPoolClient;
function buildIdentityPool(scope, userpool, userpoolclient, identityPoolProps) {
    let cognitoIdentityPoolProps = cognito_defaults_1.DefaultIdentityPoolProps(userpoolclient.userPoolClientId, userpool.userPoolProviderName);
    if (identityPoolProps) {
        cognitoIdentityPoolProps = utils_1.overrideProps(cognitoIdentityPoolProps, identityPoolProps);
    }
    const idPool = new cognito.CfnIdentityPool(scope, 'CognitoIdentityPool', cognitoIdentityPoolProps);
    return idPool;
}
exports.buildIdentityPool = buildIdentityPool;
function setupCognitoForElasticSearch(scope, domainName, options) {
    // Create the domain for Cognito UserPool
    const userpooldomain = new cognito.CfnUserPoolDomain(scope, 'UserPoolDomain', {
        domain: domainName,
        userPoolId: options.userpool.userPoolId
    });
    userpooldomain.addDependsOn(options.userpool.node.findChild('Resource'));
    // Setup the IAM Role for Cognito Authorized Users
    const cognitoPrincipal = new iam.FederatedPrincipal('cognito-identity.amazonaws.com', {
        'StringEquals': { 'cognito-identity.amazonaws.com:aud': options.identitypool.ref },
        'ForAnyValue:StringLike': { 'cognito-identity.amazonaws.com:amr': 'authenticated' }
    }, 'sts:AssumeRoleWithWebIdentity');
    const cognitoAuthorizedRole = new iam.Role(scope, 'CognitoAuthorizedRole', {
        assumedBy: cognitoPrincipal,
        inlinePolicies: {
            CognitoAccessPolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'es:ESHttp*'
                        ],
                        resources: [`arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}/*`]
                    })
                ]
            })
        }
    });
    // Attach the IAM Role for Cognito Authorized Users
    new cognito.CfnIdentityPoolRoleAttachment(scope, 'IdentityPoolRoleMapping', {
        identityPoolId: options.identitypool.ref,
        roles: {
            authenticated: cognitoAuthorizedRole.roleArn
        }
    });
    return cognitoAuthorizedRole;
}
exports.setupCognitoForElasticSearch = setupCognitoForElasticSearch;
//# sourceMappingURL=data:application/json;base64,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