"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cloudfront = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const iam = require("@aws-cdk/aws-iam");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps) {
    let defaultprops;
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.loggingBucket) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint);
    }
    else {
        // Create the Logging Bucket
        const loggingBucket = new s3.Bucket(scope, 'CloudfrontLoggingBucket', s3_bucket_defaults_1.DefaultS3Props());
        // Extract the CfnBucket from the loggingBucket
        const loggingBucketResource = loggingBucket.node.findChild('Resource');
        // Override accessControl configuration and add metadata for the logging bucket
        loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
        loggingBucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W35',
                        reason: `This S3 bucket is used as the access logging bucket for CloudFront Distribution`
                    }, {
                        id: 'W51',
                        reason: `This S3 bucket is used as the access logging bucket for CloudFront Distribution`
                    }]
            }
        };
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket);
    }
    // Create the Cloudfront Distribution
    let cfprops = defaultprops;
    if (cloudFrontDistributionProps) {
        cfprops = utils_1.overrideProps(defaultprops, cloudFrontDistributionProps);
    }
    const cfDistribution = new cloudfront.CloudFrontWebDistribution(scope, 'CloudFrontDistribution', cfprops);
    return cfDistribution;
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, 'CloudfrontLoggingBucket', s3_bucket_defaults_1.DefaultS3Props());
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    loggingBucketResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W35',
                    reason: `This S3 bucket is used as the access logging bucket for CloudFront Distribution`
                }, {
                    id: 'W51',
                    reason: `This S3 bucket is used as the access logging bucket for CloudFront Distribution`
                }]
        }
    };
    // Create CloudFront Origin Access Identity User
    const cfnOrigAccessId = new cloudfront.CfnCloudFrontOriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        cloudFrontOriginAccessIdentityConfig: {
            comment: 'Access S3 bucket content only through CloudFront'
        }
    });
    const oaiImported = cloudfront.OriginAccessIdentity.fromOriginAccessIdentityName(scope, 'OAIImported', cfnOrigAccessId.ref);
    // Create the Cloudfront Distribution
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, oaiImported);
    let cfprops = defaultprops;
    if (cloudFrontDistributionProps) {
        cfprops = utils_1.overrideProps(defaultprops, cloudFrontDistributionProps);
    }
    const cfDistribution = new cloudfront.CloudFrontWebDistribution(scope, 'CloudFrontDistribution', cfprops);
    // Add S3 Bucket Policy to allow s3:GetObject for CloudFront Origin Access Identity User
    sourceBucket.addToResourcePolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject'],
        resources: [sourceBucket.arnForObjects('*')],
        principals: [new iam.CanonicalUserPrincipal(cfnOrigAccessId.attrS3CanonicalUserId)]
    }));
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    const sourceBucketPolicy = bucketPolicy.node.findChild('Resource');
    sourceBucketPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'F16',
                    reason: `Public website bucket policy requires a wildcard principal`
                }]
        }
    };
    return cfDistribution;
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWRmcm9udC1kaXN0cmlidXRpb24taGVscGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY2xvdWRmcm9udC1kaXN0cmlidXRpb24taGVscGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCxzREFBc0Q7QUFDdEQsc0NBQXNDO0FBRXRDLHdDQUF3QztBQUV4Qyw2REFBc0Q7QUFDdEQseUZBQW9KO0FBQ3BKLG1DQUF3QztBQUV4QyxTQUFnQixtQ0FBbUMsQ0FBQyxLQUFvQixFQUFFLFdBQXdCLEVBQUUsMkJBQ2pEO0lBRS9DLElBQUksWUFBWSxDQUFDO0lBRWpCLElBQUksMkJBQTJCLElBQUksMkJBQTJCLENBQUMsYUFBYSxFQUFFO1FBQzFFLFlBQVksR0FBRyxxRkFBa0QsQ0FBQyxXQUFXLENBQUMsQ0FBQztLQUNsRjtTQUFNO1FBQ0gsNEJBQTRCO1FBQzVCLE1BQU0sYUFBYSxHQUFjLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUseUJBQXlCLEVBQUUsbUNBQWMsRUFBRSxDQUFDLENBQUM7UUFFbkcsK0NBQStDO1FBQy9DLE1BQU0scUJBQXFCLEdBQUcsYUFBYSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFpQixDQUFDO1FBRXZGLCtFQUErRTtRQUMvRSxxQkFBcUIsQ0FBQyxtQkFBbUIsQ0FBQyxlQUFlLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUMvRSxxQkFBcUIsQ0FBQyxVQUFVLENBQUMsUUFBUSxHQUFHO1lBQ3hDLE9BQU8sRUFBRTtnQkFDTCxpQkFBaUIsRUFBRSxDQUFDO3dCQUNoQixFQUFFLEVBQUUsS0FBSzt3QkFDVCxNQUFNLEVBQUUsaUZBQWlGO3FCQUM1RixFQUFFO3dCQUNDLEVBQUUsRUFBRSxLQUFLO3dCQUNULE1BQU0sRUFBRSxpRkFBaUY7cUJBQzVGLENBQUM7YUFDTDtTQUNKLENBQUM7UUFFRixZQUFZLEdBQUcscUZBQWtELENBQUMsV0FBVyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0tBQ2pHO0lBRUQscUNBQXFDO0lBQ3JDLElBQUksT0FBTyxHQUFHLFlBQVksQ0FBQztJQUMzQixJQUFJLDJCQUEyQixFQUFFO1FBQzdCLE9BQU8sR0FBRyxxQkFBYSxDQUFDLFlBQVksRUFBRSwyQkFBMkIsQ0FBQyxDQUFDO0tBQ3RFO0lBQ0QsTUFBTSxjQUFjLEdBQXlDLElBQUksVUFBVSxDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSx3QkFBd0IsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUVoSixPQUFPLGNBQWMsQ0FBQztBQUMxQixDQUFDO0FBdkNELGtGQXVDQztBQUVELFNBQWdCLDJCQUEyQixDQUFDLEtBQW9CLEVBQUUsWUFBdUIsRUFBRSwyQkFDQTtJQUN2Riw0QkFBNEI7SUFDNUIsTUFBTSxhQUFhLEdBQWMsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSx5QkFBeUIsRUFBRSxtQ0FBYyxFQUFFLENBQUMsQ0FBQztJQUVuRywrQ0FBK0M7SUFDL0MsTUFBTSxxQkFBcUIsR0FBRyxhQUFhLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQWlCLENBQUM7SUFFdkYsK0VBQStFO0lBQy9FLHFCQUFxQixDQUFDLG1CQUFtQixDQUFDLGVBQWUsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO0lBQy9FLHFCQUFxQixDQUFDLFVBQVUsQ0FBQyxRQUFRLEdBQUc7UUFDeEMsT0FBTyxFQUFFO1lBQ0wsaUJBQWlCLEVBQUUsQ0FBQztvQkFDaEIsRUFBRSxFQUFFLEtBQUs7b0JBQ1QsTUFBTSxFQUFFLGlGQUFpRjtpQkFDNUYsRUFBRTtvQkFDQyxFQUFFLEVBQUUsS0FBSztvQkFDVCxNQUFNLEVBQUUsaUZBQWlGO2lCQUM1RixDQUFDO1NBQ0w7S0FDSixDQUFDO0lBRUYsZ0RBQWdEO0lBQ2hELE1BQU0sZUFBZSxHQUFHLElBQUksVUFBVSxDQUFDLGlDQUFpQyxDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRTtRQUM5RyxvQ0FBb0MsRUFBRTtZQUNsQyxPQUFPLEVBQUUsa0RBQWtEO1NBQzlEO0tBQ0osQ0FBQyxDQUFDO0lBRUgsTUFBTSxXQUFXLEdBQUcsVUFBVSxDQUFDLG9CQUFvQixDQUFDLDRCQUE0QixDQUM1RSxLQUFLLEVBQ0wsYUFBYSxFQUNiLGVBQWUsQ0FBQyxHQUFHLENBQ3RCLENBQUM7SUFFRixxQ0FBcUM7SUFDckMsTUFBTSxZQUFZLEdBQUcsNkVBQTBDLENBQUMsWUFBWSxFQUFFLGFBQWEsRUFBRSxXQUFXLENBQUMsQ0FBQztJQUMxRyxJQUFJLE9BQU8sR0FBRyxZQUFZLENBQUM7SUFDM0IsSUFBSSwyQkFBMkIsRUFBRTtRQUM3QixPQUFPLEdBQUcscUJBQWEsQ0FBQyxZQUFZLEVBQUUsMkJBQTJCLENBQUMsQ0FBQztLQUN0RTtJQUNELE1BQU0sY0FBYyxHQUF5QyxJQUFJLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLEVBQUUsd0JBQXdCLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFFaEosd0ZBQXdGO0lBQ3hGLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7UUFDckQsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO1FBQ3pCLFNBQVMsRUFBRSxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDNUMsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsc0JBQXNCLENBQUMsZUFBZSxDQUFDLHFCQUFxQixDQUFDLENBQUM7S0FDdEYsQ0FBQyxDQUFDLENBQUM7SUFFSixvREFBb0Q7SUFDcEQsTUFBTSxZQUFZLEdBQUcsWUFBWSxDQUFDLE1BQXlCLENBQUM7SUFDNUQsTUFBTSxrQkFBa0IsR0FBRyxZQUFZLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQXVCLENBQUM7SUFDekYsa0JBQWtCLENBQUMsVUFBVSxDQUFDLFFBQVEsR0FBRztRQUNyQyxPQUFPLEVBQUU7WUFDTCxpQkFBaUIsRUFBRSxDQUFDO29CQUNwQixFQUFFLEVBQUUsS0FBSztvQkFDVCxNQUFNLEVBQUUsNERBQTREO2lCQUNuRSxDQUFDO1NBQ0w7S0FDSixDQUFDO0lBQ0YsT0FBTyxjQUFjLENBQUM7QUFDMUIsQ0FBQztBQTlERCxrRUE4REMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAxOSBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCAqIGFzIGNsb3VkZnJvbnQgZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGFwaSBmcm9tICdAYXdzLWNkay9hd3MtYXBpZ2F0ZXdheSc7XG5pbXBvcnQgeyBEZWZhdWx0UzNQcm9wcyB9IGZyb20gJy4vczMtYnVja2V0LWRlZmF1bHRzJztcbmltcG9ydCB7IERlZmF1bHRDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uRm9yUzNQcm9wcywgRGVmYXVsdENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Gb3JBcGlHYXRld2F5UHJvcHMgfSBmcm9tICcuL2Nsb3VkZnJvbnQtZGlzdHJpYnV0aW9uLWRlZmF1bHRzJztcbmltcG9ydCB7IG92ZXJyaWRlUHJvcHMgfSBmcm9tICcuL3V0aWxzJztcblxuZXhwb3J0IGZ1bmN0aW9uIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JBcGlHYXRld2F5KHNjb3BlOiBjZGsuQ29uc3RydWN0LCBhcGlFbmRQb2ludDogYXBpLlJlc3RBcGksIGNsb3VkRnJvbnREaXN0cmlidXRpb25Qcm9wcz86XG4gICAgY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHMgfCBhbnkpOiBjbG91ZGZyb250LkNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24ge1xuXG4gICAgbGV0IGRlZmF1bHRwcm9wcztcblxuICAgIGlmIChjbG91ZEZyb250RGlzdHJpYnV0aW9uUHJvcHMgJiYgY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzLmxvZ2dpbmdCdWNrZXQpIHtcbiAgICAgICAgZGVmYXVsdHByb3BzID0gRGVmYXVsdENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Gb3JBcGlHYXRld2F5UHJvcHMoYXBpRW5kUG9pbnQpO1xuICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIENyZWF0ZSB0aGUgTG9nZ2luZyBCdWNrZXRcbiAgICAgICAgY29uc3QgbG9nZ2luZ0J1Y2tldDogczMuQnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzY29wZSwgJ0Nsb3VkZnJvbnRMb2dnaW5nQnVja2V0JywgRGVmYXVsdFMzUHJvcHMoKSk7XG5cbiAgICAgICAgLy8gRXh0cmFjdCB0aGUgQ2ZuQnVja2V0IGZyb20gdGhlIGxvZ2dpbmdCdWNrZXRcbiAgICAgICAgY29uc3QgbG9nZ2luZ0J1Y2tldFJlc291cmNlID0gbG9nZ2luZ0J1Y2tldC5ub2RlLmZpbmRDaGlsZCgnUmVzb3VyY2UnKSBhcyBzMy5DZm5CdWNrZXQ7XG5cbiAgICAgICAgLy8gT3ZlcnJpZGUgYWNjZXNzQ29udHJvbCBjb25maWd1cmF0aW9uIGFuZCBhZGQgbWV0YWRhdGEgZm9yIHRoZSBsb2dnaW5nIGJ1Y2tldFxuICAgICAgICBsb2dnaW5nQnVja2V0UmVzb3VyY2UuYWRkUHJvcGVydHlPdmVycmlkZSgnQWNjZXNzQ29udHJvbCcsICdMb2dEZWxpdmVyeVdyaXRlJyk7XG4gICAgICAgIGxvZ2dpbmdCdWNrZXRSZXNvdXJjZS5jZm5PcHRpb25zLm1ldGFkYXRhID0ge1xuICAgICAgICAgICAgY2ZuX25hZzoge1xuICAgICAgICAgICAgICAgIHJ1bGVzX3RvX3N1cHByZXNzOiBbe1xuICAgICAgICAgICAgICAgICAgICBpZDogJ1czNScsXG4gICAgICAgICAgICAgICAgICAgIHJlYXNvbjogYFRoaXMgUzMgYnVja2V0IGlzIHVzZWQgYXMgdGhlIGFjY2VzcyBsb2dnaW5nIGJ1Y2tldCBmb3IgQ2xvdWRGcm9udCBEaXN0cmlidXRpb25gXG4gICAgICAgICAgICAgICAgfSwge1xuICAgICAgICAgICAgICAgICAgICBpZDogJ1c1MScsXG4gICAgICAgICAgICAgICAgICAgIHJlYXNvbjogYFRoaXMgUzMgYnVja2V0IGlzIHVzZWQgYXMgdGhlIGFjY2VzcyBsb2dnaW5nIGJ1Y2tldCBmb3IgQ2xvdWRGcm9udCBEaXN0cmlidXRpb25gXG4gICAgICAgICAgICAgICAgfV1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfTtcblxuICAgICAgICBkZWZhdWx0cHJvcHMgPSBEZWZhdWx0Q2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbkZvckFwaUdhdGV3YXlQcm9wcyhhcGlFbmRQb2ludCwgbG9nZ2luZ0J1Y2tldCk7XG4gICAgfVxuXG4gICAgLy8gQ3JlYXRlIHRoZSBDbG91ZGZyb250IERpc3RyaWJ1dGlvblxuICAgIGxldCBjZnByb3BzID0gZGVmYXVsdHByb3BzO1xuICAgIGlmIChjbG91ZEZyb250RGlzdHJpYnV0aW9uUHJvcHMpIHtcbiAgICAgICAgY2Zwcm9wcyA9IG92ZXJyaWRlUHJvcHMoZGVmYXVsdHByb3BzLCBjbG91ZEZyb250RGlzdHJpYnV0aW9uUHJvcHMpO1xuICAgIH1cbiAgICBjb25zdCBjZkRpc3RyaWJ1dGlvbjogY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uID0gbmV3IGNsb3VkZnJvbnQuQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbihzY29wZSwgJ0Nsb3VkRnJvbnREaXN0cmlidXRpb24nLCBjZnByb3BzKTtcblxuICAgIHJldHVybiBjZkRpc3RyaWJ1dGlvbjtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JTMyhzY29wZTogY2RrLkNvbnN0cnVjdCwgc291cmNlQnVja2V0OiBzMy5CdWNrZXQsIGNsb3VkRnJvbnREaXN0cmlidXRpb25Qcm9wcz86XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGNsb3VkZnJvbnQuQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvblByb3BzIHwgYW55KTogY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uIHtcbiAgICAvLyBDcmVhdGUgdGhlIExvZ2dpbmcgQnVja2V0XG4gICAgY29uc3QgbG9nZ2luZ0J1Y2tldDogczMuQnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzY29wZSwgJ0Nsb3VkZnJvbnRMb2dnaW5nQnVja2V0JywgRGVmYXVsdFMzUHJvcHMoKSk7XG5cbiAgICAvLyBFeHRyYWN0IHRoZSBDZm5CdWNrZXQgZnJvbSB0aGUgbG9nZ2luZ0J1Y2tldFxuICAgIGNvbnN0IGxvZ2dpbmdCdWNrZXRSZXNvdXJjZSA9IGxvZ2dpbmdCdWNrZXQubm9kZS5maW5kQ2hpbGQoJ1Jlc291cmNlJykgYXMgczMuQ2ZuQnVja2V0O1xuXG4gICAgLy8gT3ZlcnJpZGUgYWNjZXNzQ29udHJvbCBjb25maWd1cmF0aW9uIGFuZCBhZGQgbWV0YWRhdGEgZm9yIHRoZSBsb2dnaW5nIGJ1Y2tldFxuICAgIGxvZ2dpbmdCdWNrZXRSZXNvdXJjZS5hZGRQcm9wZXJ0eU92ZXJyaWRlKCdBY2Nlc3NDb250cm9sJywgJ0xvZ0RlbGl2ZXJ5V3JpdGUnKTtcbiAgICBsb2dnaW5nQnVja2V0UmVzb3VyY2UuY2ZuT3B0aW9ucy5tZXRhZGF0YSA9IHtcbiAgICAgICAgY2ZuX25hZzoge1xuICAgICAgICAgICAgcnVsZXNfdG9fc3VwcHJlc3M6IFt7XG4gICAgICAgICAgICAgICAgaWQ6ICdXMzUnLFxuICAgICAgICAgICAgICAgIHJlYXNvbjogYFRoaXMgUzMgYnVja2V0IGlzIHVzZWQgYXMgdGhlIGFjY2VzcyBsb2dnaW5nIGJ1Y2tldCBmb3IgQ2xvdWRGcm9udCBEaXN0cmlidXRpb25gXG4gICAgICAgICAgICB9LCB7XG4gICAgICAgICAgICAgICAgaWQ6ICdXNTEnLFxuICAgICAgICAgICAgICAgIHJlYXNvbjogYFRoaXMgUzMgYnVja2V0IGlzIHVzZWQgYXMgdGhlIGFjY2VzcyBsb2dnaW5nIGJ1Y2tldCBmb3IgQ2xvdWRGcm9udCBEaXN0cmlidXRpb25gXG4gICAgICAgICAgICB9XVxuICAgICAgICB9XG4gICAgfTtcblxuICAgIC8vIENyZWF0ZSBDbG91ZEZyb250IE9yaWdpbiBBY2Nlc3MgSWRlbnRpdHkgVXNlclxuICAgIGNvbnN0IGNmbk9yaWdBY2Nlc3NJZCA9IG5ldyBjbG91ZGZyb250LkNmbkNsb3VkRnJvbnRPcmlnaW5BY2Nlc3NJZGVudGl0eShzY29wZSwgJ0Nsb3VkRnJvbnRPcmlnaW5BY2Nlc3NJZGVudGl0eScsIHtcbiAgICAgICAgY2xvdWRGcm9udE9yaWdpbkFjY2Vzc0lkZW50aXR5Q29uZmlnOiB7XG4gICAgICAgICAgICBjb21tZW50OiAnQWNjZXNzIFMzIGJ1Y2tldCBjb250ZW50IG9ubHkgdGhyb3VnaCBDbG91ZEZyb250J1xuICAgICAgICB9XG4gICAgfSk7XG5cbiAgICBjb25zdCBvYWlJbXBvcnRlZCA9IGNsb3VkZnJvbnQuT3JpZ2luQWNjZXNzSWRlbnRpdHkuZnJvbU9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZShcbiAgICAgICAgc2NvcGUsXG4gICAgICAgICdPQUlJbXBvcnRlZCcsXG4gICAgICAgIGNmbk9yaWdBY2Nlc3NJZC5yZWZcbiAgICApO1xuXG4gICAgLy8gQ3JlYXRlIHRoZSBDbG91ZGZyb250IERpc3RyaWJ1dGlvblxuICAgIGNvbnN0IGRlZmF1bHRwcm9wcyA9IERlZmF1bHRDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uRm9yUzNQcm9wcyhzb3VyY2VCdWNrZXQsIGxvZ2dpbmdCdWNrZXQsIG9haUltcG9ydGVkKTtcbiAgICBsZXQgY2Zwcm9wcyA9IGRlZmF1bHRwcm9wcztcbiAgICBpZiAoY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzKSB7XG4gICAgICAgIGNmcHJvcHMgPSBvdmVycmlkZVByb3BzKGRlZmF1bHRwcm9wcywgY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzKTtcbiAgICB9XG4gICAgY29uc3QgY2ZEaXN0cmlidXRpb246IGNsb3VkZnJvbnQuQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbiA9IG5ldyBjbG91ZGZyb250LkNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24oc2NvcGUsICdDbG91ZEZyb250RGlzdHJpYnV0aW9uJywgY2Zwcm9wcyk7XG5cbiAgICAvLyBBZGQgUzMgQnVja2V0IFBvbGljeSB0byBhbGxvdyBzMzpHZXRPYmplY3QgZm9yIENsb3VkRnJvbnQgT3JpZ2luIEFjY2VzcyBJZGVudGl0eSBVc2VyXG4gICAgc291cmNlQnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ3MzOkdldE9iamVjdCddLFxuICAgICAgICByZXNvdXJjZXM6IFtzb3VyY2VCdWNrZXQuYXJuRm9yT2JqZWN0cygnKicpXSxcbiAgICAgICAgcHJpbmNpcGFsczogW25ldyBpYW0uQ2Fub25pY2FsVXNlclByaW5jaXBhbChjZm5PcmlnQWNjZXNzSWQuYXR0clMzQ2Fub25pY2FsVXNlcklkKV1cbiAgICB9KSk7XG5cbiAgICAvLyBFeHRyYWN0IHRoZSBDZm5CdWNrZXRQb2xpY3kgZnJvbSB0aGUgc291cmNlQnVja2V0XG4gICAgY29uc3QgYnVja2V0UG9saWN5ID0gc291cmNlQnVja2V0LnBvbGljeSBhcyBzMy5CdWNrZXRQb2xpY3k7XG4gICAgY29uc3Qgc291cmNlQnVja2V0UG9saWN5ID0gYnVja2V0UG9saWN5Lm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIHMzLkNmbkJ1Y2tldFBvbGljeTtcbiAgICBzb3VyY2VCdWNrZXRQb2xpY3kuY2ZuT3B0aW9ucy5tZXRhZGF0YSA9IHtcbiAgICAgICAgY2ZuX25hZzoge1xuICAgICAgICAgICAgcnVsZXNfdG9fc3VwcHJlc3M6IFt7XG4gICAgICAgICAgICBpZDogJ0YxNicsXG4gICAgICAgICAgICByZWFzb246IGBQdWJsaWMgd2Vic2l0ZSBidWNrZXQgcG9saWN5IHJlcXVpcmVzIGEgd2lsZGNhcmQgcHJpbmNpcGFsYFxuICAgICAgICAgICAgfV1cbiAgICAgICAgfVxuICAgIH07XG4gICAgcmV0dXJuIGNmRGlzdHJpYnV0aW9uO1xufVxuIl19