"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const utils_1 = require("./utils");
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.LambdaRestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    configureApiAccessLogging(scope, _api);
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    });
    // Return the API object
    return _api;
}
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureApiAccessLogging(scope, _api) {
    var _a;
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    // Configure the API stage
    const stage = _api.deploymentStage.node.findChild('Resource');
    stage.accessLogSetting = {
        destinationArn: logGroup.logGroupArn,
        format: "$context.identity.sourceIp $context.identity.caller $context.identity.user [$context.requestTime] \"$context.httpMethod $context.resourcePath $context.protocol\" $context.status $context.responseLength $context.requestId"
    };
    // Configure the API deployment
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    deployment.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W45',
                    reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
                }]
        }
    };
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:aws:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })
                ]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    // Define the API
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.RestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    configureApiAccessLogging(scope, _api);
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    });
    // Return the API
    return _api;
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps) {
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj);
    return configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps) {
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj);
    return configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps) {
    const defaultProps = apiDefaults.DefaultGlobalApiProps();
    return configureRestApi(scope, defaultProps, apiGatewayProps);
}
exports.GlobalRestApi = GlobalRestApi;
//# sourceMappingURL=data:application/json;base64,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